﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_IHidServer.sfdl.h>
#include <nn/hid/hid_ResultPalma.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_AppletResourceUserId.h"
#include "hid_HidServer.h"
#include "hid_PalmaImpl.h"
#include "hid_Session.h"

namespace nn { namespace hid { namespace detail {

Result GetPalmaConnectionHandle(PalmaConnectionHandle* pOutHandle, const NpadIdType& id) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto aruid = GetAppletResourceUserId();
    NN_RESULT_DO(pProxy->GetPalmaConnectionHandle(pOutHandle, id, aruid));
    NN_RESULT_SUCCESS;
}

Result InitializePalma(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto nativeHandle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->InitializePalma(handle));
    NN_RESULT_SUCCESS;
}

Result BindPalmaOperationCompleteEvent(const PalmaConnectionHandle& handle,
                                      nn::os::SystemEventType* pEvent,
                                      ::nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto nativeHandle = ::nn::sf::NativeHandle();
    NN_RESULT_DO(pProxy->AcquirePalmaOperationCompleteEvent(&nativeHandle, handle));

    ::nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                                nativeHandle.GetOsHandle(),
                                                nativeHandle.IsManaged(),
                                                clearMode);

    // ハンドルの管理権を放棄
    nativeHandle.Detach();

    NN_RESULT_SUCCESS;
}

Result GetPalmaOperationInfo(PalmaOperationInfo* pOutValue,
                            const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    uint64_t typeRaw;
    auto outBuffer = nn::sf::OutBuffer(reinterpret_cast<char*>(pOutValue->individual.raw), sizeof(pOutValue->individual.raw));

    // OperationInfo を取得する
    NN_RESULT_DO(pProxy->GetPalmaOperationInfo(&typeRaw, outBuffer, handle));

    // OperationInfo の Result を取得する
    auto result = pProxy->GetPalmaOperationResult(handle);

    if (result.IsSuccess())
    {
        pOutValue->result = ::nn::ResultSuccess();
    }
    else
    {
        NN_RESULT_TRY(result)
        // ResultPalma のうち ResultPalmaInvalidHandle, ResultPalmaNoOperationInfo は結果が含まれないのですぐにエラーを返す
        NN_RESULT_CATCH(ResultPalmaInvalidHandle)
        {
            NN_RESULT_THROW(result);
        }
        NN_RESULT_CATCH(ResultPalmaNoOperationInfo)
        {
            NN_RESULT_THROW(result);
        }
        NN_RESULT_CATCH(ResultPalma)
        {
            pOutValue->result = *reinterpret_cast<Result*>(&result);
        }
        // ResultPalma 以外のエラー
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(result);
        }
        NN_RESULT_END_TRY;
    }

    switch (typeRaw)
    {
    case PalmaOperationType_PlayActivity:
    case PalmaOperationType_SetFrModeType:
    case PalmaOperationType_ReadStep:
    case PalmaOperationType_EnableStep:
    case PalmaOperationType_ResetStep:
    case PalmaOperationType_ReadApplicationSection:
    case PalmaOperationType_WriteApplicationSection:
    case PalmaOperationType_ReadUniqueCode:
    case PalmaOperationType_SetUniqueCodeInvalid:
    case PalmaOperationType_WriteActivityEntry:
    case PalmaOperationType_WriteRgbLedPatternEntry:
    case PalmaOperationType_WriteWaveEntry:
    case PalmaOperationType_ReadDataBaseIdentificationVersion:
    case PalmaOperationType_WriteDataBaseIdentificationVersion:
    case PalmaOperationType_SuspendFeature:
    case PalmaOperationType_ReadPlayLog:
    case PalmaOperationType_ResetPlayLog:
        pOutValue->type = static_cast<PalmaOperationType>(typeRaw);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    NN_RESULT_SUCCESS;
}

Result PlayPalmaActivity(const PalmaConnectionHandle& handle, const PalmaActivityIndex& index) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->PlayPalmaActivity(handle, static_cast<uint64_t>(index)));
    NN_RESULT_SUCCESS;
}

Result SetPalmaFrModeType(const PalmaConnectionHandle& handle, const PalmaFrModeType& type) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetPalmaFrModeType(handle, static_cast<uint64_t>(type)));
    NN_RESULT_SUCCESS;
}

Result ReadPalmaStep(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ReadPalmaStep(handle));
    NN_RESULT_SUCCESS;
}

Result EnablePalmaStep(const PalmaConnectionHandle& handle, bool isEnabled) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->EnablePalmaStep(handle, isEnabled));
    NN_RESULT_SUCCESS;
}

Result ResetPalmaStep(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ResetPalmaStep(handle));
    NN_RESULT_SUCCESS;
}

Result ReadPalmaApplicationSection(const PalmaConnectionHandle& handle, int address, size_t size) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ReadPalmaApplicationSection(handle, static_cast<uint64_t>(address), static_cast<uint64_t>(size)));
    NN_RESULT_SUCCESS;
}

Result WritePalmaApplicationSection(const PalmaConnectionHandle& handle, int address, size_t size, const PalmaApplicationSectionAccessBuffer& buffer) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->WritePalmaApplicationSection(handle, static_cast<uint64_t>(address), static_cast<uint64_t>(size), buffer));
    NN_RESULT_SUCCESS;
}

Result ReadPalmaUniqueCode(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ReadPalmaUniqueCode(handle));
    NN_RESULT_SUCCESS;
}

Result SetPalmaUniqueCodeInvalid(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetPalmaUniqueCodeInvalid(handle));
    NN_RESULT_SUCCESS;
}

Result WritePalmaActivityEntry(const PalmaConnectionHandle& handle, const PalmaActivityIndex& index, const PalmaActivityEntry* pEntry) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->WritePalmaActivityEntry(handle,
                                                 static_cast<uint64_t>(index),
                                                 static_cast<uint64_t>(pEntry->rgbLedPatternIndex),
                                                 static_cast<uint64_t>(pEntry->waveSet),
                                                 static_cast<uint64_t>(pEntry->waveIndex)));
    NN_RESULT_SUCCESS;
}

Result WritePalmaRgbLedPatternEntry(const PalmaConnectionHandle& handle, const PalmaRgbLedPatternIndex& index, const void* pRgbLedPatternBuffer, size_t size) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    nn::sf::InBuffer pInBuffer(reinterpret_cast<const char*>(pRgbLedPatternBuffer), size);

    NN_RESULT_DO(pProxy->WritePalmaRgbLedPatternEntry(handle, static_cast<uint64_t>(index), pInBuffer));
    NN_RESULT_SUCCESS;
}

Result WritePalmaWaveEntry(const PalmaConnectionHandle& handle, const PalmaWaveSet& waveSet, const PalmaWaveIndex& index, const void* pWaveBuffer, size_t bufferSize, size_t waveDataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_ALIGNED(pWaveBuffer, ::nn::os::MemoryPageSize);
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    ::nn::os::TransferMemory waveBuffer(const_cast<void*>(pWaveBuffer), bufferSize, ::nn::os::MemoryPermission_ReadOnly);
    auto nativeHandle = nn::sf::NativeHandle(waveBuffer.Detach(), true);

    NN_RESULT_DO(pProxy->WritePalmaWaveEntry(handle, static_cast<uint64_t>(waveSet), static_cast<uint64_t>(index), std::move(nativeHandle), static_cast<uint64_t>(bufferSize), static_cast<uint64_t>(waveDataSize)));
    NN_RESULT_SUCCESS;
}

Result SetPalmaDataBaseIdentificationVersion(const PalmaConnectionHandle& handle, int version) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetPalmaDataBaseIdentificationVersion(handle, static_cast<int32_t>(version)));
    NN_RESULT_SUCCESS;
}

Result GetPalmaDataBaseIdentificationVersion(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->GetPalmaDataBaseIdentificationVersion(handle));
    NN_RESULT_SUCCESS;
}

Result SuspendPalmaFeature(const PalmaConnectionHandle& handle, const PalmaFeatureSet& suspendFeatureSet) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SuspendPalmaFeature(handle, suspendFeatureSet));
    NN_RESULT_SUCCESS;
}

Result ReadPalmaPlayLog(const PalmaConnectionHandle& handle, const PalmaPlayLogFieldIndex& index) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ReadPalmaPlayLog(handle, static_cast<uint16_t>(index)));
    NN_RESULT_SUCCESS;
}

Result ResetPalmaPlayLog(const PalmaConnectionHandle& handle, const PalmaPlayLogFieldIndex& index) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->ResetPalmaPlayLog(handle, static_cast<uint16_t>(index)));
    NN_RESULT_SUCCESS;
}

Result EnablePairedPalmaConnection() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto aruid = GetAppletResourceUserId();
    NN_RESULT_DO(pProxy->SetIsPalmaPairedConnectable(aruid, true));
    NN_RESULT_SUCCESS;
}

Result DisablePairedPalmaConnection() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto aruid = GetAppletResourceUserId();
    NN_RESULT_DO(pProxy->SetIsPalmaPairedConnectable(aruid, false));
    NN_RESULT_SUCCESS;
}

Result EnableAnyPalmaConnection() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto aruid = GetAppletResourceUserId();
    NN_RESULT_DO(pProxy->SetIsPalmaAllConnectable(aruid, true));
    NN_RESULT_SUCCESS;
}

Result DisableAnyPalmaConnection() NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    auto aruid = GetAppletResourceUserId();
    NN_RESULT_DO(pProxy->SetIsPalmaAllConnectable(aruid, false));
    NN_RESULT_SUCCESS;
}

Result PairPalma(const PalmaConnectionHandle& handle) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->PairPalma(handle));
    NN_RESULT_SUCCESS;
}

Result SetPalmaBoostMode(bool enabled) NN_NOEXCEPT
{
    auto pProxy = ::nn::sf::SharedPointer<IHidServer>();
    CreateHidServerProxy(&pProxy);

    NN_RESULT_DO(pProxy->SetPalmaBoostMode(enabled));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
