﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedPadXcd.h"
#include "hid_AbstractedGcAdapterPad.h"
#include "hid_CommonStateUtility.h"
#include "hid_NpadInternalState.h"
#include "hid_NpadVibrationController.h"
#include "hid_NpadStateUtility.h"
#include "hid_Settings.h"
#include "hid_VibrationDeviceHandle.h"

namespace nn { namespace hid { namespace detail {

NpadVibrationController::NpadVibrationController() NN_NOEXCEPT
    : m_pAbstractedPadHolder(nullptr)
    , m_pDeviceManager(nullptr)
    , m_NpadActivationCount()
{
    // 何もしない
}

NpadVibrationController::~NpadVibrationController() NN_NOEXCEPT { /* 何もしない */ }

void NpadVibrationController::SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());
    m_pAbstractedPadHolder = pHolder;
}

void NpadVibrationController::SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);

    m_pDeviceManager = pHolder;
}

void NpadVibrationController::SetVibratorXcdDriver(VibratorXcdDriver vibratorArray[], int vibratorCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(vibratorArray);
    NN_SDK_REQUIRES_MINMAX(vibratorCount, 0, NpadVibratorCountPerPlayerMax);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());

    for(int i = 0; i < vibratorCount; i++)
    {
        m_pVibratorXcdDrivers[i] = &vibratorArray[i];
    }
}

void NpadVibrationController::SetVibratorGcDriver(VibratorGcDriver* pVibrator) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVibrator);
    NN_SDK_REQUIRES(m_NpadActivationCount.IsZero());

    m_pVibratorGcDriver = pVibrator;
}
::nn::Result NpadVibrationController::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadHolder);
    NN_SDK_REQUIRES_NOT_NULL(m_pDeviceManager);
    NN_SDK_REQUIRES_NOT_NULL(m_pVibratorXcdDrivers[0]);

    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsMax(),
                           ResultNpadActivationUpperLimitOver());

    if (m_NpadActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施
    }

    ++m_NpadActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result NpadVibrationController::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_NpadActivationCount.IsZero(),
                           ResultNpadDeactivationLowerLimitOver());

    --m_NpadActivationCount;

    if (m_NpadActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
    }

    NN_RESULT_SUCCESS;
}

void NpadVibrationController::UpdateDeviceStates() NN_NOEXCEPT
{
    struct XcdDeviceHandleUnit
    {
        bool isAvailable;
        nn::xcd::DeviceHandle xcdHandle;
    };
    XcdDeviceHandleUnit deviceHandles[NN_ARRAY_SIZE(m_pVibratorXcdDrivers)];
    bool isGcErmEnabled = false;
    auto gcIndex = GcControllerIndex();

    for (int i = 0; i < NN_ARRAY_SIZE(m_pVibratorXcdDrivers); i++)
    {
        deviceHandles[i].isAvailable = false;
        deviceHandles[i].xcdHandle = nn::xcd::DeviceHandle::GetInvalidHandle();
    }

    IAbstractedPad* pPads[AbstractedPadCountMaxPerNpad];
    auto count = m_pAbstractedPadHolder->GetAbstractedPads(pPads, NN_ARRAY_SIZE(pPads));

    for (int i = 0; i < count; i++)
    {
        if (pPads[i]->IsConnected())
        {
            if (pPads[i]->GetType() == AbstractedPadType_Xcd)
            {
                if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::LraLeft>())
                {
                    if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                    {
                        deviceHandles[NpadVibrationDeviceIdx_Left].isAvailable = true;
                        deviceHandles[NpadVibrationDeviceIdx_Left].xcdHandle = reinterpret_cast<AbstractedPadXcd*>(pPads[i])->GetXcdDeviceHandle();
                    }
                }
                if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::LraRight>())
                {
                    if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                    {
                        deviceHandles[NpadVibrationDeviceIdx_Right].isAvailable = true;
                        deviceHandles[NpadVibrationDeviceIdx_Right].xcdHandle = reinterpret_cast<AbstractedPadXcd*>(pPads[i])->GetXcdDeviceHandle();
                    }
                }
            }
            if (pPads[i]->GetType() == AbstractedPadType_GcAdapter)
            {
                if (pPads[i]->GetFeatureSet().Test<AbstractedPadFeature::GcErm>())
                {
                    if (m_pDeviceManager->IsStyleActiveForDevice(pPads[i]->GetDeviceType()))
                    {
                        isGcErmEnabled = true;
                        gcIndex = reinterpret_cast<AbstractedGcAdapterPad*>(pPads[i])->GetIndex();
                    }
                }
            }
        }
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_pVibratorXcdDrivers); i++)
    {
        if (deviceHandles[i].isAvailable == true)
        {
            VibratorXcdConfig vibratorConfig;
            // VibratorXcdDriver を有効にする
            vibratorConfig.position = (i == NpadVibrationDeviceIdx_Left) ? ::nn::xcd::VibratorPosition_Left : ::nn::xcd::VibratorPosition_Right;
            vibratorConfig.handle = deviceHandles[i].xcdHandle;
            m_pVibratorXcdDrivers[i]->ActivateVibrator(vibratorConfig);
        }
        else
        {
            m_pVibratorXcdDrivers[i]->DeactivateVibrator();
        }
    }

    if (isGcErmEnabled == true)
    {
        m_pVibratorGcDriver->ActivateVibrator(gcIndex);
    }
    else
    {
        m_pVibratorGcDriver->DeactivateVibrator();
    }
}


}}} // namespace nn::hid::detail
