﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <mutex>
#include <new>
#include <nn/nn_Macro.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_LockableMutexType.h"

namespace nn { namespace hid { namespace detail {

//!< Npad のセッションを管理するためのクラスです。
template<typename T>
class NpadSessionManager final
{
    NN_DISALLOW_COPY(NpadSessionManager);
    NN_DISALLOW_MOVE(NpadSessionManager);

private:
    //!< Npad のセッションのエントリを表す構造体です。
    struct Entry final
    {
        // セッションが初期化済みか否かを表す値
        ::std::atomic_bool isInitialized;

        // セッションの初期化排他用ミューテックス
        LockableMutexType mutex;

        // セッションの格納先となるメモリ領域
        ::nn::util::TypedStorage<T, sizeof(T), NN_ALIGNOF(T)> storage;
    };

    //!< Npad のセッションのエントリ
    Entry m_Entry;

public:
    NpadSessionManager() NN_NOEXCEPT
    {
        m_Entry.isInitialized = false;
        ::nn::os::InitializeMutex(&m_Entry.mutex._mutex, false, 0);
    }

    //!< Npad のセッションを返します。
    T* GetSession() NN_NOEXCEPT
    {
        T* pSession = &::nn::util::Get(m_Entry.storage);

        if (!m_Entry.isInitialized)
        {
            ::std::lock_guard<decltype(m_Entry.mutex)> locker(m_Entry.mutex);

            if (!m_Entry.isInitialized)
            {
                new(pSession) T();

                m_Entry.isInitialized = true;
            }
        }

        return pSession;
    }
};

}}} // namespace nn::hid::detail
