﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SystemEventTypes.h>

#include "hid_ActivationCount.h"
#include "hid_NpadAbstractedPadHolder.h"
#include "hid_NpadDeviceManager.h"

namespace nn { namespace hid { namespace detail {

//!< Npad インターフェイスの解決を担うマネージャを扱うクラスです。
class NpadNfcController final
{
    NN_DISALLOW_COPY(NpadNfcController);
    NN_DISALLOW_MOVE(NpadNfcController);

private:
    //!< Npad 毎に割り当てられる AbstractedPad を保持するクラス
    NpadAbstractedPadHolder* m_pAbstractedPadHolder;

    //!< Npad の状態管理を担うマネージャー
    NpadDeviceManager* m_pDeviceManager;

    //!< アクティブ化された回数
    ActivationCount m_NpadActivationCount;

    //!< Nfc デバイスのアクティブ状態に変化があった際に通知する システムイベント
    ::nn::os::SystemEvent m_NfcActivateEvent;

    //!< 使用できる Nfc デバイスに変化があった場合に通知する システムイベント
    ::nn::os::SystemEvent* m_pNfcUpdateEvent;

    //!< Nfc の内部状態の定義
    enum NfcState
    {
        NfcState_NoDevice,
        NfcState_Deactive,
        NfcState_Active,
    };

    //!< Nfc の状態
    NfcState m_NfcState;

    //!< Nfc が割り当てられた Xcd デバイスの亜ｈンドル
    ::nn::xcd::DeviceHandle m_NfcXcdDeviceHandle;

public:
    NpadNfcController() NN_NOEXCEPT;

    ~NpadNfcController() NN_NOEXCEPT;

    //!< NpadAbstractedPadHolder を設定します。
    void SetNpadAbstractedPadHolder(NpadAbstractedPadHolder* pHolder) NN_NOEXCEPT;

    //!< NpadDeviceManager を設定します。
    void SetNpadDeviceManager(NpadDeviceManager* pHolder) NN_NOEXCEPT;

    //!< アクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< 非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Nfc デバイスの状態を更新します
    void UpdateDeviceStates() NN_NOEXCEPT;

    //!< Nfc のアクティブ状態が変化したときに受ける通知をイベントオブジェクトにバインドします。
    Result AcquireNfcActivateEventHandle(::nn::os::NativeHandle* pOutValue) NN_NOEXCEPT;

    //!< 使用できる Nfc デバイスに変化があった場合に通知するイベントオブジェクトを登録します。
    void SetNfcUpdateEvent(::nn::os::SystemEvent* pEvent) NN_NOEXCEPT;

    //!< Nfc のアクティブ状態を制御します。
    Result ActivateNfc(bool activate, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< NpadId に対して割り当てられている Nfc のデバイスハンドルを取得します
    ::nn::Result GetXcdHandleForNpadWithNfc(::nn::xcd::DeviceHandle* pOutHandle, ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< Nfc を搭載しているかどうかを返します。
    bool IsNfcAvailable() NN_NOEXCEPT;

    //!< Nfc がアクティブかどうかを返します。
    bool IsNfcActivated() NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
