﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <limits> // for numeric_limits
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_PowerState.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_NpadId.h"
#include "hid_NpadManager.h"

namespace nn { namespace hid { namespace detail {

NpadManager::NpadManager() NN_NOEXCEPT
    : m_pCommonResourceHolder(nullptr)
    , m_ActivationCount()
    , m_InterfaceTypeOfFullKey(system::InterfaceType_Unknown)
{
    // 何もしない
}

NpadManager::~NpadManager() NN_NOEXCEPT
{
    // 何もしない
}

void NpadManager::Initialize(NpadCommonResourceHolder* pHolder,
                             CaptureButtonManager* pCaptureButtonManager,
                             HomeButtonManager* pHomeButtonManager,
                             SixAxisSensorAppletSettingManager* pSixAxisSensorSettingsManager,
                             ConsoleSixAxisSensorManager* pConsoleSixAxisSensor,
                             GcAdapterDriver* pGcAdapterDriver,
                             PalmaResourceManager* pPalma
                             ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHolder);
    NN_SDK_REQUIRES_NOT_NULL(pCaptureButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(pHomeButtonManager);
    NN_SDK_REQUIRES_NOT_NULL(pSixAxisSensorSettingsManager);
    NN_SDK_REQUIRES_NOT_NULL(pConsoleSixAxisSensor);
    NN_SDK_REQUIRES_NOT_NULL(pGcAdapterDriver);
    NN_SDK_REQUIRES_NOT_NULL(pPalma);
    NN_SDK_REQUIRES(pHolder->IsInitialized());
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());

    m_pCommonResourceHolder = pHolder;

    m_DeviceManager.SetNpadCommonResourceHolder(pHolder);
    m_DeviceManager.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);

    m_Indicator.SetNpadCommonResourceHolder(pHolder);
    m_Indicator.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_Indicator.SetNpadDeviceManager(&m_DeviceManager);

    m_IrSensor.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_IrSensor.SetNpadDeviceManager(&m_DeviceManager);

    m_Nfc.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_Nfc.SetNpadDeviceManager(&m_DeviceManager);

    m_VibratorGcDriver.SetGcAdapterDriver(pGcAdapterDriver);

    m_RailAttachment.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_RailAttachment.SetNpadDeviceManager(&m_DeviceManager);

    m_Vibration.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_Vibration.SetNpadDeviceManager(&m_DeviceManager);
    m_Vibration.SetVibratorXcdDriver(m_VibratorXcdDrivers, NN_ARRAY_SIZE(m_VibratorXcdDrivers));
    m_Vibration.SetVibratorGcDriver(&m_VibratorGcDriver);

    m_SixAxisSensor.SetNpadCommonResourceHolder(pHolder);
    m_SixAxisSensor.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_SixAxisSensor.SetNpadDeviceManager(&m_DeviceManager);
    for (int i = 0; i < SixAxisSensorIdxCountMax; ++i)
    {
        m_SixAxisSensor.SetSixAxisSensorXcdDriver(&m_SixAxisSensorXcdDrivers[i], static_cast<SixAxisSensorIdx>(i));
    }
    m_SixAxisSensor.SetSixAxisSensorAppletSettingManager(pSixAxisSensorSettingsManager);
    m_SixAxisSensor.SetConsoleSixAxisSensorManager(pConsoleSixAxisSensor);

    m_InputStateUpdater.SetNpadCommonResourceHolder(pHolder);
    m_InputStateUpdater.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_InputStateUpdater.SetNpadDeviceManager(&m_DeviceManager);
    m_InputStateUpdater.SetCaptureButtonNotifier(pCaptureButtonManager->MakeSystemButtonNotifier());
    m_InputStateUpdater.SetHomeButtonNotifier(pHomeButtonManager->MakeSystemButtonNotifier());

    m_PowerInfoUpdater.SetNpadCommonResourceHolder(pHolder);
    m_PowerInfoUpdater.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_PowerInfoUpdater.SetNpadDeviceManager(&m_DeviceManager);

    m_Palma.SetNpadAbstractedPadHolder(&m_AbstractedPadHolder);
    m_Palma.SetNpadDeviceManager(&m_DeviceManager);
    m_Palma.SetPalmaResourceManager(pPalma);
}

void NpadManager::SetNpadIdType(NpadIdType id) NN_NOEXCEPT
{
    m_DeviceManager.SetNpadIdType(id);
}

::nn::Result NpadManager::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        auto needsRollback = true;

        NN_RESULT_DO(m_DeviceManager.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_DeviceManager.Deactivate();
            }
        };

        NN_RESULT_DO(m_Indicator.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_Indicator.Deactivate();
            }
        };

        NN_RESULT_DO(m_IrSensor.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_IrSensor.Deactivate();
            }
        };

        NN_RESULT_DO(m_RailAttachment.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_RailAttachment.Deactivate();
            }
        };

        NN_RESULT_DO(m_Nfc.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_Nfc.Deactivate();
            }
        };

        NN_RESULT_DO(m_Vibration.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_Vibration.Deactivate();
            }
        };

        NN_RESULT_DO(m_SixAxisSensor.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_SixAxisSensor.Deactivate();
            }
        };

        NN_RESULT_DO(m_InputStateUpdater.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_InputStateUpdater.Deactivate();
            }
        };

        NN_RESULT_DO(m_PowerInfoUpdater.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_PowerInfoUpdater.Deactivate();
            }
        };

        NN_RESULT_DO(m_Palma.Activate());
        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                m_Palma.Deactivate();
            }
        };

        needsRollback = false;
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;

}

::nn::Result NpadManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        m_PowerInfoUpdater.Deactivate();
        m_InputStateUpdater.Deactivate();
        m_SixAxisSensor.Deactivate();
        m_Vibration.Deactivate();
        m_Nfc.Deactivate();
        m_IrSensor.Deactivate();
        m_RailAttachment.Deactivate();
        m_Indicator.Deactivate();
        m_DeviceManager.Deactivate();
        m_Palma.Deactivate();
    }

    NN_RESULT_SUCCESS;
}

::nn::Result NpadManager::EnsureAppletResource(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    NN_RESULT_DO(m_DeviceManager.EnsureNpadAppletResource(aruid));
    NN_RESULT_DO(m_InputStateUpdater.EnsureNpadAppletResource(aruid));
    NN_RESULT_DO(m_SixAxisSensor.EnsureNpadAppletResource(aruid));
    NN_RESULT_DO(m_PowerInfoUpdater.EnsureNpadAppletResource(aruid));
    NN_RESULT_SUCCESS;
}

NpadAbstractedPadHolder& NpadManager::GetAbstractedPadHolder() NN_NOEXCEPT
{
    return m_AbstractedPadHolder;
}

NpadDeviceManager& NpadManager::GetDeviceManager() NN_NOEXCEPT
{
    return m_DeviceManager;
}

NpadIndicatorController& NpadManager::GetIndicator() NN_NOEXCEPT
{
    return m_Indicator;
}

NpadIrSensorController& NpadManager::GetIrSensor() NN_NOEXCEPT
{
    return m_IrSensor;
}

NpadRailAttachmentController& NpadManager::GetRailAttachment() NN_NOEXCEPT
{
    return m_RailAttachment;
}

NpadNfcController& NpadManager::GetNfc() NN_NOEXCEPT
{
    return m_Nfc;
}

NpadVibrationController& NpadManager::GetVibration() NN_NOEXCEPT
{
    return m_Vibration;
}

NpadSixAxisSensorController& NpadManager::GetSixAxisSensor() NN_NOEXCEPT
{
    return m_SixAxisSensor;
}

NpadInputStateUpdater& NpadManager::GetInputStateUpdater() NN_NOEXCEPT
{
    return m_InputStateUpdater;
}

NpadPowerInfoUpdater& NpadManager::GetPowerInfoUpdater() NN_NOEXCEPT
{
    return m_PowerInfoUpdater;
}

VibratorXcdDriver* NpadManager::GetVibratorXcdDriver(const int index) NN_NOEXCEPT
{
    return &m_VibratorXcdDrivers[index];
}

void NpadManager::GetVibratorXcdDrivers(int* pOutCount, VibratorXcdDriver** outDrivers) NN_NOEXCEPT
{
    *pOutCount = NN_ARRAY_SIZE(m_VibratorXcdDrivers);
    for (int i = 0; i < *pOutCount; i++)
    {
        outDrivers[i] = &m_VibratorXcdDrivers[i];
    }
}

VibratorGcDriver* NpadManager::GetVibratorGcDriver() NN_NOEXCEPT
{
    return &m_VibratorGcDriver;
}

const SixAxisSensorProcessor* NpadManager::GetSixAxisSensorProcessor(const SixAxisSensorHandle& handle) const NN_NOEXCEPT
{
    return m_SixAxisSensor.GetSixAxisSensorProcessor(handle);
}

IrSensorXcdDriver* NpadManager::GetIrSensorXcdDriver() NN_NOEXCEPT
{
    return &m_IrSensorXcdDriver;
}

ExternalBusXcdDriver* NpadManager::GetExternalBusXcdDriver(const int index) NN_NOEXCEPT
{
    return &m_ExternalBusXcdDriver[index];
}

NpadIdType NpadManager::GetNpadIdType() NN_NOEXCEPT
{
    return m_DeviceManager.GetNpadIdType();
}

void NpadManager::ProceedPeriodicEvent() NN_NOEXCEPT
{
    // フルコンの接続インタフェースが変化した場合は、Npad の内部ステートをまず更新する
    if (m_InterfaceTypeOfFullKey != m_DeviceManager.GetInterfaceTypeOfFullKey())
    {
        UpdateNpad();
    }

    m_DeviceManager.CheckForPlayReport();
    m_InputStateUpdater.UpdateInputStates(false);
    m_SixAxisSensor.UpdateInputStates();
    m_PowerInfoUpdater.UpdatePowerInfo();
    m_Indicator.UpdateState();
}

void NpadManager::UpdateSharedMemoryForPeriodicalStates() NN_NOEXCEPT
{
    m_InputStateUpdater.UpdateSharedMemory();
    m_SixAxisSensor.UpdateSharedMemory();
    m_PowerInfoUpdater.UpdateSharedMemory();
}

void NpadManager::UpdateSharedMemoryForTarget(::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_InputStateUpdater.UpdateSharedMemoryForTarget(aruid);
    m_SixAxisSensor.UpdateSharedMemoryForTarget(aruid);
    m_PowerInfoUpdater.UpdateSharedMemoryForTarget(aruid);
}

void NpadManager::UpdateNpad() NN_NOEXCEPT
{
    m_DeviceManager.UpdateControlStates();
    m_Indicator.UpdatePattern();
    m_Vibration.UpdateDeviceStates();
    m_SixAxisSensor.UpdateDeviceStates();
    m_Nfc.UpdateDeviceStates();
    m_IrSensor.UpdateDeviceStates();
    m_RailAttachment.UpdateDeviceStates();
    m_Palma.UpdateDeviceStates();
    m_PowerInfoUpdater.UpdatePowerInfo();
    m_InputStateUpdater.UpdateInputStates(true);

    m_InterfaceTypeOfFullKey = m_DeviceManager.GetInterfaceTypeOfFullKey();

    {
        ::std::lock_guard<decltype(m_pCommonResourceHolder->GetAppletResourceManagerMutex())
                          > locker(m_pCommonResourceHolder->GetAppletResourceManagerMutex());
        m_DeviceManager.UpdateSharedMemory();
        m_PowerInfoUpdater.UpdateSharedMemory();
        m_InputStateUpdater.UpdateSharedMemoryForced();
    }
}

}}} // namespace nn::hid::detail
