﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_Keyboard.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitPack.h>

#include "hid_KeyboardDriver-os.horizon.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< LED の点灯パターンのフィールド定義です。
struct LedField final
{
    typedef ::nn::util::BitPack<uint8_t, LedField>::Field<0, 1, bool>
            NumLock;
    typedef ::nn::util::BitPack<uint8_t, LedField>::Field<1, 1, bool>
            CapsLock;
    typedef ::nn::util::BitPack<uint8_t, LedField>::Field<2, 1, bool>
            ScrollLock;
    typedef ::nn::util::BitPack<uint8_t, LedField>::Field<3, 1, bool>
            Compose;
    typedef ::nn::util::BitPack<uint8_t, LedField>::Field<4, 1, bool>
            Kana;
};

//!< LED の点灯パターンの型です。
typedef ::nn::util::BitPack<uint8_t, LedField> LedPack;

//!< 修飾キーのフィールド定義です。
struct ModifierField final
{
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<0, 1, bool>
            LeftControl;
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<1, 1, bool>
            LeftShift;
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<2, 1, bool>
            LeftAlt;
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<3, 1, bool>
            LeftGui;
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<4, 1, bool>
            RightControl;
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<5, 1, bool>
            RightShift;
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<6, 1, bool>
            RightAlt;
    typedef ::nn::util::BitPack<uint8_t, ModifierField>::Field<7, 1, bool>
            RightGui;
};

//!< 修飾キーの集合を扱う型です。
typedef ::nn::util::BitPack<uint8_t, ModifierField> ModifierPack;

//!< レポートを表す構造体です。
struct NN_ALIGNAS(8) Report final
{
    ModifierPack modifiers;
    NN_PADDING1;
    uint8_t keycodes[6];
};

//!< 修飾キーの状態を処理します。
void ProcessModifierKeys(KeyboardKeySet* pKeys,
                         const ModifierPack& modifiers) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKeys);

    if (modifiers.Get<ModifierField::LeftControl>())
    {
        pKeys->Set<KeyboardKey::LeftControl>();
    }

    if (modifiers.Get<ModifierField::LeftShift>())
    {
        pKeys->Set<KeyboardKey::LeftShift>();
    }

    if (modifiers.Get<ModifierField::LeftAlt>())
    {
        pKeys->Set<KeyboardKey::LeftAlt>();
    }

    if (modifiers.Get<ModifierField::LeftGui>())
    {
        pKeys->Set<KeyboardKey::LeftGui>();
    }

    if (modifiers.Get<ModifierField::RightControl>())
    {
        pKeys->Set<KeyboardKey::RightControl>();
    }

    if (modifiers.Get<ModifierField::RightShift>())
    {
        pKeys->Set<KeyboardKey::RightShift>();
    }

    if (modifiers.Get<ModifierField::RightAlt>())
    {
        pKeys->Set<KeyboardKey::RightAlt>();
    }

    if (modifiers.Get<ModifierField::RightGui>())
    {
        pKeys->Set<KeyboardKey::RightGui>();
    }
}

//!< ロックキーの状態を処理します。
template<typename T, typename U, typename V>
void ProcessLockKey(LedPack* pLeds,
                    KeyboardModifierSet* pModifiers,
                    const KeyboardKeySet keys) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pLeds);
    NN_SDK_REQUIRES_NOT_NULL(pModifiers);

    if (!keys.Test<V>())
    {
        pModifiers->Reset<U>();
    }
    else
    {
        if (!pModifiers->Test<U>())
        {
            if (pLeds->Get<T>())
            {
                pLeds->Set<T>(false);
            }
            else
            {
                pLeds->Set<T>(true);
            }

            pModifiers->Set<U>();
        }
    }
}

} // namespace

KeyboardDriver::KeyboardDriver() NN_NOEXCEPT
    : m_ActivationCount()
    , m_SamplingNumber(0)
    , m_Modifiers()
    , m_Count()
    , m_Leds()
    , m_Reports()
{
    // 何もしない
}

KeyboardDriver::~KeyboardDriver() NN_NOEXCEPT
{
    // 何もしない
}

::nn::Result KeyboardDriver::Activate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultKeyboardDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        m_Modifiers.Reset();

        m_Count = 0;

        LedPack leds = {};
        leds.Set<LedField::NumLock>(true);
        m_Leds = leds.storage;

        for (size_t i = 0; i < PortCount; ++i)
        {
            m_Reports[i] = 0;
        }
    }

    // このインスタンスからアクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result KeyboardDriver::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultKeyboardDriverDeactivationLowerLimitOver());

    // このインスタンスからアクティブ化した回数をデクリメント
    --m_ActivationCount;

    NN_RESULT_SUCCESS;
}

void KeyboardDriver::GetState(KeyboardState* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    // キーボードの入力状態を初期化
    *pOutValue = KeyboardState();

    pOutValue->samplingNumber = m_SamplingNumber++;

    // キーボードの接続状態を設定
    if (0 < m_Count)
    {
        pOutValue->attributes.Set<KeyboardAttribute::IsConnected>();
    }

    // キーの状態を設定
    for (size_t i = 0; i < PortCount; ++i)
    {
        uint64_t bytes = m_Reports[i];

        auto pReport = reinterpret_cast<Report*>(&bytes);

        ProcessModifierKeys(&pOutValue->keys, pReport->modifiers);

        for (uint8_t& keycode : pReport->keycodes)
        {
            if (keycode != 0)
            {
                pOutValue->keys.Set(static_cast<int>(keycode));
            }
        }
    }

    // LED の点灯状態を更新
    LedPack leds = { static_cast<uint8_t>(m_Leds) };
    ProcessLockKey<LedField::NumLock,
                   KeyboardModifier::NumLock,
                   KeyboardKey::NumLock
                   >(&leds, &m_Modifiers, pOutValue->keys);
    ProcessLockKey<LedField::CapsLock,
                   KeyboardModifier::CapsLock,
                   KeyboardKey::CapsLock
                   >(&leds, &m_Modifiers, pOutValue->keys);
    ProcessLockKey<LedField::ScrollLock,
                   KeyboardModifier::ScrollLock,
                   KeyboardKey::ScrollLock
                   >(&leds, &m_Modifiers, pOutValue->keys);
    m_Leds = leds.storage;

    // 修飾情報を設定
    pOutValue->modifiers.Set<KeyboardModifier::Control>(
        pOutValue->keys.Test<KeyboardKey::LeftControl>() ||
        pOutValue->keys.Test<KeyboardKey::RightControl>());
    pOutValue->modifiers.Set<KeyboardModifier::Shift>(
        pOutValue->keys.Test<KeyboardKey::LeftShift>() ||
        pOutValue->keys.Test<KeyboardKey::RightShift>());
    pOutValue->modifiers.Set<KeyboardModifier::LeftAlt>(
        pOutValue->keys.Test<KeyboardKey::LeftAlt>());
    pOutValue->modifiers.Set<KeyboardModifier::RightAlt>(
        pOutValue->keys.Test<KeyboardKey::RightAlt>());
    pOutValue->modifiers.Set<KeyboardModifier::Gui>(
        pOutValue->keys.Test<KeyboardKey::LeftGui>() ||
        pOutValue->keys.Test<KeyboardKey::RightGui>());
    pOutValue->modifiers.Set<KeyboardModifier::CapsLock
        >(leds.Get<LedField::CapsLock>());
    pOutValue->modifiers.Set<KeyboardModifier::ScrollLock
        >(leds.Get<LedField::ScrollLock>());
    pOutValue->modifiers.Set<KeyboardModifier::NumLock
        >(leds.Get<LedField::NumLock>());
}

::nn::Result KeyboardDriver::SendLockKeyEvent(
    ::nn::hid::system::KeyboardLockKeyEventSet value) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());

    LedPack leds = { static_cast<uint8_t>(m_Leds) };

    // Num Lock のイベントを処理

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::NumLockOn>())
    {
        leds.Set<LedField::NumLock>(true);
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::NumLockOff>())
    {
        leds.Set<LedField::NumLock>(false);
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::NumLockToggle>())
    {
        leds.Set<LedField::NumLock>(!leds.Get<LedField::NumLock>());
    }

    // Caps Lock のイベントを処理

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::CapsLockOn>())
    {
        leds.Set<LedField::CapsLock>(true);
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::CapsLockOff>())
    {
        leds.Set<LedField::CapsLock>(false);
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::CapsLockToggle>())
    {
        leds.Set<LedField::CapsLock>(!leds.Get<LedField::CapsLock>());
    }

    // Scroll Lock のイベントを処理

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::ScrollLockOn>())
    {
        leds.Set<LedField::ScrollLock>(true);
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::ScrollLockOff>())
    {
        leds.Set<LedField::ScrollLock>(false);
    }

    if (value.Test<::nn::hid::system::KeyboardLockKeyEvent::ScrollLockToggle>())
    {
        leds.Set<LedField::ScrollLock>(!leds.Get<LedField::ScrollLock>());
    }

    m_Leds = leds.storage;

    NN_RESULT_SUCCESS;
}

void KeyboardDriver::Attach(size_t port)
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_UNUSED(port);
    const int count = ++m_Count;
    NN_SDK_ASSERT_MINMAX(count, 1, static_cast<int>(PortCount));
    NN_UNUSED(count);
}

void KeyboardDriver::Detach(size_t port)
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    NN_UNUSED(port);
    const int count = --m_Count;
    NN_SDK_ASSERT_MINMAX(count, 0, static_cast<int>(PortCount - 1));
    NN_UNUSED(count);
}

uint8_t KeyboardDriver::GetLedPattern() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    return m_Leds;
}

void KeyboardDriver::SetReport(size_t port, uint64_t reports) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_RANGE(port, 0u, static_cast<size_t>(PortCount));
    NN_SDK_REQUIRES(!m_ActivationCount.IsZero());
    m_Reports[port] = reports;
}

}}} // namespace nn::hid::detail
