﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace hid { namespace detail {

//!< ハッシュテーブルの状態を表す列挙体です。
enum class HashTableState
{
    Empty,
    InUse,
    Removed,
};

//!< ハッシュテーブルを扱うクラスです。
template <typename KeyT, typename ValueT, typename HasherT, size_t N>
class HashTable final
{
private:
    HashTableState m_States[N];

    KeyT m_Keys[N];

    ValueT m_Values[N];

public:
    HashTable() NN_NOEXCEPT
    {
        for (HashTableState& state : m_States)
        {
            state = HashTableState::Empty;
        }
    }

    //!< キーに関連付けられた値を取得します。
    ValueT* Get(const KeyT& key) NN_NOEXCEPT
    {
        const size_t index = this->Find(key);

        return (N <= index) ? nullptr : &m_Values[index];
    }

    //!< キーに関連付けられた値を取得します。
    const ValueT* Get(const KeyT& key) const NN_NOEXCEPT
    {
        const size_t index = this->Find(key);

        return (N <= index) ? nullptr : &m_Values[index];
    }

    //!< キーに関連付けて値を登録します。
    ValueT* Set(const KeyT& key, const ValueT& value) NN_NOEXCEPT
    {
        size_t position = N;

        bool exists = false;

        for (size_t i = 0, index = HasherT::Get(key); i < N; ++i, ++index)
        {
            index = index % N;

            const HashTableState state = m_States[index];

            if (state == HashTableState::Empty)
            {
                if (position == N)
                {
                    position = index;
                }

                break;
            }

            if (state == HashTableState::InUse)
            {
                if (m_Keys[index] == key)
                {
                    position = index;

                    exists = true;

                    break;
                }
            }
            else if (state == HashTableState::Removed)
            {
                if (position == N)
                {
                    position = index;
                }
            }
        }

        if (position == N)
        {
            return nullptr;
        }
        else
        {
            if (!exists)
            {
                m_States[position] = HashTableState::InUse;

                m_Keys[position] = key;
            }

            m_Values[position] = value;

            return &m_Values[position];
        }
    }

    //!< キーに関連付けられた値を削除します。
    void Remove(const KeyT& key) NN_NOEXCEPT
    {
        const size_t index = this->Find(key);

        if (index < N)
        {
            m_States[index] = HashTableState::Removed;
        }
    }

private:
    size_t Find(const KeyT& key) const NN_NOEXCEPT
    {
        for (size_t i = 0, index = HasherT::Get(key); i < N; ++i, ++index)
        {
            index = index % N;

            if (m_States[index] == HashTableState::Empty)
            {
                break;
            }

            if (m_States[index] == HashTableState::InUse)
            {
                if (m_Keys[index] == key)
                {
                    return index;
                }
            }
        }

        return N;
    }
};

}}} // namespace nn::hid::detail
