﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath> // for pow
#include <limits> // for numeric_limits

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_Gyroscope.h"
#include "hid_SixAxisSensorMathUtility.h"

namespace
{

const int16_t               DriftReviseRange             = ::std::numeric_limits<int16_t>::max() - 30;

const nn::util::Float3 DefaultZeroCount = {{{ 0.f, 0.f, 0.f }}}; //!< ゼロ点のデフォルト [count]
const nn::util::Float3 DefaultSensitivityCount = {{{ 13371.f, 13371.f, 13371.f }}}; //!< 感度のデフォルト [count]

void ApproximateByFactor( float* pOut, const float& target, const float& factor )
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    // factor が 1.0f に近いほど pOut は target に近づく
    *pOut += ( target - *pOut ) * factor;
}

void CalcZeroPlay( float* pOutValue, const float& playRadius )
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    if( -playRadius < *pOutValue &&
        *pOutValue  < playRadius )
    {
        *pOutValue = 0.0f;
    }
}

float GetRevisePower(float angularVelocity,
                     float samplingInterval,
                     float magnification)
{
    float       f1;
    float       v1;

    v1 = magnification * angularVelocity;
    f1 = samplingInterval * 360.0f * v1;
    f1 = f1 * f1 * 0.0001f + 1.0f;
    f1 *= samplingInterval * 2 * nn::hid::detail::Pi * v1;

    return f1;
}

void GetZeroDriftRevisePower(float* pOutPower,
                             int32_t stableCountTimes,
                             int32_t stableCountTimesMax)
{
    float work = static_cast<float>(stableCountTimes);
    work /= stableCountTimesMax;
    *pOutPower = ::std::pow(work, 32);
}

}

namespace nn { namespace hid { namespace detail {

Gyroscope::Gyroscope() NN_NOEXCEPT
    : m_ZeroCount(DefaultZeroCount)
    , m_SensitivityCount(DefaultSensitivityCount)
    , m_Scale()
#ifdef NERD_USE_DENOISED_ROTATION_SPEED
    , m_ZeroCountTotalWeight(0) // TODO pass this to GyroscopeSettings
    , m_ZeroCountTotalWeightCap(250)
    , m_DenoisedRotationSpeed()
#endif
    , m_LastAngularVelocityCount()
    , m_StableCountSum()
    , m_StableCountTimes()
    , m_AngularVelocityMax()
    , m_AngularVelocityMin()
    , m_IsAtRest(false)
    , m_pGyroscopeSetting(nullptr)
{
    // 何もしない
}

void Gyroscope::SetGyroscopeSetting(const GyroscopeSetting* const pSetting) NN_NOEXCEPT
{
    m_pGyroscopeSetting = pSetting;
}

void Gyroscope::GetState(nn::util::Float3* pOutState,
                         const nn::util::Float3& count,
                         const nn::hid::SixAxisSensorAttributeSet& attributes) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);

    // 内挿補間でないときにのみ内部状態を更新します。
    if (attributes.Test<nn::hid::SixAxisSensorAttribute::IsInterpolated>() == false)
    {
        nn::util::Float3 angularVelocityCount; // 最新の角速度カウント

        // 静止判定を行い内部パラメータを更新
        UpdateGyroParameter(&m_StableCountTimes.x,
                            &m_StableCountSum.x,
                            &m_AngularVelocityMax.x,
                            &m_AngularVelocityMin.x,
                            count.x,
                            m_LastAngularVelocityCount.x, // Here could be &m_DenoisedRotationSpeed.x if NERD_USE_DENOISED_ROTATION_SPEED
                            m_Scale.x);

        UpdateGyroParameter(&m_StableCountTimes.y,
                            &m_StableCountSum.y,
                            &m_AngularVelocityMax.y,
                            &m_AngularVelocityMin.y,
                            count.y,
                            m_LastAngularVelocityCount.y, // Here could be &m_DenoisedRotationSpeed.y if NERD_USE_DENOISED_ROTATION_SPEED
                            m_Scale.y);

        UpdateGyroParameter(&m_StableCountTimes.z,
                            &m_StableCountSum.z,
                            &m_AngularVelocityMax.z,
                            &m_AngularVelocityMin.z,
                            count.z,
                            m_LastAngularVelocityCount.z, // Here could be &m_DenoisedRotationSpeed.z if NERD_USE_DENOISED_ROTATION_SPEED
                            m_Scale.z);

        // 静止判定
        m_IsAtRest = (m_StableCountTimes.x == ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count &&
                      m_StableCountTimes.y == ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count &&
                      m_StableCountTimes.z == ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count);

        // 静止期間が長いほど静止時の角速度カウント平均に現在の角速度カウントを近づける
        ReviseAngularVelocityCountByStability(&angularVelocityCount, count);

        // 静止期間が長いほどゼロ点を現在の角速度カウントに近づける
        if(m_pGyroscopeSetting->enableZeroDrift)
        {
            UpdateZeroCount(angularVelocityCount);
        }

        // 内部で確保された最新の角速度カウント値を更新する。
        UpdateLastAngularVelocityCount(angularVelocityCount);

        // 更新されたゼロ点を使って角速度値に変換する
        ConvertToAngularVelocity(pOutState, angularVelocityCount);
    }
    else
    {
        ConvertToAngularVelocity(pOutState, count);
    }


    // 遊び補正
    ReviseAngularVelocityByZeroPlay(pOutState);
}

void Gyroscope::SetScaleValue() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);

    // TODO, change this to use variables ?
    // The sensitivity count default values come from a jig turning at 936 dps. 13371 value was then obtained, @+-2000dps settings
    const float Delta        = 936.0f - 0.0f; // 936dpsスケール - ゼロ点オフセット
    const float ConvertScale = 1.0f / 360.0f; // 360dps を 1.0 とする単位系に変換するための係数

    m_Scale.x = Delta / (m_SensitivityCount.x) * ConvertScale;
    m_Scale.y = Delta / (m_SensitivityCount.y) * ConvertScale;
    m_Scale.z = Delta / (m_SensitivityCount.z) * ConvertScale;
}

//!< 6軸センサーのキャリブレーションを実行します。
void Gyroscope::SetCalibrationValue( const nn::xcd::SensorState& gyroscopeOrigin, const nn::xcd::SensorState& gyroscopeSensitivity ) NN_NOEXCEPT
{
    m_ZeroCount.x  = static_cast<float>(gyroscopeOrigin.x);
    m_ZeroCount.y  = static_cast<float>(gyroscopeOrigin.y);
    m_ZeroCount.z  = static_cast<float>(gyroscopeOrigin.z);

    if (gyroscopeSensitivity.x == 0 ||
        gyroscopeSensitivity.y == 0 ||
        gyroscopeSensitivity.z == 0)
    {
        m_SensitivityCount = DefaultSensitivityCount;
    }
    else
    {
        m_SensitivityCount.x = static_cast<float>(gyroscopeSensitivity.x);
        m_SensitivityCount.y = static_cast<float>(gyroscopeSensitivity.y);
        m_SensitivityCount.z = static_cast<float>(gyroscopeSensitivity.z);
    }

    ShiftCalibrationValue(m_pGyroscopeSetting->shiftOrigin,
                          m_pGyroscopeSetting->shiftSensitivity);

    SetScaleValue();
}

void Gyroscope::ShiftCalibrationValue(const float& origin,
                                      const float& sensitivity) NN_NOEXCEPT
{
    m_SensitivityCount.x *= sensitivity;
    m_SensitivityCount.y *= sensitivity;
    m_SensitivityCount.z *= sensitivity;

    const float Delta = 936.0f - 0.0f; // 936dpsスケール - ゼロ点オフセット
    const float ConvertScale = 1.0f / 360.0f; // 360dps を 1.0 とする単位系に変換するための係数
    const float Coef = Delta * ConvertScale;

    m_ZeroCount.x -= origin * static_cast<float>(m_SensitivityCount.x / Coef);
    m_ZeroCount.y -= origin * static_cast<float>(m_SensitivityCount.y / Coef);
    m_ZeroCount.z -= origin * static_cast<float>(m_SensitivityCount.z / Coef);
}

bool Gyroscope::IsAtRest() const NN_NOEXCEPT
{
    return m_IsAtRest;
}

void Gyroscope::ResetIsAtRest() NN_NOEXCEPT
{
    m_IsAtRest = false;
}

void Gyroscope::UpdateGyroParameter(int32_t* pStableCountTimes,
                                    float* pStableCountSum,
                                    float* pAngularVelocityMax,
                                    float* pAngularVelocityMin,
                                    const float currentCount,
                                    const float previousCount, //float* pDenoisedRotationSpeed, if NERD_USE_DENOISED_ROTATION_SPEED
                                    const float& scale) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStableCountTimes);
    NN_SDK_REQUIRES_NOT_NULL(pStableCountSum);
    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);


    const float ZeroDriftCountRange = ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].radius / scale;  //!< 前回サンプルとの差がこの値以下なら静止と判定する
    const int   StableCountTimesMax = ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count;   //!< 平均を取る回数の最大値

#ifdef NERD_USE_DENOISED_ROTATION_SPEED
    const float RotationSpeedCountForStabilityDetection = 0.035f;
    const float previousDenoisedRotationSpeed = *pDenoisedRotationSpeed;
    //janos: it's important to denoise the sensor values to establish that the rotation speed is constant. It allows us to be much more strict, and thus removes the probability of false positives.
    const float denoisedRotationSpeed = previousDenoisedRotationSpeed + (currentCount - previousDenoisedRotationSpeed) * RotationSpeedCountForStabilityDetection;
    *pDenoisedRotationSpeed = denoisedRotationSpeed;
    const float DeltaCount = denoisedRotationSpeed - previousDenoisedRotationSpeed;

    if (*pAngularVelocityMax < denoisedRotationSpeed)
    {
        *pAngularVelocityMax = denoisedRotationSpeed;
    }
    if (*pAngularVelocityMin > denoisedRotationSpeed)
    {
        *pAngularVelocityMin = denoisedRotationSpeed;
    }

    const float CountRange = *pAngularVelocityMax - *pAngularVelocityMin;
#else
    const float DeltaCount          = currentCount - previousCount;


    if(*pAngularVelocityMax < currentCount)
    {
        *pAngularVelocityMax = currentCount;
    }
    if(*pAngularVelocityMin > currentCount)
    {
        *pAngularVelocityMin = currentCount;
    }
    const float CountRange = *pAngularVelocityMax - *pAngularVelocityMin;
#endif

    //!< ゼロ点付近であれば静止と見做す
    const bool  IsStable = ( -ZeroDriftCountRange < DeltaCount && DeltaCount < ZeroDriftCountRange );

    //!< 静止判定時の値域が十分狭いか
    const bool IsInRange = (CountRange < ZeroDriftCountRange * 2);

    //!< カウント値が振り切れているときはゼロ点の動的補正を無効に
    const bool IsValid = (-DriftReviseRange < currentCount && currentCount  < DriftReviseRange);

    if(IsStable && IsInRange && IsValid)
    {
        if( *pStableCountTimes < StableCountTimesMax )
        {
            (*pStableCountSum) += currentCount;
            (*pStableCountTimes)++;
        }
    }
    else
    {
        // 状態のクリア
        *pStableCountSum = currentCount;
        *pStableCountTimes = 1;
#ifdef NERD_USE_DENOISED_ROTATION_SPEED
        *pAngularVelocityMax = denoisedRotationSpeed;
        *pAngularVelocityMin = denoisedRotationSpeed;
#else
        *pAngularVelocityMax = ::std::numeric_limits<int16_t>::min();
        *pAngularVelocityMin = ::std::numeric_limits<int16_t>::max();
#endif
    }
}

void Gyroscope::UpdateZeroCount(const nn::util::Float3& count) NN_NOEXCEPT
{
#ifdef NERD_USE_DENOISED_ROTATION_SPEED
    const int16_t                            MinStabilityCountToConsiderUpdatingZeroDrift = 100;
    //we don't want to update the zero drift value, even by a bit, if we are not stable.
    if (m_StableCountTimes.x < MinStabilityCountToConsiderUpdatingZeroDrift)
        return;
    if (m_StableCountTimes.y < MinStabilityCountToConsiderUpdatingZeroDrift)
        return;
    if (m_StableCountTimes.z < MinStabilityCountToConsiderUpdatingZeroDrift)
        return;
    const int16_t minStableCount = int16_t(std::min(m_StableCountTimes.x, std::min(m_StableCountTimes.y, m_StableCountTimes.z)));
    const float revisePower = float(minStableCount - MinStabilityCountToConsiderUpdatingZeroDrift) / (ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count - MinStabilityCountToConsiderUpdatingZeroDrift);
    const float weightedRevisePower = revisePower / (1 + m_ZeroCountTotalWeight);
    if (m_ZeroCountTotalWeight + revisePower <= m_ZeroCountTotalWeightCap)
        m_ZeroCountTotalWeight += revisePower;

    ApproximateByFactor(&m_ZeroCount.x, count.x, weightedRevisePower);
    ApproximateByFactor(&m_ZeroCount.y, count.y, weightedRevisePower);
    ApproximateByFactor(&m_ZeroCount.z, count.z, weightedRevisePower);
    return;
#endif

    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);

    float work;

    GetZeroDriftRevisePower(&work, m_StableCountTimes.x, ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count);
    ApproximateByFactor(&m_ZeroCount.x, count.x, work);

    GetZeroDriftRevisePower(&work, m_StableCountTimes.y, ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count);
    ApproximateByFactor(&m_ZeroCount.y, count.y, work);

    GetZeroDriftRevisePower(&work, m_StableCountTimes.z, ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count);
    ApproximateByFactor(&m_ZeroCount.z, count.z, work);
}

void Gyroscope::UpdateLastAngularVelocityCount(const nn::util::Float3& angularVelocity) NN_NOEXCEPT
{
    m_LastAngularVelocityCount = angularVelocity;
}

void Gyroscope::ConvertToAngularVelocity(nn::util::Float3* pOut, const nn::util::Float3& count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);

    pOut->x = ( count.x - m_ZeroCount.x ) * m_Scale.x * m_pGyroscopeSetting->magnification.x;
    pOut->y = ( count.y - m_ZeroCount.y ) * m_Scale.y * m_pGyroscopeSetting->magnification.y;
    pOut->z = ( count.z - m_ZeroCount.z ) * m_Scale.z * m_pGyroscopeSetting->magnification.z;
}

void Gyroscope::ReviseAngularVelocityByZeroPlay( nn::util::Float3* pOut ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);

    if(m_pGyroscopeSetting->enableZeroPlay )
    {
        CalcZeroPlay( &pOut->x, m_pGyroscopeSetting->zeroPlayRadius );
        CalcZeroPlay( &pOut->y, m_pGyroscopeSetting->zeroPlayRadius );
        CalcZeroPlay( &pOut->z, m_pGyroscopeSetting->zeroPlayRadius );
    }
}

void Gyroscope::ReviseAngularVelocityCountByStability(nn::util::Float3* pOut, const nn::util::Float3& count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);
    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);

    *pOut = count;
    const float ooZeroDriftCount = 1.f / ZeroDrift[m_pGyroscopeSetting->zeroDriftMode].count;

    ApproximateByFactor(
        &pOut->x,
        m_StableCountSum.x / m_StableCountTimes.x,
        m_StableCountTimes.x * ooZeroDriftCount
    );

    ApproximateByFactor(
        &pOut->y,
        m_StableCountSum.y / m_StableCountTimes.y,
        m_StableCountTimes.y * ooZeroDriftCount
    );

    ApproximateByFactor(
        &pOut->z,
        m_StableCountSum.z / m_StableCountTimes.z,
        m_StableCountTimes.z * ooZeroDriftCount
    );
}

void Gyroscope::UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& angularVelocity, const float& samplingInterval ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDirection);
    NN_SDK_REQUIRES_NOT_NULL(m_pGyroscopeSetting);

    // __VPADCalcGyroDir を参照
    nn::hid::DirectionState   d1;
    float f1;
    d1 = *pOutDirection ;

    // 方向の回転を直角方向へのベクトル加算で行っている為に、
    // 角速度の応じて加算ベクトルを大きくする近似処理を行っている。
    // 近似式はオリジナル

    //----- Ｘ方向回転処理
    f1 = GetRevisePower(angularVelocity.x,
                        samplingInterval,
                        m_pGyroscopeSetting->directionMagnification.x);

    pOutDirection->y.x += f1 * d1.z.x ;
    pOutDirection->y.y += f1 * d1.z.y ;
    pOutDirection->y.z += f1 * d1.z.z ;

    pOutDirection->z.x -= f1 * d1.y.x ;
    pOutDirection->z.y -= f1 * d1.y.y ;
    pOutDirection->z.z -= f1 * d1.y.z ;

    //----- Ｙ方向回転処理
    f1 = GetRevisePower(angularVelocity.y,
                        samplingInterval,
                        m_pGyroscopeSetting->directionMagnification.y);

    pOutDirection->z.x += f1 * d1.x.x ;
    pOutDirection->z.y += f1 * d1.x.y ;
    pOutDirection->z.z += f1 * d1.x.z ;

    pOutDirection->x.x -= f1 * d1.z.x ;
    pOutDirection->x.y -= f1 * d1.z.y ;
    pOutDirection->x.z -= f1 * d1.z.z ;

    //----- Ｚ方向回転処理
    f1 = GetRevisePower(angularVelocity.z,
                        samplingInterval,
                        m_pGyroscopeSetting->directionMagnification.z);

    pOutDirection->x.x += f1 * d1.y.x ;
    pOutDirection->x.y += f1 * d1.y.y ;
    pOutDirection->x.z += f1 * d1.y.z ;

    pOutDirection->y.x -= f1 * d1.x.x ;
    pOutDirection->y.y -= f1 * d1.x.y ;
    pOutDirection->y.z -= f1 * d1.x.z ;

    //----- 正規化
    nn::hid::detail::NormalizeDirectionState( pOutDirection, 2.999f ) ;

}

}}} // namespace nn::hid::detail
