﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cmath>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Gesture.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/detail/hid_Log.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_MathTypes.h>

#include "hid_CommonStateUtility.h"
#include "hid_GestureRecognizer.h"

#if 0
//!< ジェスチャ認識器のデバッグログを出力します。
#define NN_HID_GESTURE_TRACE(...) \
    NN_DETAIL_HID_TRACE("[hid::Gesture] " __VA_ARGS__)
#else
#define NN_HID_GESTURE_TRACE(...)
#endif

namespace nn { namespace hid { namespace detail {

namespace {

//!< ステートマシンの状態を更新します。
void UpdateMachineState(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT;

//!< ジェスチャの状態を更新します。
void UpdateGestureState(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT;

} // namespace

GestureRecognizer::GestureRecognizer() NN_NOEXCEPT
    : m_GestureState()
    , m_MachineState()
{
    m_MachineState.eventNumber = -1;
    m_MachineState.contextNumber = -1;
    this->Reset();
}

const GestureState& GestureRecognizer::GetState() const NN_NOEXCEPT
{
    return m_GestureState;
}

void GestureRecognizer::Reset() NN_NOEXCEPT
{
    m_GestureState = GestureState();
    m_GestureState.eventNumber = -1;

    const int64_t eventNumber = m_MachineState.eventNumber;
    const int64_t contextNumber = m_MachineState.contextNumber;
    m_MachineState = GestureMachineState();
    m_MachineState.eventNumber = eventNumber + 1;
    m_MachineState.contextNumber = contextNumber + 1;
    m_MachineState.flags.Set<GestureMachineFlag::IsNewTouch>();
}

void GestureRecognizer::Update(
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    UpdateMachineState(&m_MachineState, touches, count);

    UpdateGestureState(&m_GestureState, m_MachineState);
}

namespace {

//!< オートマトン上の位置を表す列挙体です。
enum class Position : int32_t
{
    Idle0,      //!< 待機
    Idle1,      //!< 待機 (タッチ待ち)
    Complete0,  //!< 完了
    Cancel0,    //!< キャンセル
    Touch0,     //!< タッチ
    Touch1,     //!< タッチ (タップ待ち)
    Press0,     //!< プレス
    Tap0,       //!< タップ
    Pan0,       //!< パン
    Pan1,       //!< パン (完了待ち)
    Pan2,       //!< パン (補完)
    Swipe0,     //!< スワイプ
    Pinch0,     //!< ピンチ
    Pinch1,     //!< ピンチ (完了待ち)
    Rotate0,    //!< 回転
    Rotate1,    //!< 回転 (完了待ち)
};

//!< ジェスチャの方向を表す列挙体です。
enum class Direction : int32_t
{
    None,   //!< 無し
    Left,   //!< 左
    Up,     //!< 上
    Right,  //!< 右
    Down,   //!< 下
};

//!< サポートされるタッチの最大数
const int32_t TouchCountMax = 2;

//!< 受け付け処理の待ち時間 [ms]
const int32_t ReceptionTimeout = 30;

//!< プレス操作の待ち時間 [ms]
const int32_t PressTimeout = 500;

//!< ダブルタップ操作の待ち時間 [ms]
const int32_t DoubleTapTimeout = 300;

//!< スワイプ操作として認識される速度の下限
const int32_t SwipeSpeed = 700;

//!< タッチ操作の遊び
const int32_t TouchPlay = 8;

//!< 許容されるタップ位置のズレ
const int32_t TapPlay = 40;

//!< 変形操作の受付に必要な距離
const float MinimumDistance = 150;

//!< 速度計算の遅延の大きさ
const int32_t VelocityDelay = 8;

//!< パン操作の遊び
const int32_t PanPlay = 20;

//!< ピンチ操作の最小値
const float MinimumPinch = 2;

//!< ピンチ操作の遊び
const float PinchPlay = 18;

//!< 回転操作の最小値
const float MinimumRotate = 1;

//!< 回転操作の遊び
const float RotatePlay = 10;

//!< 原点からの距離を計算します。
float GetDistance(int32_t a, int32_t b) NN_NOEXCEPT
{
    if (a == 0 && b == 0)
    {
        return 0;
    }
    else
    {
        return ::std::sqrt(static_cast<float>(a * a + b * b));
    }
}

//!< 回転角度を取得します。
float GetRotationAngle(float a, float b) NN_NOEXCEPT
{
    const float angle = b - a;

    if (180 <= angle)
    {
        return (angle - 360);
    }
    else
    {
        if (angle <= -180)
        {
            return (angle + 360);
        }
        else
        {
            return angle;
        }
    }
}

//!< 絶対値が指定された値を超えたか否かを表す値を返します。
template<typename T>
bool IsAbsoluteValueOver(const T& value, const T& limit) NN_NOEXCEPT
{
    return (value <= -limit || limit <= value);
}

//!< 制限速度を超えたか否かを表す値を返します。
bool IsOverVelocityLimit(
    const ::nn::util::Float2& velocity, int32_t limit) NN_NOEXCEPT
{
    return (velocity.x * velocity.x + velocity.y * velocity.y >= limit * limit);
}

//!< デッドゾーンを差し引いた値を返します。
template<typename T>
T SubtractDeadZone(const T& value, const T& deadzone) NN_NOEXCEPT
{
    if (0 <= value)
    {
        if (value <= deadzone)
        {
            return 0;
        }
        else
        {
            return (value - deadzone);
        }
    }
    else
    {
        if (-deadzone <= value)
        {
            return 0;
        }
        else
        {
            return (value + deadzone);
        }
    }
}

//!< タッチが隣接しているか否かを表す値を返します。
bool Adjacents(
    const TouchState& lhs, const TouchState& rhs, int32_t play) NN_NOEXCEPT
{
    auto checker = [] (int32_t delta, int32_t play) NN_NOEXCEPT
    {
        return (-play <= delta && delta <= play);
    };

    return (checker(lhs.x - rhs.x, play) && checker(lhs.y - rhs.y, play));
}

//!< 重心を取得します。
void GetCentroid(
    int32_t* pOutX, int32_t* pOutY,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutX);
    NN_SDK_REQUIRES_NOT_NULL(pOutY);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 1, TouchCountMax);

    switch (count)
    {
    case 0:
        *pOutX = 0;
        *pOutY = 0;

        break;

    case 1:
        *pOutX = touches[0].x;
        *pOutY = touches[0].y;

        break;

    default:
        *pOutX = touches[0].x;
        *pOutY = touches[0].y;

        for (int i = 1; i < count; ++i)
        {
            *pOutX += touches[i].x;
            *pOutY += touches[i].y;
        }

        *pOutX /= count;
        *pOutY /= count;

        break;
    }
}

//!< タイマを更新します。
void UpdateTimer(
    ::nn::TimeSpanType* pTimer, const ::nn::TimeSpanType& deltaTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pTimer);

    ::nn::TimeSpanType& timer = *pTimer;

    if (timer < deltaTime)
    {
        timer = ::nn::TimeSpan();
    }
    else
    {
        timer = timer - deltaTime;
    }
}

//!< 速度を更新します
void UpdateVelocity(
    ::nn::util::Float2* pVelocity,
    int32_t dx, int32_t dy, ::nn::TimeSpanType dt) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVelocity);

    ::nn::util::Float2& velocity = *pVelocity;

    const int32_t delay = VelocityDelay;

    const int64_t timeSpan = dt.GetMicroSeconds();

    if (timeSpan > 0)
    {
        const int64_t count =
            ::nn::TimeSpan::FromSeconds(1).GetMicroSeconds() / timeSpan;

        velocity.x = (velocity.x * (delay - 1) + dx * count) / delay;
        velocity.y = (velocity.y * (delay - 1) + dy * count) / delay;
    }
}

//!< タッチ識別子の構成が同一か否かを表す値を返します。
bool HasSameFingerIds(
    const GestureTouchSet& touchSet,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(touchSet.count, 0, GesturePointCountMax);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    if (touchSet.count != count)
    {
        return false;
    }

    for (int i = 0; i < count; ++i)
    {
        if (touchSet.touches[i].fingerId != touches[i].fingerId)
        {
            return false;
        }
    }

    return true;
}

//!< 同じ位置がタッチされたか否かを表す値を返します。
bool IsSamePositionTouched(
    const GestureTouchSet& touchSet,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(touchSet.count, 0, GesturePointCountMax);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    if (touchSet.count != count)
    {
        return false;
    }

    if (count == 1)
    {
        if (Adjacents(touchSet.touches[0], touches[0], TouchPlay))
        {
            return true;
        }
    }

    if (count == 2)
    {
        if (Adjacents(touchSet.touches[0], touches[0], TouchPlay) &&
            Adjacents(touchSet.touches[1], touches[1], TouchPlay))
        {
            return true;
        }
    }

    return false;
}

//!< 同じ位置がタップされたか否かを表す値を返します。
bool IsSamePositionTapped(
    const GestureTouchSet& touchSet,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(touchSet.count, 0, GesturePointCountMax);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    if (touchSet.count != count)
    {
        return false;
    }

    if (count == 1)
    {
        if (Adjacents(touchSet.touches[0], touches[0], TapPlay))
        {
            return true;
        }
    }

    if (count == 2)
    {
        if (Adjacents(touchSet.touches[0], touches[0], TapPlay) &&
            Adjacents(touchSet.touches[1], touches[1], TapPlay))
        {
            return true;
        }

        if (Adjacents(touchSet.touches[0], touches[1], TapPlay) &&
            Adjacents(touchSet.touches[1], touches[0], TapPlay))
        {
            return true;
        }
    }

    return false;
}

//!< ジェスチャを構成するタッチの集合に値を設定します。
void SetGestureTouchSet(
    GestureTouchSet* pOutValue,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureTouchSet touchSet = {};

    touchSet.count = ::std::min(
        ::std::max(static_cast<int32_t>(0), count), GesturePointCountMax);

    ::std::copy(touches, touches + touchSet.count, touchSet.touches);

    *pOutValue = touchSet;
}

//!< タッチ操作のタイムアウトを検知したか否かを表す値を返します。
bool DetectsTouchTimeout(const GestureMachineState& state) NN_NOEXCEPT
{
    return (state.touchTimer.GetNanoSeconds() == 0);
}

//!< ダブルタップ操作のタイムアウトを検知したか否かを表す値を返します。
bool DetectsDoubleTapTimeout(const GestureMachineState& state) NN_NOEXCEPT
{
    return (state.doubleTapTimer.GetNanoSeconds() == 0);
}

//!< Idle0 に状態を遷移します。
void MoveToIdle0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.contextNumber += 1;
    state.position = static_cast<int32_t>(Position::Idle0);
    state.flags.Reset();
    state.flags.Set<GestureMachineFlag::IsNewTouch>();
    state.touch = GestureTouchSet();
    state.velocity = ::nn::util::Float2();

    NN_HID_GESTURE_TRACE("Move to Idle0\n");
}

//!< Idle1 に状態を遷移します。
void MoveToIdle1(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.position = static_cast<int32_t>(Position::Idle1);
    state.touchTimer = ::nn::TimeSpan::FromMilliSeconds(ReceptionTimeout);

    NN_HID_GESTURE_TRACE("Move to Idle1\n");
}

//!< Complete0 に状態を遷移します。
void MoveToComplete0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Complete0);

    NN_HID_GESTURE_TRACE("Move to Complete0\n");
}

//!< Cancel0 に状態を遷移します。
void MoveToCancel0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Cancel0);
    state.flags.Reset<GestureMachineFlag::IsNewTouch>();
    state.doubleTapTimer = ::nn::TimeSpan();

    NN_HID_GESTURE_TRACE("Move to Cancel0\n");
}

//!< Touch0 に状態を遷移します。
void MoveToTouch0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.contextNumber += 1;
    state.position = static_cast<int32_t>(Position::Touch0);
    state.touchTimer = ::nn::TimeSpan::FromMilliSeconds(PressTimeout);

    if (state.touch.count != 2)
    {
        state.distance = 0;
        state.angle = 0;
    }
    else
    {
        const int32_t a = state.touch.touches[1].y - state.touch.touches[0].y;
        const int32_t b = state.touch.touches[1].x - state.touch.touches[0].x;
        state.distance = GetDistance(a, b);
        state.angle =
            ::nn::util::RadianToDegree(
                ::nn::util::Atan2Est(
                    static_cast<float>(a), static_cast<float>(b)));
    }

    GetCentroid(&state.x, &state.y, state.touch.touches, state.touch.count);

    NN_HID_GESTURE_TRACE(
        "Move to Touch0:  %d (%4d, %4d) (%4d, %4d) %s\n",
        state.touch.count,
        state.touch.touches[0].x, state.touch.touches[0].y,
        state.touch.touches[1].x, state.touch.touches[1].y,
        state.flags.Test<GestureMachineFlag::IsNewTouch>() ? "new" : "reuse");
}

//!< Touch1 に状態を遷移します。
void MoveToTouch1(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.position = static_cast<int32_t>(Position::Touch1);
    state.touchTimer = ::nn::TimeSpan::FromMilliSeconds(ReceptionTimeout);

    NN_HID_GESTURE_TRACE("Move to Touch1\n");
}

//!< Press0 に状態を遷移します。
void MoveToPress0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Press0);
    state.doubleTapTimer = ::nn::TimeSpan();

    NN_HID_GESTURE_TRACE("Move to Press0\n");
}

//!< Tap0 に状態を遷移します。
void MoveToTap0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Tap0);

    const GestureTouchSet& doubleTap = state.doubleTap;

    if (!DetectsDoubleTapTimeout(state) &&
        IsSamePositionTapped(state.touch, doubleTap.touches, doubleTap.count))
    {
        state.flags.Set<GestureMachineFlag::IsDoubleTap>();
    }

    NN_HID_GESTURE_TRACE(
        "Move to Tap0: %s\n",
        state.flags.Test<GestureMachineFlag::IsDoubleTap>() ? "double"
                                                            : "single");
}

//!< Pan0 に状態を遷移します。
void MoveToPan0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Pan0);
    state.doubleTapTimer = ::nn::TimeSpan();

    NN_HID_GESTURE_TRACE(
        "Move to Pan0:      (%4d, %4d) (%4d, %4d)\n",
        state.deltaX, state.deltaY,
        static_cast<int32_t>(state.velocity.x),
        static_cast<int32_t>(state.velocity.y));
}

//!< Pan1 に状態を遷移します。
void MoveToPan1(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.position = static_cast<int32_t>(Position::Pan1);
    state.touchTimer = ::nn::TimeSpan::FromMilliSeconds(ReceptionTimeout);

    NN_HID_GESTURE_TRACE("Move to Pan1\n");
}

//!< Pan2 に状態を遷移します。
void MoveToPan2(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Pan2);

    NN_HID_GESTURE_TRACE("Move to Pan2\n");
}

//!< Swipe0 に状態を遷移します。
void MoveToSwipe0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Swipe0);

    const ::nn::util::Float2& velocity = state.velocity;

    if (velocity.x >= 0)
    {
        if (velocity.y >= 0)
        {
            if (velocity.x >= velocity.y)
            {
                state.direction = static_cast<int32_t>(Direction::Right);
            }
            else
            {
                state.direction = static_cast<int32_t>(Direction::Down);
            }
        }
        else
        {
            if (velocity.x >= -velocity.y)
            {
                state.direction = static_cast<int32_t>(Direction::Right);
            }
            else
            {
                state.direction = static_cast<int32_t>(Direction::Up);
            }
        }
    }
    else
    {
        if (velocity.y >= 0)
        {
            if (-velocity.x >= velocity.y)
            {
                state.direction = static_cast<int32_t>(Direction::Left);
            }
            else
            {
                state.direction = static_cast<int32_t>(Direction::Down);
            }
        }
        else
        {
            if (-velocity.x >= -velocity.y)
            {
                state.direction = static_cast<int32_t>(Direction::Left);
            }
            else
            {
                state.direction = static_cast<int32_t>(Direction::Up);
            }
        }
    }

    const char* const names[] = { "None", "Left", "Up", "Right", "Down" };

    NN_HID_GESTURE_TRACE("Move to Swipe0:  %s\n", names[state.direction]);

    NN_UNUSED(names);
}

//!< Pinch0 に状態を遷移します。
void MoveToPinch0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Pinch0);
    state.doubleTapTimer = ::nn::TimeSpan();

    const float scale = state.scale;
    const auto integer = static_cast<int32_t>(scale);
    const auto decimal = static_cast<int32_t>((scale - integer) * 100);

    NN_HID_GESTURE_TRACE("Move to Pinch0:  %2d.%02d\n", integer, decimal);

    NN_UNUSED(integer);
    NN_UNUSED(decimal);
}

//!< Pinch1 に状態を遷移します。
void MoveToPinch1(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.position = static_cast<int32_t>(Position::Pinch1);
    state.touchTimer = ::nn::TimeSpan::FromMilliSeconds(ReceptionTimeout);

    NN_HID_GESTURE_TRACE("Move to Pinch1\n");
}

//!< Rotate0 に状態を遷移します。
void MoveToRotate0(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.eventNumber += 1;
    state.position = static_cast<int32_t>(Position::Rotate0);
    state.doubleTapTimer = ::nn::TimeSpan();

    const int32_t sign = (0 <= state.rotationAngle) ? 1 : -1;
    const float rotationAngle = sign * state.rotationAngle;
    const auto integer = static_cast<int32_t>(rotationAngle);
    const auto decimal = static_cast<int32_t>((rotationAngle - integer) * 100);

    NN_HID_GESTURE_TRACE(
        "Move to Rotate0: %2d.%02d\n", sign * integer, decimal);

    NN_UNUSED(integer);
    NN_UNUSED(decimal);
}

//!< Rotate1 に状態を遷移します。
void MoveToRotate1(GestureMachineState* pState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;
    state.position = static_cast<int32_t>(Position::Rotate1);
    state.touchTimer = ::nn::TimeSpan::FromMilliSeconds(ReceptionTimeout);

    NN_HID_GESTURE_TRACE("Move to Rotate1\n");
}

//!< パン操作が発生したか否かを表す値を返します。
bool IsPanned(
    const GestureMachineState& state, int32_t x, int32_t y) NN_NOEXCEPT
{
    const int32_t dx = x - state.x;
    const int32_t dy = y - state.y;

    if (state.touch.count == 1 ||
        state.flags.Test<GestureMachineFlag::AcceptsPan>())
    {
        return (dx != 0 || dy != 0);
    }
    else
    {
        return (IsAbsoluteValueOver(dx, PanPlay) ||
                IsAbsoluteValueOver(dy, PanPlay));
    }
}

//!< ピンチ操作が発生したか否かを表す値を返します。
bool IsPinched(
    const GestureMachineState& state, float distance) NN_NOEXCEPT
{
    if (state.touch.count != 2)
    {
        return false;
    }
    else
    {
        const float delta = distance - state.distance;

        if (state.flags.Test<GestureMachineFlag::AcceptsPinch>())
        {
            return IsAbsoluteValueOver(delta, MinimumPinch);
        }
        else
        {
            return IsAbsoluteValueOver(delta, PinchPlay);
        }
    }
}

//!< 回転操作が発生したか否かを表す値を返します。
bool IsRotated(
    const GestureMachineState& state, float angle) NN_NOEXCEPT
{
    if (state.touch.count != 2)
    {
        return false;
    }
    else
    {
        const float delta = GetRotationAngle(state.angle, angle);

        if (state.flags.Test<GestureMachineFlag::AcceptsRotate>())
        {
            return IsAbsoluteValueOver(delta, MinimumRotate);
        }
        else
        {
            return IsAbsoluteValueOver(delta, RotatePlay);
        }
    }
}

//!< 座標値を更新します。
void UpdateCoordinate(
    GestureMachineState* pState, int32_t x, int32_t y) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;

    const int32_t dx = x - state.x;
    const int32_t dy = y - state.y;

    if (state.touch.count == 1 ||
        state.flags.Test<GestureMachineFlag::AcceptsPan>())
    {
        state.deltaX = dx;
        state.deltaY = dy;
    }
    else
    {
        state.deltaX = SubtractDeadZone(dx, PanPlay);
        state.deltaY = SubtractDeadZone(dy, PanPlay);
    }

    state.x = x;
    state.y = y;
    state.flags.Set<GestureMachineFlag::AcceptsPan>();
}

//!< タッチ間距離を更新します。
void UpdateDistance(GestureMachineState* pState, float distance) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_GREATER(pState->distance, 0);
    NN_SDK_REQUIRES_GREATER(distance, 0);

    GestureMachineState& state = *pState;

    if (state.touch.count != 2)
    {
        state.scale = float();
    }
    else
    {
        if (state.flags.Test<GestureMachineFlag::AcceptsPinch>())
        {
            state.scale = distance / state.distance;
        }
        else
        {
            float delta = distance - state.distance;

            delta = SubtractDeadZone(delta, PinchPlay);

            if (0 <= delta && delta < MinimumPinch)
            {
                delta = MinimumPinch;
            }

            if (delta < 0 && -MinimumPinch < delta)
            {
                delta = -MinimumPinch;
            }

            state.scale = distance / (distance - delta);
        }
    }

    state.distance = distance;
    state.flags.Set<GestureMachineFlag::AcceptsPinch>();
}

//!< タッチ間角度を更新します。
void UpdateAngle(GestureMachineState* pState, float angle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);

    GestureMachineState& state = *pState;

    if (state.touch.count != 2)
    {
        state.rotationAngle = float();
    }
    else
    {
        float delta = GetRotationAngle(state.angle, angle);

        if (state.flags.Test<GestureMachineFlag::AcceptsRotate>())
        {
            state.rotationAngle = delta;
        }
        else
        {
            delta = SubtractDeadZone(delta, RotatePlay);

            if (0 <= delta && delta < MinimumRotate)
            {
                delta = MinimumRotate;
            }

            if (delta < 0 && -MinimumRotate < delta)
            {
                delta = -MinimumRotate;
            }

            state.rotationAngle = delta;
        }
    }

    state.angle = angle;
    state.flags.Set<GestureMachineFlag::AcceptsRotate>();
}

//!< タッチの動きを処理します。
void ProcessAction(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 1, TouchCountMax);
    NN_SDK_REQUIRES_EQUAL(pState->touch.count, count);

    GestureMachineState& state = *pState;

    auto x = int32_t();
    auto y = int32_t();
    GetCentroid(&x, &y, touches, count);

    float distance = state.distance;
    float angle = state.angle;
    if (count == 2)
    {
        const int32_t a = touches[1].y - touches[0].y;
        const int32_t b = touches[1].x - touches[0].x;

        distance = GetDistance(a, b);
        angle = ::nn::util::RadianToDegree(
            ::nn::util::Atan2Est(static_cast<float>(a), static_cast<float>(b)));

        if (distance < MinimumDistance)
        {
            distance = MinimumDistance;
            state.distance = distance;
            state.angle = angle;
        }
    }

    const bool isPanned = IsPanned(state, x, y);
    const bool isPinched = IsPinched(state, distance);
    const bool isRotated = IsRotated(state, angle);

    if (!isPanned && !isPinched && !isRotated)
    {
        UpdateVelocity(&state.velocity, 0, 0, state.deltaTime);
    }
    else
    {
        UpdateCoordinate(&state, x, y);

        UpdateVelocity(
            &state.velocity, state.deltaX, state.deltaY, state.deltaTime);

        SetGestureTouchSet(&state.touch, touches, count);

        if (!isPinched && !isRotated)
        {
            MoveToPan0(&state);
        }
        else
        {
            if (!isPinched || !isRotated)
            {
                if (isPinched)
                {
                    UpdateDistance(&state, distance);

                    MoveToPinch0(&state);
                }
                else
                {
                    UpdateAngle(&state, angle);

                    MoveToRotate0(&state);
                }
            }
            else
            {
                const int32_t position = state.position;

                if (position == static_cast<int32_t>(Position::Rotate0))
                {
                    UpdateDistance(&state, distance);

                    MoveToPinch0(&state);
                }
                else
                {
                    UpdateAngle(&state, angle);

                    MoveToRotate0(&state);
                }
            }
        }
    }
}

//!< Idle0 状態を処理します。
void ProcessIdle0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);

    GestureMachineState& state = *pState;

    if (0 < count)
    {
        MoveToIdle1(&state);
    }
}

//!< Idle1 状態を処理します。
void ProcessIdle1(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    if (count == 0)
    {
        MoveToIdle0(&state);
    }
    else
    {
        if (DetectsTouchTimeout(state))
        {
            count = ::std::min<int32_t>(count, TouchCountMax);

            SetGestureTouchSet(&state.touch, touches, count);

            MoveToTouch0(&state);
        }
    }
}

//!< Complete0 状態を処理します。
void ProcessComplete0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);
    NN_UNUSED(count);

    GestureMachineState& state = *pState;

    MoveToIdle0(&state);
}

//!< Cancel0 状態を処理します。
void ProcessCancel0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    if (count == 0)
    {
        MoveToIdle0(&state);
    }
    else
    {
        if (count <= TouchCountMax)
        {
            SetGestureTouchSet(&state.touch, touches, count);

            MoveToTouch0(&state);
        }
    }
}

//!< Touch0 状態を処理します。
void ProcessTouch0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    if (HasSameFingerIds(state.touch, touches, count))
    {
        if (DetectsTouchTimeout(state) &&
            state.flags.Test<GestureMachineFlag::IsNewTouch>() &&
           !state.flags.Test<GestureMachineFlag::AcceptsAction>() &&
            state.touch.count == 1)
        {
            MoveToPress0(&state);
        }
        else
        {
            if (state.flags.Test<GestureMachineFlag::AcceptsAction>())
            {
                ProcessAction(&state, touches, count);
            }
            else
            {
                if (!IsSamePositionTouched(state.touch, touches, count))
                {
                    state.flags.Set<GestureMachineFlag::AcceptsAction>();

                    SetGestureTouchSet(&state.touch, touches, count);
                }
            }
        }
    }
    else
    {
        if ( state.touch.count <= count ||
            !state.flags.Test<GestureMachineFlag::IsNewTouch>() ||
             state.flags.Test<GestureMachineFlag::AcceptsAction>())
        {
            MoveToCancel0(&state);
        }
        else
        {
            if (count == 0)
            {
                MoveToTap0(&state);
            }
            else
            {
                MoveToTouch1(&state);
            }
        }
    }
}

//!< Touch1 状態を処理します。
void ProcessTouch1(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);

    GestureMachineState& state = *pState;

    if (count == 0)
    {
        MoveToTap0(&state);
    }
    else
    {
        if (DetectsTouchTimeout(state))
        {
            MoveToCancel0(&state);
        }
    }
}

//!< Press0 状態を処理します。
void ProcessPress0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    if (HasSameFingerIds(state.touch, touches, count))
    {
        if (state.flags.Test<GestureMachineFlag::AcceptsAction>())
        {
            ProcessAction(&state, touches, count);
        }
        else
        {
            if (!IsSamePositionTouched(state.touch, touches, count))
            {
                state.flags.Set<GestureMachineFlag::AcceptsAction>();

                SetGestureTouchSet(&state.touch, touches, count);
            }
        }
    }
    else
    {
        if (0 < count)
        {
            MoveToCancel0(&state);
        }
        else
        {
            MoveToComplete0(&state);
        }
    }
}

//!< Tap0 状態を処理します。
void ProcessTap0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);
    NN_UNUSED(count);

    GestureMachineState& state = *pState;

    if (state.flags.Test<GestureMachineFlag::IsDoubleTap>())
    {
        state.flags.Reset<GestureMachineFlag::IsDoubleTap>();

        state.doubleTapTimer = ::nn::TimeSpan();
    }
    else
    {
        state.doubleTapTimer =
            ::nn::TimeSpan::FromMilliSeconds(DoubleTapTimeout);

        state.doubleTap = state.touch;
    }

    MoveToComplete0(&state);
}

//!< Pan0 状態を処理します。
void ProcessPan0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    if (HasSameFingerIds(state.touch, touches, count))
    {
        ProcessAction(&state, touches, count);
    }
    else
    {
        if (state.touch.count <= count)
        {
            MoveToCancel0(&state);
        }
        else
        {
            if (count == 0)
            {
                if (IsOverVelocityLimit(state.velocity, SwipeSpeed))
                {
                    MoveToSwipe0(&state);
                }
                else
                {
                    MoveToComplete0(&state);
                }
            }
            else
            {
                MoveToPan1(&state);
            }
        }
    }
}

//!< Pan1 状態を処理します。
void ProcessPan1(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);

    GestureMachineState& state = *pState;

    if (count == 0)
    {
        if (IsOverVelocityLimit(state.velocity, SwipeSpeed))
        {
            MoveToSwipe0(&state);
        }
        else
        {
            MoveToComplete0(&state);
        }
    }
    else
    {
        if (DetectsTouchTimeout(state))
        {
            MoveToCancel0(&state);
        }
    }
}

//!< Pan2 状態を処理します。
void ProcessPan2(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);
    NN_UNUSED(count);

    GestureMachineState& state = *pState;

    MoveToSwipe0(&state);
}

//!< Swipe0 状態を処理します。
void ProcessSwipe0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);
    NN_UNUSED(count);

    GestureMachineState& state = *pState;
    state.direction = static_cast<int32_t>(Direction::None);

    MoveToComplete0(&state);
}

//!< Pinch0 状態を処理します。
void ProcessPinch0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    if (HasSameFingerIds(state.touch, touches, count))
    {
        ProcessAction(&state, touches, count);
    }
    else
    {
        if (state.touch.count <= count)
        {
            MoveToCancel0(&state);
        }
        else
        {
            if (count == 0)
            {
                if (IsOverVelocityLimit(state.velocity, SwipeSpeed))
                {
                    MoveToPan2(&state);
                }
                else
                {
                    MoveToComplete0(&state);
                }
            }
            else
            {
                MoveToPinch1(&state);
            }
        }
    }
}

//!< Pinch1 状態を処理します。
void ProcessPinch1(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);

    GestureMachineState& state = *pState;

    if (count == 0)
    {
        if (IsOverVelocityLimit(state.velocity, SwipeSpeed))
        {
            MoveToPan2(&state);
        }
        else
        {
            MoveToComplete0(&state);
        }
    }
    else
    {
        if (DetectsTouchTimeout(state))
        {
            MoveToCancel0(&state);
        }
    }
}

//!< Rotate0 状態を処理します。
void ProcessRotate0(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    if (HasSameFingerIds(state.touch, touches, count))
    {
        ProcessAction(&state, touches, count);
    }
    else
    {
        if (state.touch.count <= count)
        {
            MoveToCancel0(&state);
        }
        else
        {
            if (count == 0)
            {
                if (IsOverVelocityLimit(state.velocity, SwipeSpeed))
                {
                    MoveToPan2(&state);
                }
                else
                {
                    MoveToComplete0(&state);
                }
            }
            else
            {
                MoveToRotate1(&state);
            }
        }
    }
}

//!< Rotate1 状態を処理します。
void ProcessRotate1(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);
    NN_UNUSED(touches);

    GestureMachineState& state = *pState;

    if (count == 0)
    {
        if (IsOverVelocityLimit(state.velocity, SwipeSpeed))
        {
            MoveToPan2(&state);
        }
        else
        {
            MoveToComplete0(&state);
        }
    }
    else
    {
        if (DetectsTouchTimeout(state))
        {
            MoveToCancel0(&state);
        }
    }
}

//!< ジェスチャを構成する点を更新します。
void UpdateGesturePoints(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;

    gestureState.pointCount = machineState.touch.count;

    for (int i = 0; i < machineState.touch.count; ++i)
    {
        gestureState.points[i].x = machineState.touch.touches[i].x;
        gestureState.points[i].y = machineState.touch.touches[i].y;
    }
}

//!< ジェスチャの状態を待機中として更新します。
void UpdateGestureStateAsIdle(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);
    NN_UNUSED(machineState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Idle);
}

//!< ジェスチャの状態を完了として更新します。
void UpdateGestureStateAsComplete(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);
    NN_UNUSED(machineState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Complete);
}

//!< ジェスチャの状態をキャンセルとして更新します。
void UpdateGestureStateAsCancel(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);
    NN_UNUSED(machineState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Cancel);
}

//!< ジェスチャの状態をタッチ操作として更新します。
void UpdateGestureStateAsTouch(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Touch);
    gestureState.x = machineState.x;
    gestureState.y = machineState.y;
    gestureState.attributes.Set<GestureAttribute::IsNewTouch>(
        machineState.flags.Test<GestureMachineFlag::IsNewTouch>());
    UpdateGesturePoints(&gestureState, machineState);
}

//!< ジェスチャの状態をプレス操作として更新します。
void UpdateGestureStateAsPress(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Press);
    gestureState.x = machineState.x;
    gestureState.y = machineState.y;
    UpdateGesturePoints(&gestureState, machineState);
}

//!< ジェスチャの状態をタップ操作として更新します。
void UpdateGestureStateAsTap(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Tap);
    gestureState.x = machineState.x;
    gestureState.y = machineState.y;
    gestureState.attributes.Set<GestureAttribute::IsDoubleTap>(
        machineState.flags.Test<GestureMachineFlag::IsDoubleTap>());
    UpdateGesturePoints(&gestureState, machineState);
}

//!< ジェスチャの状態をパン操作として更新します。
void UpdateGestureStateAsPan(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Pan);
    gestureState.x = machineState.x;
    gestureState.y = machineState.y;
    gestureState.deltaX = machineState.deltaX;
    gestureState.deltaY = machineState.deltaY;
    gestureState.velocity = machineState.velocity;
    UpdateGesturePoints(&gestureState, machineState);
}

//!< ジェスチャの状態をスワイプ操作として更新します。
void UpdateGestureStateAsSwipe(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Swipe);
    gestureState._direction = static_cast<int32_t>(machineState.direction);
    gestureState.x = machineState.x;
    gestureState.y = machineState.y;
    gestureState.deltaX = machineState.deltaX;
    gestureState.deltaY = machineState.deltaY;
    gestureState.velocity = machineState.velocity;
    UpdateGesturePoints(&gestureState, machineState);
}

//!< ジェスチャの状態をピンチ操作として更新します。
void UpdateGestureStateAsPinch(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Pinch);
    gestureState.x = machineState.x;
    gestureState.y = machineState.y;
    gestureState.deltaX = machineState.deltaX;
    gestureState.deltaY = machineState.deltaY;
    gestureState.velocity = machineState.velocity;
    gestureState.scale = machineState.scale;
    UpdateGesturePoints(&gestureState, machineState);
}

//!< ジェスチャの状態を回転操作として更新します。
void UpdateGestureStateAsRotate(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;
    gestureState._type = static_cast<int32_t>(GestureType_Rotate);
    gestureState.x = machineState.x;
    gestureState.y = machineState.y;
    gestureState.deltaX = machineState.deltaX;
    gestureState.deltaY = machineState.deltaY;
    gestureState.velocity = machineState.velocity;
    gestureState.rotationAngle = machineState.rotationAngle;
    UpdateGesturePoints(&gestureState, machineState);
}

void UpdateMachineState(
    GestureMachineState* pState,
    const TouchState touches[], int32_t count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    NN_SDK_REQUIRES_NOT_NULL(touches);
    NN_SDK_REQUIRES_MINMAX(count, 0, TouchStateCountMax);

    GestureMachineState& state = *pState;

    auto timeStamp = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick());
    auto deltaTime = timeStamp - ::nn::TimeSpan(state.timeStamp);
    UpdateTimer(&state.touchTimer, deltaTime);
    UpdateTimer(&state.doubleTapTimer, deltaTime);
    state.timeStamp = timeStamp;
    state.deltaTime = deltaTime;

    switch (static_cast<Position>(state.position))
    {
    case Position::Idle0:
        ProcessIdle0(&state, touches, count);
        break;
    case Position::Idle1:
        ProcessIdle1(&state, touches, count);
        break;
    case Position::Complete0:
        ProcessComplete0(&state, touches, count);
        break;
    case Position::Cancel0:
        ProcessCancel0(&state, touches, count);
        break;
    case Position::Touch0:
        ProcessTouch0(&state, touches, count);
        break;
    case Position::Touch1:
        ProcessTouch1(&state, touches, count);
        break;
    case Position::Press0:
        ProcessPress0(&state, touches, count);
        break;
    case Position::Tap0:
        ProcessTap0(&state, touches, count);
        break;
    case Position::Pan0:
        ProcessPan0(&state, touches, count);
        break;
    case Position::Pan1:
        ProcessPan1(&state, touches, count);
        break;
    case Position::Pan2:
        ProcessPan2(&state, touches, count);
        break;
    case Position::Swipe0:
        ProcessSwipe0(&state, touches, count);
        break;
    case Position::Pinch0:
        ProcessPinch0(&state, touches, count);
        break;
    case Position::Pinch1:
        ProcessPinch1(&state, touches, count);
        break;
    case Position::Rotate0:
        ProcessRotate0(&state, touches, count);
        break;
    case Position::Rotate1:
        ProcessRotate1(&state, touches, count);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void UpdateGestureState(
    GestureState* pGestureState,
    const GestureMachineState& machineState) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGestureState);

    GestureState& gestureState = *pGestureState;

    if (gestureState.eventNumber == machineState.eventNumber)
    {
        return;
    }

    gestureState = GestureState();
    gestureState.eventNumber = machineState.eventNumber;
    gestureState.contextNumber = machineState.contextNumber;

    switch (static_cast<Position>(machineState.position))
    {
    case Position::Idle0:
    case Position::Idle1:
        UpdateGestureStateAsIdle(&gestureState, machineState);
        break;
    case Position::Complete0:
        UpdateGestureStateAsComplete(&gestureState, machineState);
        break;
    case Position::Cancel0:
        UpdateGestureStateAsCancel(&gestureState, machineState);
        break;
    case Position::Touch0:
    case Position::Touch1:
        UpdateGestureStateAsTouch(&gestureState, machineState);
        break;
    case Position::Press0:
        UpdateGestureStateAsPress(&gestureState, machineState);
        break;
    case Position::Tap0:
        UpdateGestureStateAsTap(&gestureState, machineState);
        break;
    case Position::Pan0:
    case Position::Pan1:
    case Position::Pan2:
        UpdateGestureStateAsPan(&gestureState, machineState);
        break;
    case Position::Swipe0:
        UpdateGestureStateAsSwipe(&gestureState, machineState);
        break;
    case Position::Pinch0:
    case Position::Pinch1:
        UpdateGestureStateAsPinch(&gestureState, machineState);
        break;
    case Position::Rotate0:
    case Position::Rotate1:
        UpdateGestureStateAsRotate(&gestureState, machineState);
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

} // namespace

}}} // namespace nn::hid::detail
