﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/settings/settings_DebugPad.h>

namespace nn { namespace hid { namespace detail {

//!< DebugPad 用のマッピングの解決を担うマネージャを扱うクラスです。
class DebugPadMappingManager final
{
    NN_DISALLOW_COPY(DebugPadMappingManager);
    NN_DISALLOW_MOVE(DebugPadMappingManager);

private:
    //!< DebugPad 用の割り当てを表す構造体です。
    struct DebugPadMap final
    {
        //!< 汎用コントローラ間の割り当てが初期化済みか否かを表す値
        ::std::atomic<bool> isGenericPadMapInitialized;

        //!< 汎用コントローラ間の割り当て
        ::std::atomic<::nn::settings::DebugPadGenericPadMap> genericPadMap;

        //!< キーボード間の割り当て
        ::std::atomic<::nn::settings::DebugPadKeyboardMap> keyboardMap;
    };

    //!< マッピングの初期化処理排他用のミューテックス
    ::nn::os::Mutex m_Mutex;

    //!< DebugPad 用の割り当て
    DebugPadMap m_DebugPadMap;

public:
    DebugPadMappingManager() NN_NOEXCEPT;

    //!< DebugPad-汎用コントローラ間の割り当てが有効か否かを表す値を返します。
    bool IsValidDebugPadGenericPadMap(
        const ::nn::settings::DebugPadGenericPadMap& value) NN_NOEXCEPT;

    //!< DebugPad-キーボード間の割り当てが有効か否かを表す値を返します。
    bool IsValidDebugPadKeyboardMap(
        const ::nn::settings::DebugPadKeyboardMap& value) NN_NOEXCEPT;

    ::nn::Result GetDebugPadGenericPadMap(
        ::nn::settings::DebugPadGenericPadMap* pOutValue) NN_NOEXCEPT;

    ::nn::Result SetDebugPadGenericPadMap(
        const ::nn::settings::DebugPadGenericPadMap& value) NN_NOEXCEPT;

    void ResetDebugPadGenericPadMap() NN_NOEXCEPT;

    ::nn::Result GetDebugPadKeyboardMap(
        ::nn::settings::DebugPadKeyboardMap* pOutValue) NN_NOEXCEPT;

    void SetDebugPadKeyboardMap(
        const ::nn::settings::DebugPadKeyboardMap& value) NN_NOEXCEPT;

    void ResetDebugPadKeyboardMap() NN_NOEXCEPT;

private:
    //!< DebugPad-汎用コントローラ間の割り当ての規定値を取得します。
    static void GetDefaultDebugPadGenericPadMap(
        ::nn::settings::DebugPadGenericPadMap* pOutValue) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
