﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TimerEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_TypedStorage.h>

#include "hid_AppletResourceManager.h"
#include "hid_AudioControlLifo.h"
#include "hid_AudioControlManager.h"
#include "hid_SharedMemoryFormat.h"

namespace nn { namespace hid { namespace detail {

const ::nn::TimeSpan AudioControlManager::SamplingInterval =
    ::nn::TimeSpan::FromMilliSeconds(8);

AudioControlManager::AudioControlManager() NN_NOEXCEPT
    : m_pTimer(nullptr)
    , m_pDriver(nullptr)
    , m_State()
    , m_RecentState()
    , m_Lifo()
{
     nn::os::CreateSystemEvent(&m_AudioControlEvent, nn::os::EventClearMode_ManualClear, true);
}

void AudioControlManager::SetTimer(SynchronizedTimer* timer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(timer);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pTimer = timer;
}

void AudioControlManager::SetDriver(IAudioControlDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pDriver = pDriver;
}

::nn::Result AudioControlManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultAudioControlActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        // ドライバをアクティブ化
        NN_RESULT_DO(m_pDriver->Activate());

        // タイマーイベントをアクティブ化
        m_pTimer->Enable(SamplingInterval);

        // LIFO を初期化
        m_Lifo.Clear();

        m_State = system::AudioControlState();

        // 入力状態を更新
        this->Update();
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result AudioControlManager::Deactivate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pTimer);
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultAudioControlDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施
        nn::os::DestroySystemEvent(&m_AudioControlEvent);

        // タイマーイベントを非アクティブ化
        m_pTimer->Disable();

        // ドライバを非アクティブ化
        NN_RESULT_DO(m_pDriver->Deactivate());

    }

    NN_RESULT_SUCCESS;
}

void AudioControlManager::Sample() NN_NOEXCEPT
{
    if (!m_ActivationCount.IsZero())
    {
        this->Update();
    }
}

void AudioControlManager::Update() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pDriver);

    memcpy(&m_RecentState, &m_State, sizeof(m_RecentState));

    // 入力状態を取得
    m_pDriver->GetState(&m_State);

    // 前回取得時とサンプリング番号が異なる場合は状態変化があったと見做す
    if(m_RecentState.samplingNumber != m_State.samplingNumber)
    {
        m_Lifo.Append(m_State);
        SignalAudioControlEvent();
    }
}

void AudioControlManager::SignalAudioControlEvent() NN_NOEXCEPT
{
    nn::os::SignalSystemEvent(&m_AudioControlEvent);
}

::nn::Result AudioControlManager::AcquireAudioControlEventHandle(
    ::nn::os::NativeHandle* pOutHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);
    *pOutHandle = nn::os::GetReadableHandleOfSystemEvent(&m_AudioControlEvent);
    NN_RESULT_SUCCESS;
}

::nn::Result AudioControlManager::GetStates(int* pOutCount,
                       system::AudioControlState outStates[],
                       int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultAudioControlNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultAudioControlInvalidStateCount());
    if(m_Lifo.IsEmpty())
    {
        *pOutCount = 0;
        NN_RESULT_SUCCESS;
    }
    *pOutCount = m_Lifo.Read(outStates, count);
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail

