﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_IHidSystemServer.sfdl.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/system/hid_AudioControl.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/sf_NativeHandle.h>

#include "hid_HidSystemServer.h"
#include "hid_AudioControlImpl.h"
#include "hid_AudioControlLifo.h"

namespace nn { namespace hid { namespace detail {

::nn::Result InitializeAudioControl() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
    NN_RESULT_DO(pHidSystemServer->ActivateAudioControl());
    NN_RESULT_SUCCESS;
}

::nn::Result BindAudioControlStateChangeEvent(nn::os::SystemEventType* pEvent,
                                       nn::os::EventClearMode clearMode) NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));

    nn::sf::NativeHandle handle;
    NN_RESULT_DO(pHidSystemServer->AcquireAudioControlEventHandle(&handle));

    nn::os::AttachReadableHandleToSystemEvent(pEvent,
                                              handle.GetOsHandle(),
                                              handle.IsManaged(),
                                              clearMode);

    // ハンドルの管理権を放棄
    handle.Detach();

    NN_RESULT_SUCCESS;
}

::nn::Result FinalizeAudioControl() NN_NOEXCEPT
{
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
    NN_RESULT_DO(pHidSystemServer->DeactivateAudioControl());
    NN_RESULT_SUCCESS;
}

::nn::Result GetAudioControlStates( int* pOutCount,
                                    system::AudioControlState outStates[],
                                    int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_RESULT_THROW_UNLESS(outStates != nullptr,
                           ResultAudioControlNullStateBuffer());
    NN_RESULT_THROW_UNLESS(count >= 0, ResultAudioControlInvalidStateCount());
    ::nn::sf::SharedPointer<IHidSystemServer> pHidSystemServer;
    NN_RESULT_DO(CreateHidSystemServerProxy(&pHidSystemServer));
    auto value = int64_t();
    NN_RESULT_DO(
        pHidSystemServer->GetAudioControlStates(&value,
                               ::nn::sf::OutArray<system::AudioControlState>(
                                   outStates, static_cast<size_t>(count))));
    *pOutCount = static_cast<int>(value);
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
