﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <limits>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/util/util_BitFlagSet.h>

#include "hid_ActivationCount.h"
#include "hid_ConsoleSixAxisSensorAppletSetting.h"
#include "hid_HashTable.h"
#include "hid_ISixAxisSensorFilter.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SharedMemoryHolder.h"
#include "hid_SixAxisSensorAppletSetting.h"

namespace nn { namespace hid { namespace detail {

//!< アプレットリソースのエントリの最大数
const size_t AppletResourceEntryCountMax = 32;

//!< アプレットリソースの制御フラグ定義です。
struct AppletResourceFlag final
{
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<0>
        IsRegistered;                  //!< 登録済みか否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<1>
        IsAvailable;                   //!< 利用可能か否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<16>
        EnablesInput;                  //!< HID 入力の提供を行うか否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<17>
        EnablesSixAxisSensor;          //!< 6 軸センサーの提供を行うか否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<18>
        EnablesConsoleSixAxisSensor;   //!< 本体 6 軸センサーの提供を行うか否か
    typedef ::nn::util::BitFlagSet<32, AppletResourceFlag>::Flag<19>
        EnablesPalmaScanAll;           //!< Palma のペアリング用のスキャンを行うか否か
};

//!< アプレットリソースの制御フラグ集合を扱う型です。
typedef ::nn::util::BitFlagSet<32, AppletResourceFlag> AppletResourceFlagSet;

//!< アプレットリソースのエントリを表す構造体です。
struct AppletResourceEntry final
{
    AppletResourceFlagSet flags;                                    //!< 制御フラグ
    ::nn::applet::AppletResourceUserId aruid;                       //!< ARUID
    SharedMemoryFormat* address;                                    //!< 共有メモリの開始アドレス
    SixAxisSensorAppletSetting sixAxisSensorSetting;                //!< SixAxisSensorの設定値
    ConsoleSixAxisSensorAppletSetting consoleSixAxisSensorSetting;  //!< SixAxisSensorの設定値
};

//!< ARUID のハッシュ値計算を扱うクラスです。
class AppletResourceUserIdHasher final
{
public:
    //!< ARUID のハッシュ値を返します。
    static size_t Get(
        const ::nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
    {
        return static_cast<size_t>(
            aruid.lower & ::std::numeric_limits<size_t>::max());
    }
};

//!< アプレットリソースの解決を担うマネージャを扱うクラスです。
class AppletResourceManager final
{
    NN_DISALLOW_COPY(AppletResourceManager);
    NN_DISALLOW_MOVE(AppletResourceManager);

private:
    //!< ARUID
    ::nn::applet::AppletResourceUserId m_AppletResourceUserId;

    //!< アプレットリソースのエントリマップ
    HashTable<
        ::nn::applet::AppletResourceUserId, AppletResourceEntry*,
        AppletResourceUserIdHasher, AppletResourceEntryCountMax> m_EntryMap;

    //!< アプレットリソースのエントリ
    AppletResourceEntry m_Entries[AppletResourceEntryCountMax];

    //!< 共有メモリホルダ
    SharedMemoryHolder m_SharedMemoryHolders[AppletResourceEntryCountMax];

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< 振動子の制御権を持つARUID
    ::nn::applet::AppletResourceUserId m_AruidValidForVibration;

    //!< 本体 6 軸センサーのフィルタが利用しているフィルタ
    ISixAxisSensorFilter** m_pSixAxisSensorFilter;

public:
    AppletResourceManager() NN_NOEXCEPT;

    //!< ARUID を返します。
    ::nn::applet::AppletResourceUserId GetAppletResourceUserId() NN_NOEXCEPT
    {
        return m_AppletResourceUserId;
    }

    //!< 指定した個数分の ARUID を返します。
    void GetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId outValues[], int count
        ) const NN_NOEXCEPT;

    //!< ARUID を設定します。
    void SetAppletResourceUserId(
        ::nn::applet::AppletResourceUserId value) NN_NOEXCEPT
    {
        m_AppletResourceUserId = value;
    }

    //!< アプレットリソースのエントリを返します。
    const AppletResourceEntry (&GetAppletResourceEntries() const NN_NOEXCEPT)[
        AppletResourceEntryCountMax]
    {
        return m_Entries;
    }

    //!< アプレットリソースユーザ ID を登録します。
    ::nn::Result RegisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースユーザ ID の登録を解除します。
    void UnregisterAppletResourceUserId(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリを設定します。
    ::nn::Result SetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースのエントリの設定を解除します。
    void ResetAppletResourceEntry(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースの HID 入力が有効か否かを設定します。
    void EnableAppletResourceInput(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースの 6 軸センサーが有効か否かを設定します。
    void EnableAppletResourceSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesInput) NN_NOEXCEPT;

    //!< アプレットリソースの共有メモリのハンドルを取得します。
    ::nn::Result GetAppletResourceSharedMemoryHandle(
        ::nn::os::NativeHandle* pOutValue,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 振動の制御権を持つアプレットリソースユーザ ID を設定します。
    bool SetAruidValidForVibration(
        ::nn::applet::AppletResourceUserId aruid, bool enable) NN_NOEXCEPT;

    //!< 振動の制御権を持つアプレットリソースユーザ ID であるかを判定します。
    void IsAruidValidForVibration(
        ::nn::applet::AppletResourceUserId aruid, bool* pOutValue) NN_NOEXCEPT;

    //!< 指定した Aruid に合致する SixAxisSensorAppletSetting を取得します
    SixAxisSensorAppletSetting* GetSixAxisSensorAppletSetting(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 指定した Aruid に合致する ConsoleSixAxisSensorAppletSetting を取得します
    ConsoleSixAxisSensorAppletSetting* GetConsoleSixAxisSensorAppletSetting(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 共有アプレットリソースをアクティブ化します。
    ::nn::Result ActivateSharedAppletResource() NN_NOEXCEPT;

    //!< 共有アプレットリソースを非アクティブ化します。
    ::nn::Result DeactivateSharedAppletResource() NN_NOEXCEPT;

    //!< ワークバッファの TransferMemoryType を取得します。
    ::nn::Result GetConsoleSixAxisSensorWorkBufferTransferMemoryType(
        ::nn::os::TransferMemoryType** pTransferMemory,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< 入力状態格納用バッファの TransferMemoryType を取得します。
    ::nn::Result GetConsoleSixAxisSensorStateBufferTransferMemoryType(
        ::nn::os::TransferMemoryType** pTransferMemory,
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< ワークバッファの ConsoleSixAxisSensorTansferMemoryStatus を取得します。
    ::nn::Result GetConsoleSixAxisSensorTransferMemoryWorkBuffer(
        ::nn::applet::AppletResourceUserId aruid,
        ConsoleSixAxisSensorTansferMemoryStatus** pBuffer) NN_NOEXCEPT;

    //!< 入力状態格納用バッファの ConsoleSixAxisSensorTansferMemoryStatus を取得します。
    ::nn::Result GetConsoleSixAxisSensorTransferMemoryStateBuffer(
        ::nn::applet::AppletResourceUserId aruid,
        ConsoleSixAxisSensorTansferMemoryStatus** pBuffer) NN_NOEXCEPT;

    //!< TransferMemory のワークバッファを設定します
    ::nn::Result SetConsoleSixAxisSensorTransferMemoryWorkBuffer(
        ::nn::applet::AppletResourceUserId aruid,
        ConsoleSixAxisSensorTansferMemoryStatus&& workBuffer) NN_NOEXCEPT;

    //!< TransferMemory の入力状態格納用バッファを設定します
    ::nn::Result SetConsoleSixAxisSensorTransferMemoryStateBuffer(
        ::nn::applet::AppletResourceUserId aruid,
        ConsoleSixAxisSensorTansferMemoryStatus&& stateBuffer) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を初期化します。
    ::nn::Result InitializeSevenSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid,
        ISixAxisSensorFilter** pFilter) NN_NOEXCEPT;

    //!< Seven 向け SixAxisSensor を終了します。
    ::nn::Result FinalizeSevenSixAxisSensor(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT;

    //!< アプレットリソースの Palma のペアリングのスキャンが有効か否かを設定します。
    void EnableAppletResourcePalmaScanAll(
        ::nn::applet::AppletResourceUserId aruid,
        bool enablesScan) NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
