﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/applet/applet_FundamentalTypes.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_MemoryFence.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "hid_AppletResourceManager.h"
#include "hid_SharedMemoryFormat.h"
#include "hid_SharedMemoryHolder.h"

namespace nn { namespace hid { namespace detail {

AppletResourceManager::AppletResourceManager() NN_NOEXCEPT
    : m_AppletResourceUserId(
        ::nn::applet::AppletResourceUserId::GetInvalidId())
    , m_SharedMemoryHolders()
    , m_ActivationCount()
    , m_AruidValidForVibration(
        ::nn::applet::AppletResourceUserId::GetInvalidId())
    , m_pSixAxisSensorFilter(nullptr)
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        entry = AppletResourceEntry();
        entry.sixAxisSensorSetting.Reset(true);
        entry.consoleSixAxisSensorSetting.ResetConsoleSixAxisSensorSetting(true);
    }
}

void AppletResourceManager::GetAppletResourceUserId(
    ::nn::applet::AppletResourceUserId outValues[], int count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValues);
    NN_SDK_REQUIRES_GREATER(count, 0);

    const auto EntryCount = ::std::min(
        static_cast<size_t>(count), AppletResourceEntryCountMax);

    for (size_t i = 0; i < EntryCount; i++)
    {
        outValues[i] = m_Entries[i].aruid;
    }
}

::nn::Result AppletResourceManager::RegisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesInput) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_EntryMap.Get(aruid) == nullptr,
        ::nn::hid::ResultAppletResourceDuplicateUserId());

    for (AppletResourceEntry& entry : m_Entries)
    {
        if (!entry.flags.Test<AppletResourceFlag::IsRegistered>())
        {
            entry.flags.Set<AppletResourceFlag::IsRegistered>();
            entry.flags.Set<AppletResourceFlag::EnablesInput>(enablesInput);
            entry.flags.Set<AppletResourceFlag::EnablesSixAxisSensor>(enablesInput);
            entry.flags.Set<AppletResourceFlag::EnablesConsoleSixAxisSensor>(enablesInput);
            entry.aruid = aruid;

            m_EntryMap.Set(aruid, &entry);

            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(ResultAppletResourceTableOverflow());
}

void AppletResourceManager::UnregisterAppletResourceUserId(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    this->ResetAppletResourceEntry(aruid);

    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        AppletResourceEntry& entry = **ppEntry;

        FinalizeSevenSixAxisSensor(aruid);
        entry.flags.Reset();
        entry.aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();
        entry.sixAxisSensorSetting.Reset(false);
        entry.consoleSixAxisSensorSetting.ResetConsoleSixAxisSensorSetting(false);

        m_EntryMap.Remove(aruid);
    }
}

::nn::Result AppletResourceManager::SetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    NN_RESULT_THROW_UNLESS(ppEntry != nullptr, ResultAppletResourceNotFound());

    NN_SDK_ASSERT_NOT_NULL(*ppEntry);

    AppletResourceEntry& entry = **ppEntry;

    NN_RESULT_THROW_UNLESS(
        !entry.flags.Test<AppletResourceFlag::IsAvailable>(),
        ::nn::hid::ResultAppletResourceDuplicateUserId());

    SharedMemoryHolder&
        holder = m_SharedMemoryHolders[&entry - &m_Entries[0]];

    if (!holder.IsMapped())
    {
        auto needsRollback = true;

        NN_RESULT_DO(holder.Initialize());

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                holder.Finalize();
            }
        };

        NN_RESULT_THROW_UNLESS(
            holder.GetAddress() != nullptr,
            ResultAppletResourceSharedMemoryAllocationFailed());

        needsRollback = false;
    }

    SharedMemoryFormat* const address = holder.GetAddress();

    if (address != nullptr)
    {
        // 共有メモリのフォーマットを初期化
        InitializeSharedMemoryFormat(address);

        // 初期化の完了を保証
        ::nn::os::FenceMemoryStoreLoad();
    }

    entry.flags.Set<AppletResourceFlag::IsAvailable>();
    entry.address = address;
    entry.sixAxisSensorSetting.Reset(false);

    NN_RESULT_SUCCESS;
}

void AppletResourceManager::ResetAppletResourceEntry(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        AppletResourceEntry& entry = **ppEntry;

        if (entry.flags.Test<AppletResourceFlag::IsAvailable>())
        {
            SharedMemoryFormat* const address = entry.address;

            entry.flags.Reset<AppletResourceFlag::IsAvailable>();
            entry.address = nullptr;

            if (address != nullptr)
            {
                // 共有メモリに対するアクセスの完了を保証
                ::nn::os::FenceMemoryAnyAny();

                // 共有メモリのフォーマットを破棄
                FinalizeSharedMemoryFormat(address);
            }
        }
    }
}

void AppletResourceManager::EnableAppletResourceInput(
    ::nn::applet::AppletResourceUserId aruid, bool enablesInput) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        AppletResourceEntry& entry = **ppEntry;

        entry.flags.Set<AppletResourceFlag::EnablesInput>(enablesInput);
    }
}

void AppletResourceManager::EnableAppletResourceSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid, bool enablesInput) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        AppletResourceEntry& entry = **ppEntry;

        entry.flags.Set<AppletResourceFlag::EnablesSixAxisSensor>(enablesInput);
    }
}

::nn::Result AppletResourceManager::GetAppletResourceSharedMemoryHandle(
    ::nn::os::NativeHandle* pOutValue,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    NN_RESULT_THROW_UNLESS(ppEntry != nullptr, ResultAppletResourceNotFound());

    NN_SDK_ASSERT_NOT_NULL(*ppEntry);

    AppletResourceEntry& entry = **ppEntry;

    const SharedMemoryHolder&
        holder = m_SharedMemoryHolders[&entry - &m_Entries[0]];

    *pOutValue = holder.GetSharedMemoryHandle();

    NN_RESULT_SUCCESS;
}

bool AppletResourceManager::SetAruidValidForVibration(
    ::nn::applet::AppletResourceUserId aruid, bool enable) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        if (enable)
        {
            //振動制御権を持ったARUIDを指定されたARUIDが異なる場合は
            //そのARUIDに制御権を渡す
            if (m_AruidValidForVibration != aruid)
            {
                m_AruidValidForVibration = aruid;
                return true;
            }
        }
        else
        {
            //振動制御権を持ったARUIDと指定されたARUIDが一致している場合は、
            //InvalidIdに制御権を渡す
            if (m_AruidValidForVibration == aruid)
            {
                m_AruidValidForVibration =
                    ::nn::applet::AppletResourceUserId::GetInvalidId();
                return true;
            }
        }
    }
    return false;
}

void AppletResourceManager::IsAruidValidForVibration(
    ::nn::applet::AppletResourceUserId aruid, bool* pOutValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);

    if (aruid == ::nn::applet::AppletResourceUserId::GetInvalidId() ||
        aruid == m_AruidValidForVibration)
    {
        *pOutValue = true;
    }
    else
    {
        *pOutValue = false;
    }
}

SixAxisSensorAppletSetting*
    AppletResourceManager::GetSixAxisSensorAppletSetting(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        AppletResourceEntry& entry = **ppEntry;

        return &(entry.sixAxisSensorSetting);
    }

    return nullptr;
}

ConsoleSixAxisSensorAppletSetting*
    AppletResourceManager::GetConsoleSixAxisSensorAppletSetting(
        ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        AppletResourceEntry& entry = **ppEntry;

        return &(entry.consoleSixAxisSensorSetting);
    }

    return nullptr;
}

::nn::Result AppletResourceManager::ActivateSharedAppletResource() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsMax(),
        ResultAppletResourceActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 新規に要求された場合のみアクティブ化を実施

        auto needsRollback = true;

        const auto aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();

        NN_RESULT_DO(this->RegisterAppletResourceUserId(aruid, true));

        NN_UTIL_SCOPE_EXIT
        {
            if (needsRollback)
            {
                this->UnregisterAppletResourceUserId(aruid);
            }
        };

        NN_RESULT_DO(this->SetAppletResourceEntry(aruid));

        needsRollback = false;
    }

    ++m_ActivationCount;

    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::DeactivateSharedAppletResource() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        !m_ActivationCount.IsZero(),
        ResultAppletResourceDeactivationLowerLimitOver());

    --m_ActivationCount;

    if (m_ActivationCount.IsZero())
    {
        // 全ての場所からアクティブ化を解除された時点で非アクティブ化を実施

        const auto aruid = ::nn::applet::AppletResourceUserId::GetInvalidId();

        this->UnregisterAppletResourceUserId(aruid);
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::GetConsoleSixAxisSensorWorkBufferTransferMemoryType(
    ::nn::os::TransferMemoryType** pTransferMemory,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            *pTransferMemory = &entry.consoleSixAxisSensorSetting.workBuffer.transferMemoryType;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::GetConsoleSixAxisSensorStateBufferTransferMemoryType(
    ::nn::os::TransferMemoryType** pTransferMemory,
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            *pTransferMemory = &entry.consoleSixAxisSensorSetting.stateBuffer.transferMemoryType;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::GetConsoleSixAxisSensorTransferMemoryWorkBuffer(
    ::nn::applet::AppletResourceUserId aruid,
    ConsoleSixAxisSensorTansferMemoryStatus** pBuffer) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            *pBuffer = &entry.consoleSixAxisSensorSetting.workBuffer;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::GetConsoleSixAxisSensorTransferMemoryStateBuffer(
    ::nn::applet::AppletResourceUserId aruid,
    ConsoleSixAxisSensorTansferMemoryStatus** pBuffer) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            *pBuffer = &entry.consoleSixAxisSensorSetting.stateBuffer;
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::SetConsoleSixAxisSensorTransferMemoryStateBuffer(
    ::nn::applet::AppletResourceUserId aruid,
    ConsoleSixAxisSensorTansferMemoryStatus&& stateBuffer) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            entry.consoleSixAxisSensorSetting.stateBuffer = std::move(stateBuffer);
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::SetConsoleSixAxisSensorTransferMemoryWorkBuffer(
    ::nn::applet::AppletResourceUserId aruid,
    ConsoleSixAxisSensorTansferMemoryStatus&& workBuffer) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            entry.consoleSixAxisSensorSetting.workBuffer = std::move(workBuffer);
        }
    }
    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::InitializeSevenSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid,
    ISixAxisSensorFilter** pFilter) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto& setting = entry.consoleSixAxisSensorSetting;

            // 既に初期化済の場合はエラー
            NN_RESULT_THROW_UNLESS(
                (setting.stateBuffer.IsInitialized() == false) &&
                (setting.workBuffer.IsInitialized() == false),
                ResultSevenSixAxisSensorNullStateBuffer());
        }
    }

    m_pSixAxisSensorFilter = pFilter;

    NN_RESULT_SUCCESS;
}

::nn::Result AppletResourceManager::FinalizeSevenSixAxisSensor(
    ::nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    for (AppletResourceEntry& entry : m_Entries)
    {
        if (entry.flags.Test<AppletResourceFlag::IsRegistered>() &&
            entry.aruid == aruid)
        {
            auto& setting = entry.consoleSixAxisSensorSetting;

            // フィルタにワークバッファを割り当てた aruid と Unregister する aruid が一致したときはステートバッファとワークバッファを開放する
            if (m_pSixAxisSensorFilter != nullptr &&
                (*m_pSixAxisSensorFilter)->GetAppletResourceUserId() == aruid)
            {
                // TransferMemory を破棄する前にフィルタが使っているバッファを解放
                (*m_pSixAxisSensorFilter)->ClearStateBuffer();
                (*m_pSixAxisSensorFilter)->ClearWorkBuffer();
                m_pSixAxisSensorFilter = nullptr;
            }

            if(setting.workBuffer.IsInitialized())
            {
                // Transfer メモリを破棄
                ::nn::os::UnmapTransferMemory(&setting.workBuffer.transferMemoryType);
                ::nn::os::DestroyTransferMemory(&setting.workBuffer.transferMemoryType);

                // ARUID テーブルの情報をクリアしておく
                setting.workBuffer.Reset();
            }

            if (setting.stateBuffer.IsInitialized())
            {
                // Transfer メモリを破棄
                ::nn::os::UnmapTransferMemory(&setting.stateBuffer.transferMemoryType);
                ::nn::os::DestroyTransferMemory(&setting.stateBuffer.transferMemoryType);

                // ARUID テーブルの情報をクリアしておく
                setting.stateBuffer.Reset();
            }
        }
    }
    NN_RESULT_SUCCESS;
}

void AppletResourceManager::EnableAppletResourcePalmaScanAll(
    ::nn::applet::AppletResourceUserId aruid,
    bool enablesScan) NN_NOEXCEPT
{
    AppletResourceEntry** ppEntry = m_EntryMap.Get(aruid);

    if (ppEntry != nullptr)
    {
        NN_SDK_ASSERT_NOT_NULL(*ppEntry);

        AppletResourceEntry& entry = **ppEntry;

        entry.flags.Set<AppletResourceFlag::EnablesPalmaScanAll>(enablesScan);
    }
}

}}} // namespace nn::hid::detail
