﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       加速度センサーの処理に特化した API の宣言
 */

#pragma once

#include <nn/util/util_MathTypes.h>
#include <nn/xcd/xcd_Input.h>

#include "hid_SixAxisSensorAppletSetting.h"

namespace nn { namespace hid { namespace detail {

//!< 加速度センサーの処理を担うクラス
class Accelerometer final
{
    NN_DISALLOW_COPY(Accelerometer);
    NN_DISALLOW_MOVE(Accelerometer);

private:
    nn::util::Float3            m_ZeroCount;            //!< ゼロ点カウント[count]
    nn::util::Float3            m_SensitivityCount;     //!< 感度カウント[count]
    nn::util::Float3            m_Scale;                //!< 加速度値の変換パラメータ [G/count]
    nn::util::Float3            m_LastAcceleration;     //!< 前回算出した加速度ベクトル[G]

    const AccelerometerSetting*  m_pAccelerometerSetting; //!< 設定値

public:
    Accelerometer()  NN_NOEXCEPT;
    ~Accelerometer() NN_NOEXCEPT {};

    //!< 加速度のパラメータを設定します。
    void SetAccelerometerSetting(const AccelerometerSetting* const pSetting) NN_NOEXCEPT;

    //!< 加速度を取得します。
    void GetState(nn::util::Float3* pOutState, const nn::util::Float3& accelerationCount) NN_NOEXCEPT;

    //!< 姿勢を加速度値を用いて更新します。
    float UpdateDirection( nn::hid::DirectionState* pOutDirection, const nn::util::Float3& acceleration ) const NN_NOEXCEPT;

    //!< 加速度センサーのキャリブレーション値を設定します。
    void SetCalibrationValue( const nn::xcd::SensorState& accelerometerOrigin,
                              const nn::xcd::SensorState& accelerometerSensitivity ) NN_NOEXCEPT;

private:

    //!< 加速度センサーのスケーリングパラメータを設定します。
    void SetScaleValue() NN_NOEXCEPT;

    //!< カウント値を加速度値[G] に変換します。
    void ConvertToAcceleration( nn::util::Float3* pOut, const nn::util::Float3& count ) const NN_NOEXCEPT;

    //!< 遊びの挙動方法に応じた加速度値[G]の補正を行います。
    void ReviseAccelerationByPlayMode( nn::util::Float3* pOut, const nn::util::Float3& acceleration ) const NN_NOEXCEPT;

    //!< 内部的に保持している前回サンプルの加速度カウント値を更新します。
    void UpdateLastAcceleration( const nn::util::Float3& acceleration ) NN_NOEXCEPT;

    //!<  AccelerationPlayMode に応じた加速度の補正をします。( __calcAcc 相当 )
    float CalcAcceleration(const float& currentAcc, const float& lastAcc ) const NN_NOEXCEPT;

    //!< 加速度センサーのキャリブレーション値をシフトします。
    void ShiftCalibrationValue(const float& origin,
                               const float& sensitivity) NN_NOEXCEPT;

};

}}} // namespace nn::hid::detail
