﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/xcd/xcd_Device.h>

#include "hid_AbstractedPadIdPublisher.h"
#include "hid_AbstractedPadXcd.h"
#include "hid_ActivationCount.h"
#include "hid_MultiWaitEvent.h"

namespace nn { namespace hid { namespace detail {

//!< Xcd デバイスの扱える最大数
static const int AbstractedPadXcdCountMax = nn::xcd::DeviceCountMax;

//!< Xcd のデバイス管理を担うマネージャーです
class AbstractedPadXcdManager final
{
    NN_DISALLOW_COPY(AbstractedPadXcdManager) NN_NOEXCEPT;
    NN_DISALLOW_MOVE(AbstractedPadXcdManager) NN_NOEXCEPT;

private:
    //!< AbstractedPad のリスト
    AbstractedPadXcd* m_pPads[AbstractedPadXcdCountMax];

    //!< AbstractedPad の数
    int m_PadCount;

    //!< アクティブ化された回数
    ActivationCount m_ActivationCount;

    //!< AbstractedId パブリッシャー
    AbstractedPadIdPublisher* m_pIdPublisher;

    //!< USB FullKeyController の Npad 接続を有効にするかどうか
    bool m_IsUsbFullKeyControllerEnabled;

public:
    AbstractedPadXcdManager() NN_NOEXCEPT;

    ~AbstractedPadXcdManager() NN_NOEXCEPT;

    //!< AbstractedId パブリッシャーをセットします
    void SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT;

    //!< AbstractedPadXcd をセットします
    void SetAbstractedPadXcds(AbstractedPadXcd* pPads, int count) NN_NOEXCEPT;

    //!< マネージャをアクティブ化します。
    ::nn::Result Activate() NN_NOEXCEPT;

    //!< マネージャを非アクティブ化します。
    ::nn::Result Deactivate() NN_NOEXCEPT;

    //!< Xcd のデバイスのリストの状態を更新する
    void UpdateDevices() NN_NOEXCEPT;

    //!< 定期的にデバイスの状態をチェックする
    void HandlePeriodicalEvent() NN_NOEXCEPT;

    //!< USB FullKeyController が有効かどうかを取得します。
    ::nn::Result IsUsbFullKeyControllerEnabled(bool* pOutEnabled) NN_NOEXCEPT;

    //!< USB FullKeyController の有効無効を制御します。
    ::nn::Result EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT;

    //!< 接続済みのデバイスをすべてリセットする
    void ResetAllDevices() NN_NOEXCEPT;

private:
    //!< 指定された xcd デバイスが接続済みのデバイスかどうか
    bool IsConnectedXcdDevice(::nn::xcd::DeviceHandle handle) NN_NOEXCEPT;

    //!< 切断されている xcd デバイスかどうかを確認し、切断されている場合はディアクティベートをします。
    void CheckForDeviceDetach(AbstractedPadXcd* pPad, ::nn::xcd::DeviceList* pList) NN_NOEXCEPT;

    //!< 切断を遅延しているデバイスがタイムアウトしているかどうかをチェックし、タイムアウトしている場合はディアクティベートします。
    void CheckForDelayedDeviceDetach(AbstractedPadXcd* pPad) NN_NOEXCEPT;

    //!< あいている Npad に対してデバイスを追加します。
    void AddXcdDevice(::nn::xcd::DeviceHandle handle, ::nn::xcd::DeviceInfo deviceInfo) NN_NOEXCEPT;
};

}}} // namespace nn::hid::detail
