﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include "hid_AbstractedPadXcdManager.h"

namespace nn { namespace hid { namespace detail {

namespace
{
const auto DelayedDetachTimeSpan = ::nn::TimeSpan::FromMilliSeconds(1000);    //!< コントローラーの切断を遅延するための時間
}

AbstractedPadXcdManager::AbstractedPadXcdManager() NN_NOEXCEPT
    : m_ActivationCount()
    , m_IsUsbFullKeyControllerEnabled(false)
{
}

AbstractedPadXcdManager::~AbstractedPadXcdManager() NN_NOEXCEPT
{
    // 何もしない
}

void AbstractedPadXcdManager::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pIdPublisher = pPublisher;
}

void AbstractedPadXcdManager::SetAbstractedPadXcds(AbstractedPadXcd* pPads, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(count, AbstractedPadXcdCountMax);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    for (int padIndex = 0; padIndex < count; padIndex++)
    {
        m_pPads[padIndex] = &pPads[padIndex];
    }
    m_PadCount = count;
}

::nn::Result AbstractedPadXcdManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPads);
    NN_SDK_REQUIRES_GREATER(m_PadCount, 0);
    NN_SDK_REQUIRES_NOT_NULL(m_pIdPublisher);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // Pro Controller 有線 USB 通信の設定を取得
        bool isEnabled;
        ::nn::xcd::GetFullKeyUsbEnabled(&isEnabled);
        m_IsUsbFullKeyControllerEnabled = isEnabled;
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;

}

::nn::Result AbstractedPadXcdManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

void AbstractedPadXcdManager::UpdateDevices() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        return;
    }

    ::nn::xcd::DeviceList deviceList;

    if(::nn::xcd::ListDevices(&deviceList).IsFailure())
    {
        deviceList.deviceCount = 0;
    }

    // 切断されたデバイスを探索
    for (int i = 0; i < m_PadCount; i++)
    {
        CheckForDeviceDetach(m_pPads[i], &deviceList);
    }

    // 接続済みのデバイスを処理
    for(int i = 0; i < deviceList.deviceCount; i++)
    {
        // 新規で接続されたデバイスか確認
        if (IsConnectedXcdDevice(deviceList.handleList[i]) == false)
        {
            ::nn::xcd::DeviceStatus status;
            ::nn::xcd::DeviceInfo info;
            if (::nn::xcd::GetDeviceStatus(&status, deviceList.handleList[i]).IsFailure() ||
                ::nn::xcd::GetDeviceInfo(&info, deviceList.handleList[i]).IsFailure())
            {
                // デバイス情報が正しく読み出せないので切断
                nn::xcd::Detach(deviceList.handleList[i]);
                continue;
            }

            if ( status.interfaceType == ::nn::xcd::InterfaceType_Usb && (info.deviceType != ::nn::xcd::DeviceType_FullKey || m_IsUsbFullKeyControllerEnabled == false) &&
                  info.deviceType != ::nn::xcd::DeviceType_MiyabiLeft &&
                  info.deviceType != ::nn::xcd::DeviceType_MiyabiRight
               )
            {
                // USB 通信が有効でないので切断
                nn::xcd::Detach(deviceList.handleList[i]);
                continue;
            }

            AddXcdDevice(deviceList.handleList[i], info);
        }
    }
}

void AbstractedPadXcdManager::HandlePeriodicalEvent() NN_NOEXCEPT
{
    // 切断を遅延しているデバイスがタイムアウトしているかどうかチェック
    for (int i = 0; i < m_PadCount; i++)
    {
        CheckForDelayedDeviceDetach(m_pPads[i]);
    }
}

bool AbstractedPadXcdManager::IsConnectedXcdDevice(::nn::xcd::DeviceHandle handle) NN_NOEXCEPT
{
    // NpadManager を探索して接続済みかどうかをチェック
    for (int padIndex = 0; padIndex < m_PadCount; padIndex++)
    {
        if (m_pPads[padIndex]->GetXcdDeviceHandle() == handle)
        {
            // 接続済みのデバイス
            return true;
        }
    }

    // 未接続のデバイス
    return false;
}

void AbstractedPadXcdManager::CheckForDeviceDetach(AbstractedPadXcd* pPad, ::nn::xcd::DeviceList* pList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pList);

    // AbstractedPad に対してもともとデバイスが何も割り当てされていない場合は何もしない
    if (pPad->IsAttached() == false)
    {
        return;
    }

    // まだデバイスが接続されているかどうか探索
    for (int i = 0; i < pList->deviceCount; i++)
    {
        if (pPad->GetXcdDeviceHandle() == pList->handleList[i])
        {
            // 接続中なので切断しない
            return;
        }
    }

    if (pPad->IsOnDelayedDetach())
    {
        // 遅延中はここでは何もしない
        return;
    }
    else
    {
        // USB 通信が有効で Bluetooth 接続している場合は切断を遅延する
        if (pPad->GetInterfaceType() == system::InterfaceType_Bluetooth &&
            m_IsUsbFullKeyControllerEnabled == true &&
            pPad->GetDeviceType() == system::DeviceType::FullKeyController::Mask &&
            pPad->IsConnected() == true)
        {
            // 遅延処理を開始、現時点では切断しない
            pPad->SetDelayedDetachTick(::nn::os::GetSystemTick());
            return;
        }
    }

    // ここまで到達した場合は切断
    pPad->RemoveDevice();
}

void AbstractedPadXcdManager::CheckForDelayedDeviceDetach(AbstractedPadXcd* pPad) NN_NOEXCEPT
{
    // AbstractedPad に対してもともとデバイスが何も接続されていない場合は何もしない
    if (pPad->IsConnected() == false)
    {
        return;
    }

    // 切断遅延中じゃない場合は何もしない
    if (pPad->IsOnDelayedDetach() == false)
    {
        return;
    }

    if ((::nn::os::GetSystemTick() - pPad->GetDelayedDetachTick()).ToTimeSpan() < DelayedDetachTimeSpan)
    {
        // タイムアウト前
        return;
    }

    // ここまで到達した場合は切断
    pPad->RemoveDevice();
}

void AbstractedPadXcdManager::AddXcdDevice(::nn::xcd::DeviceHandle handle, ::nn::xcd::DeviceInfo deviceInfo) NN_NOEXCEPT
{
    // USB 通信が有効な場合、Bluetooth 状態から自動的に切り替えをする
    if (m_IsUsbFullKeyControllerEnabled == true &&
        deviceInfo.deviceType == ::nn::xcd::DeviceType_FullKey)
    {
        // 接続済みの Bluetooth デバイスがないかチェック
        for (int padIndex = 0; padIndex < m_PadCount; padIndex++)
        {
            if (m_pPads[padIndex]->IsAttached() == true &&
                m_pPads[padIndex]->GetAddress() == deviceInfo.address)
            {
                // Id をバックアップ
                auto id = m_pPads[padIndex]->GetId();
                m_pPads[padIndex]->RemoveDevice();
                // バックアップした Id で再 Attach
                m_pPads[padIndex]->AttachDevice(handle, id);
                // 自動的に Connect 状態にする
                m_pPads[padIndex]->Connect();
                return;
            }
        }

    }

    for (int padIndex = 0; padIndex < m_PadCount; padIndex++)
    {
        if (m_pPads[padIndex]->IsAttached() == false)
        {
            // あいている Pad に追加 & 新規に Id 発行
            m_pPads[padIndex]->AttachDevice(handle, m_pIdPublisher->PublishId());
            return;
        }
    }

    // 割り当てできる Pad が見つからなかった場合は Xcd から Detach
    ::nn::xcd::Detach(handle);
}

::nn::Result AbstractedPadXcdManager::IsUsbFullKeyControllerEnabled(bool* pOutEnabled) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEnabled);

    *pOutEnabled = m_IsUsbFullKeyControllerEnabled;

    NN_RESULT_SUCCESS;
}

::nn::Result AbstractedPadXcdManager::EnableUsbFullKeyController(bool enabled) NN_NOEXCEPT
{
    ::nn::xcd::SetFullKeyUsbEnabled(enabled);
    m_IsUsbFullKeyControllerEnabled = enabled;

    NN_RESULT_SUCCESS;
}

void AbstractedPadXcdManager::ResetAllDevices() NN_NOEXCEPT
{
    for (int padIndex = 0; padIndex < m_PadCount; padIndex++)
    {
        if (m_pPads[padIndex]->IsAttached() == true &&
            m_pPads[padIndex]->GetInterfaceType() != system::InterfaceType_Rail)
        {
            m_pPads[padIndex]->Reboot(false);
        }
    }
}

}}} // namespace nn::hid::detail
