﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/debug/hid_AbstractedPad.h>
#include <nn/hid/detail/hid_AbstractedPadTypes.h>

#include "hid_AbstractedAutoPilotVirtualPadManager.h"
#include "hid_AbstractedPadHidShellManager-os.win.h"
#include "hid_AbstractedPadManager.h"
#include "hid_HidShellPortFile-os.win.h"
#include "hid_IAbstractedPad.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< AbstractedPad の状態を取得します。
void GetAbstractedPadState(
    ::nn::hid::debug::AbstractedPadState* pOutState,
    IAbstractedPad& pad, AbstractedPadState* pTemp) NN_NOEXCEPT;

} // namespace

AbstractedPadHidShellManager::AbstractedPadHidShellManager() NN_NOEXCEPT
{
    this->ResetIndices();
}

void AbstractedPadHidShellManager::SetAbstractedPadManager(
    AbstractedPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pAbstractedPadManager = pManager;
}

void AbstractedPadHidShellManager::SetAbstractedAutoPilotVirtualPadManager(
    AbstractedAutoPilotVirtualPadManager* pManager) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pManager);

    m_pVirtualPadManager = pManager;
}

void AbstractedPadHidShellManager::Recieve() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pVirtualPadManager);

    for (int i = 0; i < AbstractedAutoPilotVirtualPadCount; ++i)
    {
        auto flag = bool();

        if (GetHidShellAbstractedPadState(&flag, &m_State, i).IsFailure())
        {
            flag = false;
        }

        auto id = static_cast<::nn::hid::debug::AbstractedVirtualPadId>(i);

        if (flag)
        {
            m_pVirtualPadManager->SetAutoPilotState(id, m_State);
        }
        else
        {
            m_pVirtualPadManager->UnsetAutoPilotState(id);
        }
    }
}

void AbstractedPadHidShellManager::Send() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pAbstractedPadManager);

    auto idCount = int();

    if (m_pAbstractedPadManager
            ->GetActiveAbstractedPadIds(
                &idCount, m_NextIds, NN_ARRAY_SIZE(m_NextIds)).IsFailure())
    {
        idCount = 0;
    }

    for (int i = 0; i < idCount; ++i)
    {
        m_LastIds.Remove(m_NextIds[i]);
    }

    m_State = ::nn::hid::debug::AbstractedPadState();

    for (size_t i = 0; i < m_LastIds.GetCount(); ++i)
    {
        size_t* pIndex = m_Table.Get(m_LastIds[i]);

        if (pIndex != nullptr)
        {
            SetHidShellAbstractedPadState(static_cast<int>(*pIndex), m_State);

            m_Table.Remove(m_LastIds[i]);

            m_Indices.Append(*pIndex);

            if (m_Indices.GetCount() == PadCountMax)
            {
                this->ResetIndices();
            }
        }
    }

    for (int i = 0; i < idCount; ++i)
    {
        IAbstractedPad* pPad = nullptr;

        if (m_pAbstractedPadManager
                ->GetAbstractedPad(&pPad, m_NextIds[i]).IsFailure() ||
            pPad == nullptr ||
            pPad->GetType() == AbstractedPadType_AutopilotVirtual)
        {
            continue;
        }

        GetAbstractedPadState(&m_State, *pPad, &m_Temp);

        size_t* pIndex = m_Table.Get(m_NextIds[i]);

        if (pIndex != nullptr)
        {
            SetHidShellAbstractedPadState(static_cast<int>(*pIndex), m_State);
        }
        else
        {
            size_t index = m_Indices[0];

            m_Table.Set(m_NextIds[i], index);

            m_Indices.Remove(index);

            SetHidShellAbstractedPadState(static_cast<int>(index), m_State);
        }
    }

    m_LastIds.Clear();

    for (int i = 0; i < idCount; ++i)
    {
        m_LastIds.Append(m_NextIds[i]);
    }
}

void AbstractedPadHidShellManager::ResetIndices() NN_NOEXCEPT
{
    m_Indices.Clear();

    for (size_t i = 0; i < PadCountMax; ++i)
    {
        m_Indices.Append(i);
    }
}

namespace {

void GetAbstractedPadState(
    ::nn::hid::debug::AbstractedPadState* pOutState,
    IAbstractedPad& pad, AbstractedPadState* pTemp) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutState);
    NN_SDK_REQUIRES_NOT_NULL(pTemp);
    ::nn::hid::debug::AbstractedPadState& state = *pOutState;
    AbstractedPadState& temp = *pTemp;
    state.deviceType = pad.GetDeviceType();
    state.attributes.Set<
        ::nn::hid::debug::AbstractedPadAttribute::IsConnected>(
            pad.IsConnected());
    state.color = pad.GetColor();
    state.interfaceType = pad.GetInterfaceType();
    state.powerInfo = pad.GetPowerInfo();
    temp = pad.GetPadState();
    state.buttons = temp.buttons;
    state.analogStickL = temp.analogStickL;
    state.analogStickR = temp.analogStickR;
}

} // namespace

}}} // namespace nn::hid::detail
