﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedGcAdapterPadManager.h"

namespace nn { namespace hid { namespace detail {

AbstractedGcAdapterPadManager::AbstractedGcAdapterPadManager() NN_NOEXCEPT
    : m_ActivationCount()
{
}

AbstractedGcAdapterPadManager::~AbstractedGcAdapterPadManager() NN_NOEXCEPT
{
    // 何もしない
}

void AbstractedGcAdapterPadManager::SetAbstractedPadIdPublisher(AbstractedPadIdPublisher* pPublisher) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPublisher);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    m_pIdPublisher = pPublisher;
}

void AbstractedGcAdapterPadManager::SetAbstractedGcAdapterPads(AbstractedGcAdapterPad* pPads, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_MINMAX(count, 1, static_cast<int>(GcAdapterCountMax * GcPortPerAdapterCountMax));
    for (int padIndex = 0; padIndex < count; padIndex++)
    {
        m_pPads[padIndex] = &pPads[padIndex];
    }
    m_PadCount = count;
}

void AbstractedGcAdapterPadManager::SetGcAdapterDriver(GcAdapterDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);

    m_pDriver = pDriver;
}

::nn::Result AbstractedGcAdapterPadManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPads);
    NN_SDK_REQUIRES_GREATER(m_PadCount, 0);
    NN_SDK_REQUIRES_NOT_NULL(m_pIdPublisher);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;

}

::nn::Result AbstractedGcAdapterPadManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

void AbstractedGcAdapterPadManager::Update() NN_NOEXCEPT
{
    if (m_ActivationCount.IsZero())
    {
        return;
    }

    auto list = m_pDriver->ListAdapters();
    CheckForDeviceDetach(list, m_Adapters);

    for (int newIndex = 0; newIndex < list.count; ++newIndex)
    {
        // 新規に接続されたデバイスかどうか
        bool newDevice = true;
        for (int prevIndex = 0; prevIndex < m_Adapters.count; ++prevIndex)
        {
            if (m_Adapters.indexes[prevIndex] == list.indexes[newIndex])
            {
                newDevice = false;
                break;
            }
        }

        // 新規に接続されたデバイスは AbstractedPad に追加
        if (newDevice == true)
        {
            GcControllerIndex index = { list.indexes[newIndex], 0 };
            for (index.portIndex = 0; index.portIndex < GcPortPerAdapterCountMax; ++index.portIndex)
            {
                this->AddDevice(index);
            }
        }
    }

    m_Adapters = list;

    // Attach されているものについて Update を実行
    for (int i = 0; i < m_PadCount; ++i)
    {
        if (m_pPads[i]->IsAttached() == true)
        {
            m_pPads[i]->Update();
        }
    }
}

void AbstractedGcAdapterPadManager::CheckForDeviceDetach(GcAdapterList& newList, GcAdapterList& prevList) NN_NOEXCEPT
{
    // もともとリストの中にデバイスがない場合は何もしない
    if (prevList.count == 0)
    {
        return;
    }

    // まだデバイスが接続されているかどうか探索
    for (int prevIndex = 0; prevIndex < prevList.count; ++prevIndex)
    {
        bool isConnected = false;

        for (int newIndex = 0; newIndex < newList.count; ++newIndex)
        {
            if (prevList.indexes[prevIndex] == newList.indexes[newIndex])
            {
                isConnected = true;
                break;
            }
        }

        if (isConnected == false)
        {
            // アダプターインデックスが一致するデバイスをすべて Remove する
            for (int padIndex = 0; padIndex < m_PadCount; ++padIndex)
            {
                auto gcIndex = m_pPads[padIndex]->GetIndex();
                if (gcIndex.adapterIndex == prevList.indexes[prevIndex])
                {
                    m_pPads[padIndex]->RemoveDevice();
                }
            }
        }
    }
}

void AbstractedGcAdapterPadManager::AddDevice(GcControllerIndex index) NN_NOEXCEPT
{
    for (int padIndex = 0; padIndex < m_PadCount; padIndex++)
    {
        if (m_pPads[padIndex]->IsAttached() == false)
        {
            // あいている Pad に追加 & 新規に Id 発行
            m_pPads[padIndex]->AttachDevice(index, m_pIdPublisher->PublishId());
            return;
        }
    }
}



}}} // namespace nn::hid::detail
