﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include "hid_AbstractedGcAdapterPad.h"
#include "hid_CommonStateUtility.h"
#include "hid_GcAdapterDriver.h"

namespace nn { namespace hid { namespace detail {

namespace {

//!< c コントローラー共通の電源状態
const system::PowerInfo StaticUsbPowerInfo = { false, false, system::BatteryLevel_High };

} // namespace

AbstractedGcAdapterPad::AbstractedGcAdapterPad() NN_NOEXCEPT
    : m_IsConnected(false)
{
    // Usb 接続
    m_InterfaceType = system::InterfaceType_Usb;
    // Switch Pro Controller 互換
    m_DeviceType = system::DeviceType::SwitchProController::Mask;

    // FeatureSet
    m_FeatureSet.Reset();
    m_FeatureSet.Set<AbstractedPadFeature::AnalogStick>();
    m_FeatureSet.Set<AbstractedPadFeature::GcErm>();
}

AbstractedGcAdapterPad::~AbstractedGcAdapterPad() NN_NOEXCEPT
{
    // 何もしない
}

AbstractedPadType AbstractedGcAdapterPad::GetType() NN_NOEXCEPT
{
    return s_Type;
}

bool AbstractedGcAdapterPad::IsConnected() NN_NOEXCEPT
{
    return m_IsConnected;
}

system::PowerInfo AbstractedGcAdapterPad::GetPowerInfo() NN_NOEXCEPT
{
    return StaticUsbPowerInfo;
}

AbstractedPadState AbstractedGcAdapterPad::GetPadState() NN_NOEXCEPT
{
    return m_PadState;
}

bool AbstractedGcAdapterPad::GetButtonTriggerElapsedTime(nn::os::Tick* pOutTick, AbstractedPadButtonSet button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(button.CountPopulation(), 1);
    NN_UNUSED(pOutTick);
    NN_UNUSED(button);

    // ボタン押しをさかのぼる機能は提供しない
    return false;
}

int AbstractedGcAdapterPad::GetSensorStates(nn::xcd::SixAxisSensorState* pOutValue, int count) NN_NOEXCEPT
{
    NN_UNUSED(pOutValue);
    NN_UNUSED(count);
    return 0;
}

void AbstractedGcAdapterPad::SetIndicator(uint8_t pattern, bool blink) NN_NOEXCEPT
{
    NN_UNUSED(blink);

    if (m_IsConnected)
    {
        m_Pattern = pattern;
    }
}

uint8_t AbstractedGcAdapterPad::GetIndicator() NN_NOEXCEPT
{
    if (m_IsConnected)
    {
        return m_Pattern;
    }

    return 0;
}

bool AbstractedGcAdapterPad::IsWired() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    // USB コントローラーは常時 Wired 状態
    return true;
}

bool AbstractedGcAdapterPad::IsUsbConnected() NN_NOEXCEPT
{
    if (m_IsConnected == false)
    {
        return false;
    }
    // USB コントローラーは常時 USB 接続状態
    return true;
}

void AbstractedGcAdapterPad::Connect() NN_NOEXCEPT
{
    if (m_IsAttached == false)
    {
        return;
    }

    // 接続状態に変更
    if (m_IsConnected == false)
    {
        m_IsConnected = true;
    }
}

void AbstractedGcAdapterPad::Detach() NN_NOEXCEPT
{
    m_IsConnected = false;
    m_PadState = AbstractedPadState();
    m_TriggerL = 0;
    m_TriggerR = 0;
}

void AbstractedGcAdapterPad::VibrationOnConnect() NN_NOEXCEPT
{
    m_pDriver->ControlMotorOnConnect(m_Index);
}


bool AbstractedGcAdapterPad::HasBattery() NN_NOEXCEPT
{
    return false;
}

bool AbstractedGcAdapterPad::SetDeviceInfoOnPlayReportControllerUsage(system::PlayReportControllerUsage* pOutValue) NN_NOEXCEPT
{
    pOutValue->deviceType = system::PlayReportDeviceType_GcController;
    for (auto& byte : pOutValue->raw)
    {
        byte = 0;
    }
    return true;
}

void AbstractedGcAdapterPad::SetGcAdapterDriver(GcAdapterDriver* pDriver) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDriver);

    m_pDriver = pDriver;
}

void AbstractedGcAdapterPad::AttachDevice(const GcControllerIndex& index, AbstractedPadId id) NN_NOEXCEPT
{
    if(m_IsAttached == true)
    {
        return;
    }
    m_Id = id;
    m_Index = index;
    m_IsAttached = true;
    m_IsNewSampleReceived = false;
    m_ButtonMask.Reset();
}

void AbstractedGcAdapterPad::RemoveDevice() NN_NOEXCEPT
{
    m_Id = AbstractedPadId();
    m_Index = GcControllerIndex();
    m_IsConnected = false;
    m_IsAttached = false;
}

GcControllerIndex AbstractedGcAdapterPad::GetIndex() NN_NOEXCEPT
{
    if(m_IsAttached == false)
    {
        return GcControllerIndex();
    }

    return m_Index;
}

void AbstractedGcAdapterPad::Update() NN_NOEXCEPT
{
    if (m_IsAttached == false)
    {
        return;
    }
    auto state = GcControllerState();
    if (m_pDriver->GetState(&state, m_Index) == true)
    {
        // 離されたボタンは無効化を解除
        m_ButtonMask |= ~state.buttons;

        // 無効化
        state.buttons &= m_ButtonMask;

        // 何かボタンが押されたら接続状態へと変更する
        if (m_IsConnected == false &&
            ((m_OldButtons ^ state.buttons) & state.buttons).IsAnyOn())
        {
            m_IsConnected = true;
            // 入力無効化するボタンを追加
            m_ButtonMask &= ~state.buttons;
        }
        m_OldButtons = state.buttons;

        if (m_IsConnected == true)
        {
            m_IsNewSampleReceived = true;

            m_PadState.buttons.Reset();
            m_PadState.buttons.Set<AbstractedPadButton::A>(state.buttons.Test<GcControllerButton::A>());
            m_PadState.buttons.Set<AbstractedPadButton::B>(state.buttons.Test<GcControllerButton::B>());
            m_PadState.buttons.Set<AbstractedPadButton::X>(state.buttons.Test<GcControllerButton::X>());
            m_PadState.buttons.Set<AbstractedPadButton::Y>(state.buttons.Test<GcControllerButton::Y>());
            m_PadState.buttons.Set<AbstractedPadButton::L>(state.buttons.Test<GcControllerButton::L>());
            m_PadState.buttons.Set<AbstractedPadButton::R>(state.buttons.Test<GcControllerButton::R>());
            m_PadState.buttons.Set<AbstractedPadButton::ZR>(state.buttons.Test<GcControllerButton::Z>());
            m_PadState.buttons.Set<AbstractedPadButton::Start>(state.buttons.Test<GcControllerButton::Start>());
            m_PadState.buttons.Set<AbstractedPadButton::Left>(state.buttons.Test<GcControllerButton::Left>());
            m_PadState.buttons.Set<AbstractedPadButton::Up>(state.buttons.Test<GcControllerButton::Up>());
            m_PadState.buttons.Set<AbstractedPadButton::Right>(state.buttons.Test<GcControllerButton::Right>());
            m_PadState.buttons.Set<AbstractedPadButton::Down>(state.buttons.Test<GcControllerButton::Down>());

            AnalogStickState center;
            int16_t originPlay;
            int16_t outRange;
            if (m_pDriver->GetLeftAnalogStickCalibration(&center, &originPlay, &outRange, m_Index).IsSuccess())
            {
                m_PadState.analogStickL = ConvertGcAnalogStickState(state.analogStickL, center, originPlay, outRange);
            }
            if (m_pDriver->GetRightAnalogStickCalibration(&center, &originPlay, &outRange, m_Index).IsSuccess())
            {
                m_PadState.analogStickR = ConvertGcAnalogStickState(state.analogStickR, center, originPlay, outRange);
            }
            SetCrossStickEmulationButtonsOnAbstractedPad(&m_PadState);

            m_TriggerL = state.triggerL;
            m_TriggerR = state.triggerR;
        }
    }
    else
    {
        m_IsConnected = false;
        m_PadState = AbstractedPadState();
        m_TriggerL = 0;
        m_TriggerR = 0;
    }
}

void AbstractedGcAdapterPad::GetTrigger(int32_t* pOutTriggerL, int32_t* pOutTriggerR) NN_NOEXCEPT
{
    *pOutTriggerL = m_TriggerL;
    *pOutTriggerR = m_TriggerR;
}

}}} // namespace nn::hid::detail
