﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/os/os_Tick.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/xcd/xcd.h>

#include "hid_AbstractedAutoPilotPadManager.h"
#include "hid_NpadId.h"

namespace nn { namespace hid { namespace detail {

AbstractedAutoPilotPadManager::AbstractedAutoPilotPadManager() NN_NOEXCEPT
    : m_ActivationCount()
{
}

AbstractedAutoPilotPadManager::~AbstractedAutoPilotPadManager() NN_NOEXCEPT
{
    // 何もしない
}

void AbstractedAutoPilotPadManager::SetAbstractedAutoPilotPads(AbstractedAutoPilotPad* pPads, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pPads);
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_SDK_REQUIRES_LESS_EQUAL(count, AbstractedAutoPilotPadCount);
    NN_SDK_REQUIRES(m_ActivationCount.IsZero());
    for (int padIndex = 0; padIndex < count; padIndex++)
    {
        m_pPads[padIndex] = &pPads[padIndex];
        m_pPads[padIndex]->SetNpadIdType(SupportedNpadIdList[padIndex]);
    }
    m_PadCount = count;
}

::nn::Result AbstractedAutoPilotPadManager::Activate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pPads);
    NN_SDK_REQUIRES_GREATER(m_PadCount, 0);

    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsMax(),
                           ResultGamePadDriverActivationUpperLimitOver());

    if (m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    // アクティブ化した回数をインクリメント
    ++m_ActivationCount;

    NN_RESULT_SUCCESS;

}

::nn::Result AbstractedAutoPilotPadManager::Deactivate() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(!m_ActivationCount.IsZero(),
                           ResultGamePadDriverDeactivationLowerLimitOver());

    // アクティブ化した回数をデクリメント
    --m_ActivationCount;

    if(m_ActivationCount.IsZero())
    {
        // 何もしない
    }

    NN_RESULT_SUCCESS;
}

::nn::Result AbstractedAutoPilotPadManager::SetAutoPilotState(int playerNumber,
    const ::nn::hid::debug::BasicXpadAutoPilotState& value) NN_NOEXCEPT
{
    if (playerNumber > m_PadCount)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(m_pPads[playerNumber]->SetAutoPilotState(value));
    NN_RESULT_SUCCESS;
}

::nn::Result AbstractedAutoPilotPadManager::UnsetAutoPilotState(int playerNumber) NN_NOEXCEPT
{
    if (playerNumber > m_PadCount)
    {
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_DO(m_pPads[playerNumber]->UnsetAutoPilotState());
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::hid::detail
