﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using nn;
using nn.sf;
using std;
using Nintendo.ServiceFramework;
using Nintendo.ServiceFramework.CppCode;
using Nintendo.ServiceFramework.Hipc;

namespace nn.ncm
{
    [ExternalStruct(8, 8)]
    [CppRefPath("<nn/ncm/ncm_ContentMetaId.h>")]
    public struct ApplicationId
    {
    }
}

namespace nn.applet
{
    [ExternalStruct(8, 8)]
    [CppRefPath("<nn/applet/applet_FundamentalTypes.h>")]
    public struct AppletResourceUserId
    {
    }

    [CppRefPath("<nn/applet/applet_FundamentalTypes.h>")]
    [Nintendo.ServiceFramework.Applet.CheckAppletResourceUserId]
    [CppFullName("nn::applet::AppletResourceUserId")]
    [ExternalStruct(8, 8)]
    public struct CheckedAppletResourceUserId
    {
    }
}

namespace nns.hosbinder
{
    [CppRefPath("<nn/vi/sf/hosbinder_IHOSBinderDriver.sfdl.h>")]
    public interface IHOSBinderDriver : IServiceObject
    {
    }
}

namespace nn.grcsrv
{
    [CppRefPath(Location.GrcServiceSfdlPath)]
    public struct ContinuousRecordingParameter
    {
        [FixedArray(16)]
        public Bit32[] data;
    }

    [CppRefPath(Location.GrcServiceSfdlPath)]
    public struct ContinuousRecordingFlushParameter
    {
        [FixedArray(64)]
        public Bit8[] data;
    }

    [CppRefPath(Location.GrcServiceSfdlPath)]
    public interface IGrcService : IServiceObject
    {
        [MethodId(1)]
        Result CreateContinuousRecorder(Out<IContinuousRecorder> pOut, NativeHandle workMemoryHandle, uint64_t workMemorySize, ContinuousRecordingParameter parameter);

        [MethodId(2)]
        Result CreateGameMovieTrimmer(Out<IGameMovieTrimmer> pOut, NativeHandle workMemoryHandle, uint64_t workMemorySize);

        [MethodId(3)]
        Result CreateOffscreenRecorder(Out<IOffscreenRecorder> pOut, NativeHandle workMemoryHandle, uint64_t workMemorySize);

        // アプリが grc と通信するためのサービスオブジェクトを作成する。
        // aruid にはアプリの aruid と applicationId を指定する。
        [MethodId(101)]
        Result CreateMovieMakerForApplication(Out<IMovieMaker> outProxy, applet.AppletResourceUserId aruid, ncm.ApplicationId applicationId);

        // テスト用。
        [MethodId(9903)]
        Result SetOffscreenRecorderCheckPointForTesting(uint64_t value);
    }

    [CppRefPath(Location.GrcServiceSfdlPath)]
    public interface IContinuousRecorder : IServiceObject
    {
        #region Continuous Recoding

        /// <summary>
        /// 常時録画を開始する。
        /// </summary>
        /// <returns></returns>
        [MethodId(1)]
        Result StartRecording();

        /// <summary>
        /// 常時録画を停止する。
        /// </summary>
        /// <returns></returns>
        [MethodId(2)]
        Result StopRecording();

#if false // 未実装
        [MethodId(3)]
        Result ClearRecodingData();
#endif

        #endregion

        #region Flushing

        /// <summary>
        /// Flush が走っていないときにシグナルされているイベントを取得する。
        /// </summary>
        /// <param name="pOutSystemEventHandle"></param>
        /// <returns></returns>
        /// <remarks>
        /// このイベントがシグナルされているときは StartFlush() は成功する。
        /// </remarks>
        [MethodId(10)]
        Result GetNotFlushingEvent(Out<NativeHandle> pOutSystemEventHandle);

        /// <summary>
        /// Flush を開始する。
        /// </summary>
        /// <param name="flushParameter"></param>
        /// <returns></returns>
        /// <remarks>
        /// Flush が動作中の場合、grc::ResultFlushIsInProgress が返る。
        /// Flush が動作中でなかった場合、Flush を開始する。
        /// </remarks>
        [MethodId(11)]
        Result StartFlush(ContinuousRecordingFlushParameter flushParameter);

        /// <summary>
        /// 実行中の Flush 処理を可能な限り早く中断する。
        /// </summary>
        /// <returns></returns>
        /// <remarks>
        /// Flush 処理がすでに終わっていたような場合には、何もしない。
        /// </remarks>
        [MethodId(12)]
        Result CancelFlush();

        /// <summary>
        /// 次回の Flush を行う時間を再設定する。
        /// </summary>
        /// <param name="additionalTimeNs"></param>
        /// <returns></returns>
        [MethodId(13)]
        Result ResetFlushTime(int64_t additionalTimeNs);

        /// <summary>
        /// Flush を開始する。
        /// </summary>
        /// <param name="flushParameter"></param>
        /// <returns></returns>
        /// <remarks>
        /// Flush が動作中の場合、grc::ResultFlushIsInProgress が返る。
        /// Flush が動作中でなかった場合、Flush を開始する。
        /// </remarks>
        [MethodId(14)]
        Result StartFlushWithEvent(Out<NativeHandle> pOut, ContinuousRecordingFlushParameter flushParameter);

        #endregion
    }

    [CppRefPath(Location.GrcServiceSfdlPath)]
    public struct GameMovieId
    {
        [FixedArray(8)]
        public Bit64[] data;
    }

    [CppRefPath(Location.GrcServiceSfdlPath)]
    public interface IGameMovieTrimmer : IServiceObject
    {
        [MethodId(1)]
        Result BeginTrim(GameMovieId id, int beginIndex, int endIndex);

        [MethodId(2)]
        Result EndTrim(Out<GameMovieId> pOut);

        [MethodId(10)]
        Result GetNotTrimmingEvent(Out<NativeHandle> pOutSystemEventHandle);

        [MethodId(20)]
        Result SetThumbnailRgba([NonSecureMapTransfer] InBuffer imageData, int width, int height);
    }

    [CppRefPath(Location.GrcServiceSfdlPath)]
    public interface IOffscreenRecorder : IServiceObject
    {
        [MethodId(201)]
        Result CreateOffscreenLayer(Out<uint64_t> outHandle, applet.AppletResourceUserId rendererAruid);

        [MethodId(202)]
        Result DestroyOffscreenLayer(uint64_t handle);
    }

#region アプリ向けインタフェース
    [CppRefPath(Location.GrcServiceSfdlPath)]
    public struct OffscreenRecordingParameter
    {
        [FixedArray(16)]
        public Bit64[] data;
    }

    // アプリから grc に直接 IPC を張る場合の入り口。
    // AM を経由するなら IGrcService を使う。
    [CppRefPath(Location.GrcServiceSfdlPath)]
    public interface IMovieMakerService : IServiceObject
    {
        [MethodId(101)]
        Result CreateMovieMaker(Out<IMovieMaker> outProxy, applet.CheckedAppletResourceUserId aruid, ncm.ApplicationId applicationId);
    }

    [CppRefPath(Location.GrcServiceSfdlPath)]
    public interface IMovieMaker : IServiceObject
    {
        // libandroid(nvnflinger) が定義するビデオ用インタフェースを返す。
        [MethodId(2)]
        Result CreateVideoProxy(Out<nns.hosbinder.IHOSBinderDriver> outProxy);

        [MethodId(10)]
        Result OpenOffscreenLayer(Out<int32_t> outProducerHandle, uint64_t layerHandle);

        [MethodId(11)]
        Result CloseOffscreenLayer(uint64_t layerHandle);

        // 暫定版。パラメータはデフォルト値で設定する。
        [MethodId(20)]
        Result StartOffscreenRecording(uint64_t layerHandle);

        [MethodId(21)]
        Result AbortOffscreenRecording(uint64_t layerHandle);

        [MethodId(22)]
        Result RequestOffscreenRecordingFinishReady(uint64_t layerHandle);

        [MethodId(23)]
        Result CompleteOffscreenRecordingFinish(uint64_t layerHandle, InBuffer userData);

        [MethodId(24)]
        Result StartOffscreenRecordingEx(uint64_t layerHandle, OffscreenRecordingParameter paramData);

        // MethodId=23 のサムネイル追加版
        [MethodId(25)]
        Result CompleteOffscreenRecordingFinishEx0(uint64_t layerHandle, InBuffer userData, InBuffer thumbnailImage, int thumbnailImageWidth, int thumbnailImageHeight);

        [MethodId(30)]
        Result GetOffscreenLayerError(uint64_t layerHandle);

        [MethodId(41)]
        Result EncodeOffscreenLayerAudioSample(Out<uint64_t> outEncodedSize, uint64_t layerHandle, InBuffer buffer);

        [MethodId(50)]
        Result GetOffscreenLayerRecordingFinishReadyEvent(Out<sf.NativeHandle> outHandle, uint64_t layerHandle);

        [MethodId(52)]
        Result GetOffscreenLayerAudioEncodeReadyEvent(Out<sf.NativeHandle> outHandle, uint64_t layerHandle);
    }
#endregion
}
