﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_Optional.h>
#include <nn/os.h>
#include <nvnflinger_service.h>
#include "../native/grcsrv_SyncpointWaiter.h"
#include "../util/grcsrv_TQueue.h"
#include "../util/grcsrv_TNullableContainer.h"
#include "../grcsrvOffscreen_Config.h"
#include "../grcsrvOffscreen_MultiWaitHandler.h"
#include "../../../capsrv/capture/capsrv_CaptureModule.h"
#include "grcsrvOffscreen_QueueType.h"

namespace nn{ namespace grcsrv{ namespace offscreen{ namespace detail{

    class VideoRendererHandler
    {
    public:
        enum class State
        {
            NotInitialized,
            Initializing, // Initialize 中の状態。即座に Idle に変更される。
            Finalizing,   // Finalize 中の状態。即座に NotInitialized に変更される。
            Idle,
            Encoding,
            Finishing,
            Exited,
        };

        typedef VideoCaptureToVideoRendererQueue AcquirableOutputBufferQueue;
        typedef VideoRendererToVideoCaptureQueue QueuedOutputBufferQueue;

    public:
        VideoRendererHandler() NN_NOEXCEPT;

        bool IsInitialized() const NN_NOEXCEPT;

        nn::util::optional<nn::Result> GetExitResult() const NN_NOEXCEPT;

        nn::Result Initialize(
            const android::sp<android::IGraphicBufferConsumer>& pConsumer,
            nn::os::SystemEventType* pConsumerBufferAcquirableEvent,
            AcquirableOutputBufferQueue* pAcquirableOutputBufferQueue,
            QueuedOutputBufferQueue* pQueuedOutputBufferQueue
        ) NN_NOEXCEPT;

        void Finalize() NN_NOEXCEPT;

        nn::Result Start(int frameRate) NN_NOEXCEPT;
        void Abort() NN_NOEXCEPT;
        void Reset() NN_NOEXCEPT;

        void RequestFinish() NN_NOEXCEPT;
        void NotifyError() NN_NOEXCEPT;

        void InitializeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;
        void FinalizeExitWaitHolder(nn::os::MultiWaitHolderType* pHolder) NN_NOEXCEPT;

    private:
        void ChangeStateIdleImpl() NN_NOEXCEPT;
        void ChangeStateEncodingImpl() NN_NOEXCEPT;
        void ChangeStateFinishingImpl() NN_NOEXCEPT;
        void ChangeStateExitedImpl(nn::Result exitResult) NN_NOEXCEPT;
        void ChangeStateFinalizingImpl() NN_NOEXCEPT;

        // ChangeState 内部で使用。 MultiWait の解除を行う。
        void ExitCurrentStateImpl() NN_NOEXCEPT;

    private:
        nn::Result AcquireConsumerBufferImpl(android::sp<android::GraphicBuffer>* pOutBuffer, android::IGraphicBufferConsumer::BufferItem* pOutItem) NN_NOEXCEPT;
        nn::Result ReleaseConsumerBufferImpl(const android::IGraphicBufferConsumer::BufferItem& item, const android::sp<android::Fence>& fence) NN_NOEXCEPT;

        // Cunsumer からバッファを Acquire して即座に Release する。
        // 失敗した場合、 nn::Result で返す。この関数は State を変更しない。
        nn::Result AcquireInputBufferAndReleaseImpl() NN_NOEXCEPT;

        // Consumer からバッファを Acquire してエンコーダに送る
        // 失敗した場合、 nn::Result で返す。この関数は State を変更しない。
        nn::Result AcquireInputBufferAndSendToEncoderImpl(bool isNoBufferAvailableErrorIgnored) NN_NOEXCEPT;

        // Output からバッファを Dequeue して Consumer に返す。
        nn::Result DequeueOutputBufferAndReleaseToConsumerImpl() NN_NOEXCEPT;

        // すべてのエンコード中のバッファを Cunsumer に返却する。
        void ForceReleaseAllEncodingBuffersImpl() NN_NOEXCEPT;

        // usrPtr == this
        static void OnInputAcquirableCallbackFunction(MultiWaitHandler* pHandler, void* usrPtr) NN_NOEXCEPT;
        static void OnOutputDequeueableCallbackFunction(MultiWaitHandler* pHandler, void* usrPtr) NN_NOEXCEPT;

    private:
        State m_State;

        nn::util::optional<nn::Result> m_ExitResult;
        nn::os::EventType m_ExitEvent;

        android::sp<android::IGraphicBufferConsumer> m_pConsumer;
        android::sp<android::GraphicBuffer> m_ConsumerBufferList[ClientImageBufferCountMax];
        nn::os::SystemEventType* m_pConsumerBufferAcquirableEvent;

        AcquirableOutputBufferQueue* m_pAcquirableOutputBufferQueue;
        QueuedOutputBufferQueue*     m_pQueuedOutputBufferQueue;

        MultiWaitHandler m_InputAcquirableWaitHandler;
        MultiWaitHandler m_OutputDequeueableWaitHandler;

        int     m_FrameRate;
        int64_t m_FrameCount;

        struct EncodingBufferEntry
        {
            struct InvalidValue
            {
                static EncodingBufferEntry Get() NN_NOEXCEPT
                {
                    EncodingBufferEntry e;
                    e.item = {};
                    e.frameNumber = -1;
                    return e;
                }
            };
        public:
            bool operator==(const EncodingBufferEntry& v) const NN_NOEXCEPT { return this->frameNumber == v.frameNumber; }
            bool operator!=(const EncodingBufferEntry& v) const NN_NOEXCEPT { return this->frameNumber != v.frameNumber; }

        public:
            android::IGraphicBufferConsumer::BufferItem item;
            int64_t frameNumber;
        };
        typedef util::TNullableContainer<EncodingBufferEntry, ClientImageBufferCountMax, EncodingBufferEntry::InvalidValue, util::ContainerThreadSafetyOption_SingleThread> EncodingBufferList;
        EncodingBufferList m_EncodingBufferList;

    };

}}}}
