﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_MovieMakerClient.h"

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/grc/grc_Result.h>

#include <nn/grcsrv/grcsrv_Offscreen.h>
#include "grcsrv_OffscreenRecorderImpl.h"

namespace nn{ namespace grcsrv{

    MovieMakerClient::MovieMakerClient(nn::applet::AppletResourceUserId aruid, ncm::ApplicationId applicationId) NN_NOEXCEPT
        : m_IsAlive(true)
        , m_Aruid(aruid)
        , m_ApplicationId(applicationId)
        , m_LayerDestroyingCallbackSlot(-1)
    {
        // コールバックを登録
        RegisterOffscreenLayerDestroyingCallback(
            &m_LayerDestroyingCallbackSlot,
            [](uint64_t layerHandle, void* usrPtr) -> void
            {
                auto pSelf = reinterpret_cast<MovieMakerClient*>(usrPtr);
                (void)pSelf->CloseOffscreenLayer(layerHandle);
            },
            this
        );
    }

    MovieMakerClient::~MovieMakerClient() NN_NOEXCEPT
    {
        NN_SDK_ASSERT(!m_IsAlive);
    }

    bool MovieMakerClient::IsAlive() const NN_NOEXCEPT
    {
        return m_IsAlive;
    }

    void MovieMakerClient::Invalidate() NN_NOEXCEPT
    {
        CloseAllOffscreenLayerImpl();

        // コールバックを登録解除
        UnregisterOffscreenLayerDestroyingCallback(m_LayerDestroyingCallbackSlot);
        m_LayerDestroyingCallbackSlot = -1;

        m_IsAlive = false;
    }

    void MovieMakerClient::CloseAllOffscreenLayerImpl() NN_NOEXCEPT
    {
        if(m_RecordingLayerHandle != nn::util::nullopt)
        {
            NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
            nn::grcsrv::offscreen::AbortLayerRecording(*m_RecordingLayerHandle);
            m_RecordingLayerHandle = nn::util::nullopt;
        }

        if(m_BoundLayerHandle != nn::util::nullopt)
        {
            NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
            nn::grcsrv::offscreen::UnbindRendererLayer(m_BoundLayerHandle->first);
            m_BoundLayerHandle = nn::util::nullopt;
        }
    }

    nn::Result MovieMakerClient::OpenOffscreenLayer(std::int32_t* pOutProducerHandle, std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_BoundLayerHandle == nn::util::nullopt, nn::grc::ResultOutOfResource());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());

        int32_t producerHandle = {};
        NN_RESULT_DO(nn::grcsrv::offscreen::BindRendererLayer(&producerHandle, layerHandle, m_Aruid));

        m_BoundLayerHandle.emplace(std::make_pair(layerHandle, producerHandle));
        *pOutProducerHandle = producerHandle;
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::CloseOffscreenLayer(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_BoundLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == m_BoundLayerHandle->first, nn::grc::ResultInvalidCall());

        if(m_RecordingLayerHandle != nn::util::nullopt && *m_RecordingLayerHandle == layerHandle)
        {
            nn::grcsrv::offscreen::AbortLayerRecording(layerHandle);
            m_RecordingLayerHandle = nn::util::nullopt;
        }

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        nn::grcsrv::offscreen::UnbindRendererLayer(layerHandle);
        m_BoundLayerHandle = nn::util::nullopt;

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::StartOffscreenRecording(std::uint64_t layerHandle, const nn::grc::OffscreenRecordingParameter& srcParam) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_BoundLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == m_BoundLayerHandle->first, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());

        nn::grc::OffscreenRecordingParameter param = srcParam;
        param.applicationId = m_ApplicationId;
        NN_RESULT_DO(nn::grcsrv::offscreen::StartLayerRecording(layerHandle, param));
        m_RecordingLayerHandle = layerHandle;

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::AbortOffscreenRecording(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_RecordingLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == *m_RecordingLayerHandle, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        nn::grcsrv::offscreen::AbortLayerRecording(layerHandle);
        m_RecordingLayerHandle = nn::util::nullopt;

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::RequestOffscreenRecordingFinishReady(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_RecordingLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == *m_RecordingLayerHandle, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        NN_RESULT_DO(nn::grcsrv::offscreen::RequestFinishLayerRecording(layerHandle));

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::CompleteOffscreenRecordingFinish(std::uint64_t layerHandle, const capsrv::ApplicationData& applicationData, const sf::InBuffer& thumbnailImage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_RecordingLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == *m_RecordingLayerHandle, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        NN_RESULT_DO(nn::grcsrv::offscreen::FinishLayerRecording(layerHandle, applicationData, thumbnailImage));
        m_RecordingLayerHandle = nn::util::nullopt;

        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::GetOffscreenLayerError(std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_BoundLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == m_BoundLayerHandle->first, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());

        return nn::grcsrv::offscreen::GetLayerError(layerHandle);
    }

    nn::Result MovieMakerClient::EncodeOffscreenLayerAudioSample(nn::sf::Out<std::uint64_t> outEncodedSize, std::uint64_t layerHandle, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_RecordingLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == *m_RecordingLayerHandle, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());

        size_t outSize = 0;
        NN_RESULT_DO(nn::grcsrv::offscreen::EncodeLayerAudioSample(&outSize, layerHandle, buffer.GetPointerUnsafe(), buffer.GetSize()));

        outEncodedSize.Set(static_cast<uint64_t>(outSize));
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::GetOffscreenLayerRecordingFinishReadyEvent(nn::sf::Out<nn::sf::NativeHandle>& outHandle, std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_BoundLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == m_BoundLayerHandle->first, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        nn::sf::NativeHandle h;
        NN_RESULT_DO(nn::grcsrv::offscreen::GetLayerRecordingFinishReadyEvent(h, layerHandle));

        *outHandle = std::move(h);
        NN_RESULT_SUCCESS;
    }

    nn::Result MovieMakerClient::GetOffscreenLayerAudioEncodeReadyEvent(nn::sf::Out<nn::sf::NativeHandle>& outHandle, std::uint64_t layerHandle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        NN_RESULT_THROW_UNLESS(m_BoundLayerHandle != nn::util::nullopt, nn::grc::ResultInvalidCall());
        NN_RESULT_THROW_UNLESS(layerHandle == m_BoundLayerHandle->first, nn::grc::ResultInvalidCall());

        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        nn::sf::NativeHandle h;
        NN_RESULT_DO(nn::grcsrv::offscreen::GetLayerAudioEncodeReadyEvent(h, layerHandle));

        *outHandle = std::move(h);
        NN_RESULT_SUCCESS;
    }

    //-------------------------------------------------------------

    void MovieMakerClient::TransactParcel(std::int32_t handle, std::uint32_t code, const nn::sf::InBuffer& requestBuffer, const nn::sf::OutBuffer& replyBuffer, std::uint32_t flags) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        if(m_BoundLayerHandle == nn::util::nullopt)
        {
            return;
        }
        if(handle != m_BoundLayerHandle->second)
        {
            return;
        }
        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        nn::grcsrv::offscreen::TransactParcel(handle, code, requestBuffer, replyBuffer, flags);
    }

    void MovieMakerClient::AdjustRefcount(std::int32_t handle, std::int32_t diff, std::int32_t isStrong) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        if(m_BoundLayerHandle == nn::util::nullopt)
        {
            return;
        }
        if(handle != m_BoundLayerHandle->second)
        {
            return;
        }
        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        nn::grcsrv::offscreen::AdjustRefcount(handle, diff, isStrong);
    }

    void MovieMakerClient::GetNativeHandle(std::int32_t handle, std::uint32_t code, nn::sf::Out<nn::sf::NativeHandle> result) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsAlive);
        if(m_BoundLayerHandle == nn::util::nullopt)
        {
            return;
        }
        if(handle != m_BoundLayerHandle->second)
        {
            return;
        }
        NN_SDK_ASSERT(nn::grcsrv::offscreen::IsInitialized());
        nn::grcsrv::offscreen::GetNativeHandle(handle, code, result);
    }


}}

