﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "grcsrv_CapsrvMovieFileWriter.h"

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <utility>
#include <nn/grc/grc_Result.h>

#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/movie/capsrv_MovieMetaDataBuilder.h>
#include <nn/capsrv/movie/capsrv_MovieWriterFileSystem.h>
#include <mutex>
#include <memory>

namespace nn { namespace grcsrv {

Result CapsrvMovieFileWriter::Write(const capsrv::AlbumFileId& albumFileId, DataWriter* pDataWriter, capsrv::movie::MovieMetaDataBuilder* pMetaBuilder) NN_NOEXCEPT
{
    auto success = false;

    auto& metaBuilder = *pMetaBuilder;

    capsrv::AlbumMovieWriteStreamHandle handle;
    NN_RESULT_DO(capsrv::OpenAlbumMovieWriteStream(&handle, albumFileId));
    NN_UTIL_SCOPE_EXIT
    {
        if (!success)
        {
            capsrv::DiscardAlbumMovieWriteStream(handle);
        }
    };

    NN_RESULT_DO(capsrv::StartAlbumMovieWriteStreamDataSection(handle));

    {
        capsrv::movie::MovieWriterFileSystem mwfs;
        mwfs.Initialize("movie");
        NN_UTIL_SCOPE_EXIT
        {
            mwfs.Finalize();
        };

        const auto ChunkCacheCount = 3;
        auto mwfsBufferSize = mwfs.GetRequiredMemorySizeForFile(capsrv::movie::MovieDataChunkUnitSize, ChunkCacheCount);
        auto mwfsBuffer = std::unique_ptr<char[]>(new char[mwfsBufferSize]);
        NN_RESULT_THROW_UNLESS(mwfsBuffer, grc::ResultOutOfMemory());

        fs::FileHandle fsHandle;
        auto fsPriorityRaw = fs::GetPriorityRawOnCurrentThread();
        NN_RESULT_DO(mwfs.AttachMovieWriteStreamDataSection(&fsHandle, handle, capsrv::movie::MovieDataChunkUnitSize, ChunkCacheCount, mwfsBuffer.get(), mwfsBufferSize, fsPriorityRaw));
        NN_UTIL_SCOPE_EXIT
        {
            mwfs.DetachMovieWriteStreamDataSection(fsHandle);
        };

        NN_RESULT_DO(pDataWriter->WriteData(fsHandle));

        int64_t dataSize;
        NN_RESULT_DO(nn::fs::GetFileSize(&dataSize, fsHandle));
        // ファイルサイズを設定
        metaBuilder.SetMovieDataSize(static_cast<size_t>(dataSize));
    }

    NN_RESULT_DO(capsrv::EndAlbumMovieWriteStreamDataSection(handle));

    NN_RESULT_DO(capsrv::StartAlbumMovieWriteStreamMetaSection(handle));
    NN_RESULT_DO(metaBuilder.Build());
    NN_RESULT_DO(metaBuilder.WriteToStream(handle));
    NN_RESULT_DO(capsrv::EndAlbumMovieWriteStreamMetaSection(handle));

    NN_RESULT_DO(capsrv::FinishAlbumMovieWriteStream(handle));
    NN_RESULT_DO(capsrv::CommitAlbumMovieWriteStream(handle));

    success = true;
    NN_RESULT_SUCCESS;
}

}}
