﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdarg>
#include <filesystem>

#include <nn/cstd/cstd_CStdArg.h>
#if !defined( WIN32_LEAN_AND_MEAN )
    #define WIN32_LEAN_AND_MEAN
#endif
#if !defined( NOMINMAX )
    #define NOMINMAX
#endif
#include <nn/nn_Windows.h>

#include <nn/gfxTool/gfxTool_Util.h>
#include <nn/gfxTool/gfxTool_Error.h>
#include <nn/gfxTool/gfxTool_BackTrace.h>

namespace nn {
namespace gfxTool {

nngfxToolResultCode ReportException()
{
    try
    {
        throw;
    }
    catch( Exception& ex )
    {
        if( ex.GetResultCode() == nngfxToolResultCode_CompileError )
        {
            // 長いログが消えることがある(?)ことへの対策
            for( auto pChar = ex.what(); ; )
            {
                auto pLineEnd = strchr( pChar, '\n' );
                if( pLineEnd == nullptr )
                {
                    NN_GFXTOOL_PRINT_ERROR( pChar );
                    break;
                }
                else
                {
                    auto length = pLineEnd - pChar + 1;
                    char format[ 32 ];
                    sprintf( format, "%%.%ds", static_cast< int >( length ) );
                    NN_GFXTOOL_PRINT_ERROR( format, pChar );
                    pChar = pLineEnd + 1;
                }
            }
        }
        else
        {
            NN_GFXTOOL_PRINT_ERROR( ex.what() ); // TODO
        }
        return ex.GetResultCode();
    }
    catch( std::exception& ex )
    {
        NN_GFXTOOL_PRINT_ERROR( "Internal Error:\n\t" );
        NN_GFXTOOL_PRINT_ERROR( ex.what() ); // TODO
        return nngfxToolResultCode_InternalError;
    }
    catch( _EXCEPTION_POINTERS* ep )
    {
        NN_GFXTOOL_PRINT_ERROR( "Internal Error:\n" );
        auto pRecord = ep->ExceptionRecord;
        NN_GFXTOOL_PRINT_ERROR( "\tcode:%lx\n\tflag:%lx\n\taddr:%p\n\tparams:%lu\n", // TODO
            pRecord->ExceptionCode,
            pRecord->ExceptionFlags,
            pRecord->ExceptionAddress,
            pRecord->NumberParameters );

        for( int idxParam = 0, paramCount = NumericCastAuto( pRecord->NumberParameters );
            idxParam < paramCount; ++idxParam )
        {
            NN_GFXTOOL_PRINT_ERROR( "\t\tparam[ %d ]: %p\n", // TODO
                idxParam, reinterpret_cast< void* >( static_cast< uintptr_t >( pRecord->ExceptionInformation[ idxParam ] ) ) );
        }

        NN_GFXTOOL_PRINT_ERROR( "[%s] %s: %s\n",
            GetModuleNameFromAddress( pRecord->ExceptionAddress ).c_str(),
            GetLineFromAddress( pRecord->ExceptionAddress ).c_str(),
            GetNameFromAddress( pRecord->ExceptionAddress ).c_str() );

        return nngfxToolResultCode_InternalError;
    }
    catch( ... )
    {
        return nngfxToolResultCode_InternalError;
    }
}

Exception::Exception( const char* pFileName, int line,
    nngfxToolResultCode code, const char* pFormat, ... )
{
    va_list arg;
    va_start( arg, pFormat );
    m_Message = Format( pFormat, arg );
    va_end( arg );

    m_Line = line;
    m_Code = code;
#if _MSC_VER >= 1900
    m_FileName = std::experimental::filesystem::path( pFileName ).filename().string();
#else
    m_FileName = std::tr2::sys::path( pFileName ).filename();
#endif

    auto pCodeString = nngfxToolResultCodeToString( m_Code );
    static const char* prefix = "nngfxToolResultCode_";
    const char* pCode = pCodeString;
    if( strncmp( pCode, prefix, strlen( prefix ) ) == 0 )
    {
        pCode += strlen( prefix );
    }

    m_What = Format( "%s (%d):\n\t%s: %s\n",
        m_FileName.data(), m_Line, pCode, m_Message.data() );
}

Logger::LogLevel Logger::s_LogOutputLevels[ static_cast< int >( LogType::End ) ] =
{
    LogLevel::Medium,
    LogLevel::Medium,
    LogLevel::Medium
};

void* Logger::s_Streams[ static_cast< int >( LogType::End ) ] =
{
    nullptr,
    nullptr,
    nullptr
};

void Logger::PrintFormat( LogType type, LogLevel level, const char* pFormat, ... )
{
    if( s_LogOutputLevels[ static_cast< int >( type ) ] >= level &&
        s_Streams[ static_cast< int >( type ) ] )
    {
        va_list arg, arg2;
        va_start( arg, pFormat );
        NN_CSTD_VA_COPY( arg2, arg );
        auto length = vsnprintf( nullptr, 0, pFormat, arg );
        std::unique_ptr< void, decltype( &free ) > buffer( malloc( length + 1 ), free );
        vsnprintf( StaticCastAuto( buffer.get() ), length + 1, pFormat, arg2 );
        DWORD written;
        WriteFile( s_Streams[ static_cast< int >( type ) ], buffer.get(), length, &written, nullptr );
        va_end( arg );
        va_end( arg2 );
    }
}

void Logger::SetLogOutputLevel( LogType type, LogLevel level )
{
    s_LogOutputLevels[ static_cast< int >( type ) ] = level;
}

void* Logger::GetStream( LogType type )
{
    return s_Streams[ static_cast<int>( type ) ];
}

void Logger::SetStream( LogType type, void* stream )
{
    s_Streams[ static_cast<int>( type ) ] = stream;
}

void Logger::SetDefaultStream()
{
    SetStream( LogType::Default, GetStdHandle( STD_OUTPUT_HANDLE ) );
    SetStream( LogType::Warning, GetStdHandle( STD_OUTPUT_HANDLE ) );
    SetStream( LogType::Error, GetStdHandle( STD_ERROR_HANDLE ) );
}

}
}
