﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_TransientDescriptorAllocator.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>

namespace nn{ namespace gfx{ namespace util{

    TransientDescriptorAllocator::TransientDescriptorAllocator() NN_NOEXCEPT
        : m_pDescriptorPool(NULL)
        , m_SlotIndexRingBuffer()
        , m_pHistory(NULL)
        , m_HistoryMemorySize(0)
        , m_HistoryRingSize(0)
        , m_HistoryHead(0)
        , m_HistoryTail(0)
        , m_IsInitialized(false)
    {
    }

    void TransientDescriptorAllocator::Initialize(
        nn::gfx::DescriptorPool* pDescriptorPool,
        int baseSlotIndex,
        int slotCount,
        int historyLengthMax
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        NN_SDK_REQUIRES_NOT_NULL(pDescriptorPool);
        NN_SDK_REQUIRES_GREATER_EQUAL(baseSlotIndex, 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(slotCount, 1);
        NN_SDK_REQUIRES_LESS_EQUAL(slotCount, static_cast<int>(SlotCountMax));
        NN_SDK_REQUIRES_GREATER_EQUAL(historyLengthMax, 1);
        NN_SDK_REQUIRES_NOT_NULL(m_pHistory);
        NN_SDK_REQUIRES_ALIGNED(m_pHistory, GetRequiredMemoryAlignment());
        NN_SDK_REQUIRES_GREATER_EQUAL(m_HistoryMemorySize, GetRequiredMemorySize(historyLengthMax));

        m_pDescriptorPool = pDescriptorPool;
        m_SlotIndexRingBuffer.Initialize(baseSlotIndex, slotCount);
        m_HistoryRingSize = historyLengthMax + 1;
        m_HistoryHead = 0;
        m_HistoryTail = m_HistoryRingSize - 1;

        m_IsInitialized = true;
    }
    void TransientDescriptorAllocator::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        m_pDescriptorPool = NULL;
        m_SlotIndexRingBuffer.Finalize();
        m_IsInitialized = false;
    }

    bool TransientDescriptorAllocator::IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    bool TransientDescriptorAllocator::IsRecording() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_SlotIndexRingBuffer.IsRecording();
    }

    size_t TransientDescriptorAllocator::GetRequiredMemorySize(int historyLengthMax) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(historyLengthMax, 1);
        return sizeof(detail::IndexRange) * (static_cast<size_t>(historyLengthMax) + 1);
    }

    size_t TransientDescriptorAllocator::GetRequiredMemoryAlignment() NN_NOEXCEPT
    {
        return NN_ALIGNOF(detail::IndexRange);
    }

    void TransientDescriptorAllocator::SetMemory(void* pMemory, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!m_IsInitialized);
        m_pHistory = static_cast<detail::IndexRange*>(pMemory);
        m_HistoryMemorySize = size;
    }

    nn::gfx::DescriptorPool* TransientDescriptorAllocator::GetDescriptorPool() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_pDescriptorPool;
    }
    int TransientDescriptorAllocator::GetBaseSlotIndex() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_SlotIndexRingBuffer.GetBaseIndex();
    }
    int TransientDescriptorAllocator::GetSlotCount() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_SlotIndexRingBuffer.GetIndexCount();
    }

    void TransientDescriptorAllocator::Begin() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_LESS(GetHistoryLength(), GetHistoryLengthMax());
        m_SlotIndexRingBuffer.Begin();
    }

    void TransientDescriptorAllocator::End() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_ASSERT_LESS(GetHistoryLength(), GetHistoryLengthMax());
        m_SlotIndexRingBuffer.End(&m_pHistory[m_HistoryHead]);
        m_HistoryHead = (m_HistoryHead + 1) % m_HistoryRingSize;
    }

    void TransientDescriptorAllocator::Free() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_GREATER(GetHistoryLength(), 0);
        m_HistoryTail = (m_HistoryTail + 1) % m_HistoryRingSize;
        m_SlotIndexRingBuffer.ReleaseIndexRange(&m_pHistory[m_HistoryTail]);
    }

    int TransientDescriptorAllocator::Allocate() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_SlotIndexRingBuffer.AcquireIndex();
    }

    int TransientDescriptorAllocator::Allocate(int count) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);
        return m_SlotIndexRingBuffer.AcquireIndexRange(count);
    }

    int TransientDescriptorAllocator::GetHistoryLengthMax() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        return m_HistoryRingSize - 1;
    }

    int TransientDescriptorAllocator::GetHistoryLength() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        if(m_HistoryTail < m_HistoryHead)
        {
            return m_HistoryHead - m_HistoryTail - 1;
        }
        else
        {
            return m_HistoryRingSize - 1 - m_HistoryTail + m_HistoryHead;
        }
    }

    void TransientDescriptorAllocator::FillHistory() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsInitialized);
        NN_SDK_REQUIRES(!m_SlotIndexRingBuffer.IsRecording());
        while(GetHistoryLength() < GetHistoryLengthMax())
        {
            Begin();
            End();
        }
    }

}}}
