﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_TextureCompressor.h>

namespace nn {
namespace gfx {
namespace util {

#if NN_GFX_IS_TARGET_NVN

namespace {

    const int BlockSize = 4;
    const int InvalidIndex = -1;

    enum SupportedFormat
    {
        SupportedFormat_Bc1_Unorm,
        SupportedFormat_Bc1_UnormSrgb,
        SupportedFormat_Bc2_Unorm,
        SupportedFormat_Bc2_UnormSrgb,
        SupportedFormat_Bc3_Unorm,
        SupportedFormat_Bc3_UnormSrgb,
        SupportedFormat_Bc4_Unorm,
        SupportedFormat_Bc4_Snorm,
        SupportedFormat_Bc5_Unorm,
        SupportedFormat_Bc5_Snorm,
        SupportedFormat_End
    };
    NN_STATIC_ASSERT(SupportedFormat::SupportedFormat_End >= TextureCompressor::SupportedFormatCount);

    int GetSupportedFormatIndex(nn::gfx::ImageFormat format)
    {
        switch (format)
        {
        case nn::gfx::ImageFormat_Bc1_Unorm:
            return SupportedFormat_Bc1_Unorm;
        case nn::gfx::ImageFormat_Bc1_UnormSrgb:
            return SupportedFormat_Bc1_UnormSrgb;
        case nn::gfx::ImageFormat_Bc2_Unorm:
            return SupportedFormat_Bc2_Unorm;
        case nn::gfx::ImageFormat_Bc2_UnormSrgb:
            return SupportedFormat_Bc2_UnormSrgb;
        case nn::gfx::ImageFormat_Bc3_Unorm:
            return SupportedFormat_Bc3_Unorm;
        case nn::gfx::ImageFormat_Bc3_UnormSrgb:
            return SupportedFormat_Bc3_UnormSrgb;
        case nn::gfx::ImageFormat_Bc4_Unorm:
            return SupportedFormat_Bc4_Unorm;
        case nn::gfx::ImageFormat_Bc4_Snorm:
            return SupportedFormat_Bc4_Snorm;
        case nn::gfx::ImageFormat_Bc5_Unorm:
            return SupportedFormat_Bc5_Unorm;
        case nn::gfx::ImageFormat_Bc5_Snorm:
            return SupportedFormat_Bc5_Snorm;
        default:
            return InvalidIndex;
        }
    }

    nn::gfx::ImageFormat GetWorkTextureImageFormat(nn::gfx::ImageFormat format)
    {
        NN_SDK_ASSERT(TextureCompressor::IsSupportedFormat(format));

        switch(format)
        {
            case nn::gfx::ImageFormat_Bc1_Unorm:
            case nn::gfx::ImageFormat_Bc1_UnormSrgb:
            case nn::gfx::ImageFormat_Bc4_Unorm:
            case nn::gfx::ImageFormat_Bc4_Snorm:
                return nn::gfx::ImageFormat_R16_G16_B16_A16_Uint;

            case nn::gfx::ImageFormat_Bc2_Unorm:
            case nn::gfx::ImageFormat_Bc2_UnormSrgb:
            case nn::gfx::ImageFormat_Bc3_Unorm:
            case nn::gfx::ImageFormat_Bc3_UnormSrgb:
            case nn::gfx::ImageFormat_Bc5_Unorm:
            case nn::gfx::ImageFormat_Bc5_Snorm:
                return nn::gfx::ImageFormat_R32_G32_B32_A32_Uint;

            default:
                NN_UNEXPECTED_DEFAULT;
        }
    }
}

TextureCompressor::TextureCompressor() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!IsInitialized());
}

TextureCompressor::~TextureCompressor() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
}

void TextureCompressor::Initialize(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(!IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pDevice);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pDevice));

    memset(m_pShaders,0,sizeof(m_pShaders));
    m_CommonState.Initialize(pDevice);
    NN_SDK_ASSERT(m_CommonState.IsInitialized());
}

void TextureCompressor::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    m_CommonState.Finalize();
}

bool TextureCompressor::IsInitialized() const NN_NOEXCEPT
{
    return m_CommonState.IsInitialized();
}

void TextureCompressor::RegisterCompressionShader(nn::gfx::ImageFormat targetFormat,const nn::gfx::Shader* pShader) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsSupportedFormat(targetFormat));
    NN_SDK_REQUIRES_NOT_NULL(pShader);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pShader));

    int index = GetSupportedFormatIndex(targetFormat);
    m_pShaders[index] = pShader;
}

void TextureCompressor::UnregisterCompressionShader(nn::gfx::ImageFormat targetFormat) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES(IsSupportedFormat(targetFormat));
    int index = GetSupportedFormatIndex(targetFormat);
    m_pShaders[index] = NULL;
}

bool TextureCompressor::IsSupportedFormat(nn::gfx::ImageFormat format) NN_NOEXCEPT
{
    return GetSupportedFormatIndex(format) != InvalidIndex;
}

const nn::gfx::Shader* TextureCompressor::GetShader(nn::gfx::ImageFormat imageFormat) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CommonState.IsInitialized());
    NN_SDK_REQUIRES(IsSupportedFormat(imageFormat));

    int index = GetSupportedFormatIndex(imageFormat);
    return m_pShaders[index];
}

void TextureCompressor::MakeCommand(
    nn::gfx::CommandBuffer*             pCommandBuffer,
    const nn::gfx::DescriptorSlot&      srcTextureDescriptorSlot,
    const nn::gfx::DescriptorSlot&      samplerDescriptorSlot,
    const TextureCompressorTargetInfo&  targetTextureInfo
    ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pCommandBuffer);
    NN_SDK_REQUIRES(nn::gfx::IsInitialized(*pCommandBuffer));

    const nn::gfx::Texture::InfoType textureInfo = targetTextureInfo.GetTextureInfo();
    NN_SDK_REQUIRES(IsSupportedFormat(textureInfo.GetImageFormat()));
    NN_SDK_REQUIRES_RANGE(textureInfo.GetArrayLength(),0,0x10000);
    NN_SDK_REQUIRES_RANGE(textureInfo.GetMipCount(),0,0x10);
    NN_SDK_REQUIRES_NOT_NULL(GetShader(textureInfo.GetImageFormat()));


    //各種ステートを書き込む
    m_CommonState.MakeCommand(pCommandBuffer,srcTextureDescriptorSlot,samplerDescriptorSlot);

    // シェーダを選択する
    const nn::gfx::Shader* pShader = GetShader(textureInfo.GetImageFormat());
    pCommandBuffer->SetShader(pShader,nn::gfx::ShaderStageBit_All);

    nn::gfx::Device* pDevice = m_CommonState.GetDevice();

    int mipCount = textureInfo.GetMipCount();
    int arraySize = (textureInfo.GetArrayLength() == 0) ? 1 : textureInfo.GetArrayLength();

    for (int layer = 0; layer < arraySize; layer++)
    {
        for (int miplevel = 0; miplevel < mipCount; miplevel++)
        {
            // mip に応じたサイズを算出
            int width = std::max(textureInfo.GetWidth() >> miplevel,1);
            int height = std::max(textureInfo.GetHeight() >> miplevel,1);

            // BC1 のブロックサイズに応じたサイズを算出
            int compressedwidth = std::max((width + BlockSize - 1) / BlockSize,1);
            int compressedheight = std::max((height + BlockSize - 1) / BlockSize,1);

            nn::gfx::ImageFormat workImageFormat = GetWorkTextureImageFormat(textureInfo.GetImageFormat());

            // BC1 圧縮先に書き込むためのビューを初期化
            nn::gfx::ColorTargetView colorTargetView;
            {
                nn::gfx::ColorTargetView::InfoType info;
                info.SetDefault();
                info.SetImageFormat(workImageFormat);
                info.SetTexturePtr(targetTextureInfo.GetTexture());
                info.SetMipLevel(miplevel);
                info.EditArrayRange().SetBaseArrayIndex(layer);

                colorTargetView.Initialize(pDevice,info);
            }

            // ビューポートシザーステートを作成
            nn::gfx::ViewportScissorState viewportScissorState;
            {
                nn::gfx::ViewportScissorState::InfoType info;

                // Viewport State
                nn::gfx::ViewportStateInfo viewportInfo;
                {
                    viewportInfo.SetDefault();
                    viewportInfo.SetWidth(static_cast<float>(compressedwidth));
                    viewportInfo.SetHeight(static_cast<float>(compressedheight));
                }

                // Scissor State
                nn::gfx::ScissorStateInfo scissorInfo;
                {
                    scissorInfo.SetDefault();
                    scissorInfo.SetWidth(compressedwidth);
                    scissorInfo.SetHeight(compressedheight);
                }

                {
                    info.SetDefault();
                    info.SetScissorEnabled(true);
                    info.SetViewportStateInfoArray(&viewportInfo, 1);
                    info.SetScissorStateInfoArray(&scissorInfo, 1);
                }

               viewportScissorState.Initialize(pDevice, info);
            }

            //レンダーターゲットを切り替え
            nn::gfx::ColorTargetView* pColorTargetView = &colorTargetView;
            pCommandBuffer->SetRenderTargets(1,&pColorTargetView,NULL);
            pCommandBuffer->SetViewportScissorState(&viewportScissorState);

            // miplevel: 4bit layer: 16bit
            int texInfo = layer & 0xFFFF;
            texInfo |= ((miplevel & 0xF) << 16);
            texInfo <<= 2;

            //描画
            pCommandBuffer->Draw(nn::gfx::PrimitiveTopology_TriangleStrip,4,texInfo,1,0);

            //gfxオブジェクトを破棄
            viewportScissorState.Finalize(pDevice);
            colorTargetView.Finalize(pDevice);
        }
    }
} // NOLINT(impl/function_size)

#endif //NN_GFX_IS_TARGET_NVN

}
}
}
