﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gfx/util/gfx_ScalableViewport.h>

namespace nn { namespace gfx { namespace util {

    bool ScalableViewport::WindowCoordinate::IsValid() const NN_NOEXCEPT
    {
        if (m_Width <= .0f)
        {
            return false;
        }

        if (m_Height <= .0f)
        {
            return false;
        }

        if (m_OriginMode < OriginMode_LowerLeft || m_OriginMode >= OriginMode_Count)
        {
            return false;
        }

        return true;
    }

    void ScalableViewport::Initialize(const WindowCoordinate& virtualWindowCoordinate, const WindowCoordinate& physicalWindowCoordinate) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(!IsInitialized());
        NN_SDK_REQUIRES(virtualWindowCoordinate.IsValid());
        NN_SDK_REQUIRES(physicalWindowCoordinate.IsValid());

        m_VirtualWindow = virtualWindowCoordinate;
        m_PhysicalWindow = physicalWindowCoordinate;
        m_IsInitialized = true;
    }

    void ScalableViewport::Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        m_IsInitialized = false;
    }

    void ScalableViewport::SetVirtualWindowCoordinate(const WindowCoordinate& virtualWindow) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(virtualWindow.IsValid());
        m_VirtualWindow = virtualWindow;
    }

    void ScalableViewport::SetPhysicalWindowCoordinate(const WindowCoordinate& physicalWindow) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES(physicalWindow.IsValid());
        m_PhysicalWindow = physicalWindow;
    }

    void ScalableViewport::ConvertPointVirtualToPhysical(float* pOutPhysicalX, float* pOutPhysicalY, float virtualX, float virtualY) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        // virtual 空間で 0 ~ 1 の値の間に変換する
        float scaleX = virtualX / m_VirtualWindow.GetWidth();
        float scaleY = virtualY / m_VirtualWindow.GetHeight();

        if (m_PhysicalWindow.GetOriginMode() == m_VirtualWindow.GetOriginMode())
        {
            *pOutPhysicalX = m_PhysicalWindow.GetWidth() * scaleX;
            *pOutPhysicalY = m_PhysicalWindow.GetHeight() * scaleY;
        }
        // 原点が異なる
        else
        {
            *pOutPhysicalX = m_PhysicalWindow.GetWidth() * scaleX;
            *pOutPhysicalY = m_PhysicalWindow.GetHeight() * (1.0f - scaleY);
        }
    }

    void ScalableViewport::ConvertLengthVirtualToPhysical(float* pOutPhysicalWidth, float* pOutPhysicalHeight, float virtualWidth, float virtualHeight) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        // virtual -> physical の割合を求める
        float scaleX = m_PhysicalWindow.GetWidth() / m_VirtualWindow.GetWidth();
        float scaleY = m_PhysicalWindow.GetHeight() / m_VirtualWindow.GetHeight();

        *pOutPhysicalWidth = virtualWidth * scaleX;
        *pOutPhysicalHeight = virtualHeight * scaleY;
    }

    void ScalableViewport::ConvertPointPhysicalToVirtual(float* pOutVirtualX, float* pOutVirtualY, float physicalX, float physicalY) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        // physical 空間で 0 ~ 1 の値の間に変換する
        float scaleX = physicalX / m_PhysicalWindow.GetWidth();
        float scaleY = physicalY / m_PhysicalWindow.GetHeight();

        if (m_PhysicalWindow.GetOriginMode() == m_VirtualWindow.GetOriginMode())
        {
            *pOutVirtualX = m_VirtualWindow.GetWidth() * scaleX;
            *pOutVirtualY = m_VirtualWindow.GetHeight() * scaleY;
        }
        else
        {
            *pOutVirtualX = m_VirtualWindow.GetWidth() * scaleX;
            *pOutVirtualY = m_VirtualWindow.GetHeight() * (1.0f - scaleY);
        }
    }

    void ScalableViewport::ConvertLengthPhysicalToVirtual(float* pOutVirtualWidth, float* pOutVirtualHeight, float physicalWidth, float physicalHeight) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());

        // physical -> virtual の割合を求める
        float scaleX = m_VirtualWindow.GetWidth() / m_PhysicalWindow.GetWidth();
        float scaleY = m_VirtualWindow.GetHeight() / m_PhysicalWindow.GetHeight();

        *pOutVirtualWidth = physicalWidth * scaleX;
        *pOutVirtualHeight = physicalHeight * scaleY;
    }

    void ScalableViewport::ConvertRectVirtualToPhysical(Rect* pOutPhysicalRect, const Rect& virtualRect) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutPhysicalRect);
        ConvertPointVirtualToPhysical(&pOutPhysicalRect->originX, &pOutPhysicalRect->originY, virtualRect.originX, virtualRect.originY);
        ConvertLengthVirtualToPhysical(&pOutPhysicalRect->width, &pOutPhysicalRect->height, virtualRect.width, virtualRect.height);

        // 原点が異なる場合は、高さをオフセット. (原点を上下で切り替える)
        if (m_PhysicalWindow.GetOriginMode() != m_VirtualWindow.GetOriginMode())
        {
            pOutPhysicalRect->originY -= pOutPhysicalRect->height;
        }
    }

    void ScalableViewport::ConvertRectPhysicalToVirtual(Rect* pOutVirtualRect, const Rect& physicalRect) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsInitialized());
        NN_SDK_REQUIRES_NOT_NULL(pOutVirtualRect);
        ConvertPointPhysicalToVirtual(&pOutVirtualRect->originX, &pOutVirtualRect->originY, physicalRect.originX, physicalRect.originY);
        ConvertLengthPhysicalToVirtual(&pOutVirtualRect->width, &pOutVirtualRect->height, physicalRect.width, physicalRect.height);

        // 原点が異なる場合は、高さをオフセット. (原点を上下で切り替える)
        if (m_PhysicalWindow.GetOriginMode() != m_VirtualWindow.GetOriginMode())
        {
            pOutVirtualRect->originY -= pOutVirtualRect->height;
        }
    }

    void ScalableViewport::SetupViewportStateInfo(nn::gfx::ViewportStateInfo* pOutViewportStateInfo, const Rect& physicalRect) const NN_NOEXCEPT
    {
        pOutViewportStateInfo->SetOriginX(physicalRect.originX);
        pOutViewportStateInfo->SetOriginY(physicalRect.originY);
        pOutViewportStateInfo->SetWidth(physicalRect.width);
        pOutViewportStateInfo->SetHeight(physicalRect.height);
    }

    void ScalableViewport::SetupScissorStateInfo(nn::gfx::ScissorStateInfo* pOutScissorStateInfo, const Rect& physicalRect) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(physicalRect.originX), 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(physicalRect.originY), 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(physicalRect.width), 0);
        NN_SDK_REQUIRES_GREATER_EQUAL(static_cast<int>(physicalRect.height), 0);

        pOutScissorStateInfo->SetOriginX(static_cast<int>(physicalRect.originX + .5f));
        pOutScissorStateInfo->SetOriginY(static_cast<int>(physicalRect.originY + .5f));
        pOutScissorStateInfo->SetWidth(static_cast<int>(physicalRect.width + .5f));
        pOutScissorStateInfo->SetHeight(static_cast<int>(physicalRect.height + .5f));
    }

    void ScalableViewport::SetupRectFromViewportStateInfo(Rect* pOutRect, const nn::gfx::ViewportStateInfo& viewportStateInfo) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutRect);

        pOutRect->originX = viewportStateInfo.GetOriginX();
        pOutRect->originY = viewportStateInfo.GetOriginY();
        pOutRect->width = viewportStateInfo.GetWidth();
        pOutRect->height = viewportStateInfo.GetHeight();
    }

    void ScalableViewport::SetupRectFromScissorStateInfo(Rect* pOutRect, const nn::gfx::ScissorStateInfo& scissorStateInfo) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutRect);

        pOutRect->originX = static_cast<float>(scissorStateInfo.GetOriginX());
        pOutRect->originY = static_cast<float>(scissorStateInfo.GetOriginY());
        pOutRect->width = static_cast<float>(scissorStateInfo.GetWidth());
        pOutRect->height = static_cast<float>(scissorStateInfo.GetHeight());
    }

} } }
