﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_StaticAssert.h>
#include <nn/gfx/util/detail/gfx_DebugFontGpuBuffer.h>

#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
#include <cafe/gx2.h> // GX2EndianSwap
#endif

namespace nn {
namespace gfx {
namespace util {
namespace detail {

    inline bool IsAligned(const void* address, int alignment)
    {
        return (reinterpret_cast< uintptr_t >(address)& (alignment - 1)) == 0;
    }

    inline bool IsAligned(const size_t offset, size_t alignment)
    {
        return (offset & (alignment - 1)) == 0;
    }

//----------------------------------------
bool GpuBuffer::Initialize(
    nn::gfx::Device* pDevice,
    const InitializeArg& arg
    )
{
    NN_SDK_ASSERT_NOT_NULL(pDevice);
    NN_SDK_ASSERT(arg.m_BufferSize > 0);
    NN_SDK_ASSERT(arg.m_BufferCount >= 1);
    NN_SDK_ASSERT_NOT_NULL(arg.m_pAllocateFunction);
    NN_SDK_ASSERT_NOT_NULL(arg.m_pMemoryPool);

    m_Size = arg.m_BufferSize;
    m_Count = arg.m_BufferCount;

    // nn::gfx::Buffer 用のメモリをバッファ数に応じて確保する
    m_pBuffers = static_cast<nn::gfx::Buffer*>((*arg.m_pAllocateFunction)(sizeof(nn::gfx::Buffer) * m_Count, 4, arg.m_pAllocateFunctionUserData));

    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags(arg.m_GpuAccessFlag);
    info.SetSize(m_Size);

    for(uint32_t index = 0; index < m_Count;++index)
    {
        // nn::gfx::Buffer のコンストラクタ呼び出し
        new (&m_pBuffers[index]) nn::gfx::Buffer();
        m_pBuffers[index].Initialize(pDevice, info, arg.m_pMemoryPool, arg.m_MemoryPoolOffset + m_Size * index, m_Size);
    }

    // バッファのアライメント値を保存
    m_Alignment = nn::gfx::Buffer::GetBufferAlignment(pDevice, info);

    // 渡されたメモリプールオフセットのアライメントチェック
    NN_SDK_ASSERT(IsAligned(arg.m_MemoryPoolOffset, m_Alignment));

    if (arg.m_AllocateSyncFlag)
    {
        m_Flags |= Flags_AllocateWithSync;
#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
        NN_SDK_ASSERT(false, "Syncrounous allocation is not supported on Cafe platform.");
#else
        m_BufferAllocatedSize.m_pSync = static_cast<std::atomic<size_t>*>((*arg.m_pAllocateFunction)(sizeof(std::atomic<size_t>), 4, arg.m_pAllocateFunctionUserData));
#endif
    }

    return true;
}

bool GpuBuffer::Initialize(
    nn::gfx::Device* pDevice,
    nn::gfx::GpuAccess gpuAccess,
    size_t bufferSize,
    int bufferCount,
    nn::gfx::MemoryPool* pMemoryPool,
    size_t memoryPoolOffset,
    nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
    void* pAllocateFunctionUserData
    )
{
    InitializeArg   arg;

    arg.SetGpuAccessFlag(gpuAccess);
    arg.SetBufferSize(bufferSize);
    arg.SetBufferCount(bufferCount);
    arg.SetMemoryPool(pMemoryPool);
    arg.SetMemoryPoolOffset(memoryPoolOffset);
    arg.SetAllocator(pAllocateFunction, pAllocateFunctionUserData);

    return Initialize(pDevice, arg);
}

//----------------------------------------
void GpuBuffer::Finalize(
    nn::gfx::Device* pDevice,
    nn::FreeFunctionWithUserData pFreeFunction,
    void* pFreeFunctionUserData)
{
    NN_UNUSED(pDevice);

    if (m_Flags & Flags_AllocateWithSync)
    {
        (*pFreeFunction)(m_BufferAllocatedSize.m_pSync, pFreeFunctionUserData);
    }

    for(uint32_t index = 0; index < m_Count;++index)
    {
        m_pBuffers[index].Finalize(pDevice);
    }

    // バッファを開放する
    (*pFreeFunction)(m_pBuffers, pFreeFunctionUserData);

    // 再利用に備えてすべてのパラメータをクリアする。
    m_pBuffers = NULL;
    m_Size = 0;
    m_Alignment = 0;
    m_Count = 0;
    m_MappedBufferIndex = -1;
    m_GpuAccessBufferIndex = 0;
    m_pMappedPointer = NULL;
}

//----------------------------------------
void GpuBuffer::Map(int bufferIndex)
{
    if(m_MappedBufferIndex >= 0)
    {
        return ;
    }

    m_MappedBufferIndex = bufferIndex;

    NN_SDK_ASSERT_NOT_NULL(m_pBuffers);

    m_pMappedPointer = m_pBuffers[m_MappedBufferIndex].Map();

    if (m_Flags & Flags_AllocateWithSync)
    {
        *(m_BufferAllocatedSize.m_pSync) = 0;
    }
    else
    {
        m_BufferAllocatedSize.m_Raw = 0;
    }
}

//----------------------------------------
void GpuBuffer::Unmap()
{
    NN_SDK_ASSERT_NOT_NULL(m_pBuffers);

    if(m_MappedBufferIndex < 0)
    {
        return ;
    }

    size_t  bufferAllocatedSize = m_BufferAllocatedSize.m_Raw;

    if (m_Flags & Flags_AllocateWithSync)
    {
        bufferAllocatedSize = *(m_BufferAllocatedSize.m_pSync);
    }

#if defined(NN_BUILD_CONFIG_SPEC_CAFE)
    GX2EndianSwap(m_pMappedPointer, bufferAllocatedSize);
#endif

    m_pBuffers[m_MappedBufferIndex].FlushMappedRange(0, bufferAllocatedSize);
    m_pBuffers[m_MappedBufferIndex].Unmap();

    m_pMappedPointer = NULL;
    m_MappedBufferIndex = -1;

    if (m_Flags & Flags_AllocateWithSync)
    {
        *(m_BufferAllocatedSize.m_pSync) = 0;
    }
    else
    {
        m_BufferAllocatedSize.m_Raw = 0;
    }
}

}   // namespace detail
}   // namespace util
}   // namespace gfx
}   // namespace nn
