﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_TextureInfo.h>

#include <nn/gfx/detail/gfx_Texture-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

namespace
{

void SetupTextureBuilder( NVNtextureBuilder *pBuilder, NVNdevice *pDevice,
    const TextureImpl< Target >::InfoType& info ) NN_NOEXCEPT
{
    ImageStorageDimension imageStorageDimension = info.GetImageStorageDimension();
    NN_SDK_ASSERT_NOT_EQUAL( imageStorageDimension, ImageStorageDimension_Undefined );
    int arrayLength = info.GetArrayLength();
    NVNtextureTarget target = Nvn::GetImageTarget( GetImageDimension(
        imageStorageDimension, arrayLength > 1, info.GetMultisampleCount() > 1 ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetDefaults( pBuilder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetDevice( pBuilder, pDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetWidth( pBuilder, info.GetWidth() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetHeight(
        pBuilder, target == NVN_TEXTURE_TARGET_1D_ARRAY ? arrayLength : info.GetHeight() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetDepth(
        pBuilder, ( target == NVN_TEXTURE_TARGET_2D_ARRAY ||
        target == NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY ) ? arrayLength : info.GetDepth() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetLevels( pBuilder, info.GetMipCount() ) );
    if ( info.GetMultisampleCount() > 1 )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetSamples( pBuilder, info.GetMultisampleCount() ) );
    }
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetTarget( pBuilder, target ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetFormat(
        pBuilder, Nvn::GetImageFormat( info.GetImageFormat() ) ) );

    int flags = NVN_TEXTURE_FLAGS_IMAGE_BIT;
    flags |= (info.GetGpuAccessFlags() & (GpuAccess_ColorBuffer | GpuAccess_DepthStencil))
        && info.GetTileMode() != nn::gfx::TileMode_Linear ? NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT : 0;
    flags |= ( info.GetGpuAccessFlags() & GpuAccess_ScanBuffer ) ? NVN_TEXTURE_FLAGS_DISPLAY_BIT : 0;
    // TODO: 機能の有無のチェック
    flags |= info.ToData()->flags.GetBit( TextureInfo::DataType::Flag_SparseResidency )
        ? NVN_TEXTURE_FLAGS_SPARSE_BIT : 0;
    if( info.GetTileMode() == TileMode_Linear )
    {
        NVNdeviceInfo strideAlignmentInfo;
        if( info.GetGpuAccessFlags() & ( GpuAccess_ColorBuffer | GpuAccess_DepthStencil ) )
        {
            strideAlignmentInfo = NVN_DEVICE_INFO_LINEAR_RENDER_TARGET_STRIDE_ALIGNMENT;
            flags |= NVN_TEXTURE_FLAGS_LINEAR_RENDER_TARGET_BIT;
        }
        else
        {
            strideAlignmentInfo = NVN_DEVICE_INFO_LINEAR_TEXTURE_STRIDE_ALIGNMENT;
            flags |= NVN_TEXTURE_FLAGS_LINEAR_BIT;
        }
        int strideAlignment;
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger(
            pDevice, strideAlignmentInfo, &strideAlignment ) );
        ptrdiff_t stride = static_cast< ptrdiff_t >( nn::util::align_up( CalculateRowSize(
            info.GetWidth(), GetChannelFormat( info.GetImageFormat() ) ), strideAlignment ) );
        NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetStride( pBuilder, stride ) );
    }
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetFlags( pBuilder, flags ) );

    NN_STATIC_ASSERT( sizeof( info.ToData()->textureLayout ) == sizeof( NVNpackagedTextureLayout ) );
    if( info.ToData()->flags.GetBit( TextureInfo::DataType::Flag_SpecifyTextureLayout ) )
    {
        const NVNpackagedTextureLayout* pLayout = reinterpret_cast<
            const NVNpackagedTextureLayout* >( info.ToData()->textureLayout );
        NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetPackagedTextureLayout( pBuilder, pLayout ) );
    }
}

size_t CalculateMipDataOffsetSize( ptrdiff_t* pMipOffsets, const TextureInfo& info ) NN_NOEXCEPT
{
    ImageStorageDimension imageStorageDimension = info.GetImageStorageDimension();
    NN_SDK_ASSERT_NOT_EQUAL( imageStorageDimension, ImageStorageDimension_Undefined );
    int arrayLength = info.GetArrayLength();

    int width = info.GetWidth();
    int height = ( imageStorageDimension == ImageStorageDimension_1d
        && arrayLength > 1 ) ? arrayLength : info.GetHeight();
    int depth = ( imageStorageDimension == ImageStorageDimension_2d
        && arrayLength > 1 ) ? arrayLength : info.GetDepth();

    int minHeight = 1;
    int minDepth = 1;
    if( arrayLength > 1 )
    {
        if( imageStorageDimension == ImageStorageDimension_1d )
        {
            minHeight = height;
        }
        else if( imageStorageDimension == ImageStorageDimension_2d )
        {
            minDepth = depth;
        }
    }

    ChannelFormat channelFormat = GetChannelFormat( info.GetImageFormat() );
    size_t size = 0;
    for ( size_t mipLevel = 0; mipLevel < static_cast< size_t >( info.GetMipCount() ); ++mipLevel )
    {
        if ( pMipOffsets )
        {
            pMipOffsets[ mipLevel ] = size;
        }
        size += CalculateImageSize( channelFormat,
            std::max NN_PREVENT_MACRO_FUNC( width >> mipLevel, 1 ),
            std::max NN_PREVENT_MACRO_FUNC( height >> mipLevel, minHeight ),
            std::max NN_PREVENT_MACRO_FUNC( depth >> mipLevel, minDepth ) );
    }
    return size;
}

}

size_t TextureImpl< Target >::CalculateMipDataAlignment(
    DeviceImpl< Target > *pNnDevice, const InfoType& info ) NN_NOEXCEPT
{
    NVNtextureBuilder builder;
    SetupTextureBuilder( &builder, pNnDevice->ToData()->pNvnDevice, info );
    return NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderGetStorageAlignment( &builder ) );
}

size_t TextureImpl< Target >::CalculateMipDataSize(
    DeviceImpl< Target > *pNnDevice, const InfoType& info ) NN_NOEXCEPT
{
    NVNtextureBuilder builder;
    SetupTextureBuilder( &builder, pNnDevice->ToData()->pNvnDevice, info );
    return NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderGetStorageSize( &builder ) );
}

void TextureImpl< Target >::CalculateMipDataOffsets(
    ptrdiff_t* pMipOffsets, DeviceImpl< Target > *pNnDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pMipOffsets );
    NN_UNUSED( pNnDevice );

    // Not sure how this applies to NVN
    CalculateMipDataOffsetSize( pMipOffsets, info );
}

size_t TextureImpl< Target >::GetRowPitch( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );

    int stride;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice->ToData()->pNvnDevice,
        ( info.GetGpuAccessFlags() & ( GpuAccess_ColorBuffer | GpuAccess_DepthStencil ) ) ?
        NVN_DEVICE_INFO_LINEAR_RENDER_TARGET_STRIDE_ALIGNMENT :
        NVN_DEVICE_INFO_LINEAR_TEXTURE_STRIDE_ALIGNMENT, &stride ) );
    size_t row = CalculateRowSize( info.GetWidth(), GetChannelFormat( info.GetImageFormat() ) );
    return nn::util::align_up( row, stride );
}

TextureImpl< Target >::TextureImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureImpl< Target >::~TextureImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT_ALIGNED( memoryPoolOffset, CalculateMipDataAlignment( pDevice, info ) );
    NN_SDK_ASSERT_NOT_NULL( pMemoryPool );
    NN_STATIC_ASSERT( sizeof( this->nvnTexture ) == sizeof( NVNtexture ) );

    NVNtextureBuilder textureBuilder;

    SetupTextureBuilder( &textureBuilder, pDevice->ToData()->pNvnDevice, info );

    NN_SDK_ASSERT( info.GetTileMode() == TileMode_Optimal || NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureBuilderGetTarget( &textureBuilder ) ) == NVN_TEXTURE_TARGET_2D );

    NN_SDK_ASSERT( nvnTextureBuilderGetStorageSize( &textureBuilder ) <= memoryPoolSize );

    // Attach the memory pool to the texture
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureBuilderSetStorage(
        &textureBuilder, pMemoryPool->ToData()->pNvnMemoryPool, memoryPoolOffset ) );

    Nvn::SetPackagedTextureDataImpl( &textureBuilder, info, pMemoryPool, memoryPoolOffset, memoryPoolSize );

    // Create a pointer to the static NVNtexture storage
    this->pNvnTexture = &nvnTexture;

    NVNboolean isTextureOK = NN_GFX_CALL_NVN_FUNCTION(
        nvnTextureInitialize( this->pNvnTexture, &textureBuilder ) );
    NN_SDK_ASSERT( isTextureOK );
    NN_UNUSED( isTextureOK );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void TextureImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    NN_GFX_CALL_NVN_FUNCTION( nvnTextureFinalize( this->pNvnTexture ) );

    this->pNvnTexture = NULL;

    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::TextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::~TextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureViewImpl< Target >::Initialize( DeviceImpl< Target >* , const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnTextureView ) == sizeof( NVNtextureView ) );

    const TextureImpl< Target >* pSourceTexture = info.GetTexturePtr();
    NN_SDK_REQUIRES_NOT_NULL( pSourceTexture );
    NN_SDK_ASSERT_NOT_NULL( pSourceTexture->ToData()->pNvnTexture.ptr );
    this->pNvnTexture = pSourceTexture->ToData()->pNvnTexture;

    // Create another view from the original pool. This allows us to change the
    // depth component rules
    this->pNvnTextureView = &this->nvnTextureView;

    static const NVNtextureSwizzle s_ChannelMappingTable[] =
    {
        NVN_TEXTURE_SWIZZLE_ZERO,
        NVN_TEXTURE_SWIZZLE_ONE,
        NVN_TEXTURE_SWIZZLE_R,
        NVN_TEXTURE_SWIZZLE_G,
        NVN_TEXTURE_SWIZZLE_B,
        NVN_TEXTURE_SWIZZLE_A,
    };

    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( this->pNvnTextureView ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels( this->pNvnTextureView,
        info.GetSubresourceRange().GetMipRange().GetMinMipLevel(),
        info.GetSubresourceRange().GetMipRange().GetMipCount() ) );
    if( info.GetImageDimension() != nn::gfx::ImageDimension_3d )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLayers( this->pNvnTextureView,
            info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex(),
            info.GetSubresourceRange().GetArrayRange().GetArrayLength() ) );
    }
    Nvn::SetTextureViewFormat( this->pNvnTextureView,
        Nvn::GetImageFormat( info.GetImageFormat() ), this->pNvnTexture );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetSwizzle( this->pNvnTextureView,
        s_ChannelMappingTable[ info.GetChannelMapping( ColorChannel_Red ) ],
        s_ChannelMappingTable[ info.GetChannelMapping( ColorChannel_Green ) ],
        s_ChannelMappingTable[ info.GetChannelMapping( ColorChannel_Blue ) ],
        s_ChannelMappingTable[ info.GetChannelMapping( ColorChannel_Alpha ) ] ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDepthStencilMode( this->pNvnTextureView,
        info.GetDepthStencilTextureMode() == DepthStencilFetchMode_DepthComponent ?
        NVN_TEXTURE_DEPTH_STENCIL_MODE_DEPTH : NVN_TEXTURE_DEPTH_STENCIL_MODE_STENCIL ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetTarget( this->pNvnTextureView,
        Nvn::GetImageTarget( info.GetImageDimension() ) ) );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void TextureViewImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    this->pNvnTexture = NULL;
    this->pNvnTextureView = NULL;

    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::ColorTargetViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::~ColorTargetViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void ColorTargetViewImpl< Target >::Initialize( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnTextureView ) == sizeof( NVNtextureView ) );

    const TextureImpl< Target >* pSourceTexture = info.GetTexturePtr();

    NN_SDK_ASSERT_NOT_NULL( pSourceTexture->ToData()->pNvnTexture.ptr );
    this->pNvnTexture = pSourceTexture->ToData()->pNvnTexture;

    this->pNvnTextureView = &this->nvnTextureView;

    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( this->pNvnTextureView ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels( this->pNvnTextureView, info.GetMipLevel(), 1 ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLayers( this->pNvnTextureView,
        info.GetArrayRange().GetBaseArrayIndex(), info.GetArrayRange().GetArrayLength() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetFormat( this->pNvnTextureView,
        Nvn::GetImageFormat( info.GetImageFormat() ) ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetTarget( this->pNvnTextureView,
        Nvn::GetImageTarget( info.GetImageDimension() ) ) );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void ColorTargetViewImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    this->pNvnTexture = NULL;
    this->pNvnTextureView = NULL;

    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::DepthStencilViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::~DepthStencilViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void DepthStencilViewImpl< Target >::Initialize( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnTextureView ) == sizeof( NVNtextureView ) );

    const TextureImpl< Target >* pSourceTexture = info.GetTexturePtr();
    NN_SDK_REQUIRES_NOT_NULL( pSourceTexture );
    NN_SDK_ASSERT_NOT_NULL( pSourceTexture->ToData()->pNvnTexture.ptr );
    this->pNvnTexture = pSourceTexture->ToData()->pNvnTexture;

    this->pNvnTextureView = &this->nvnTextureView;

    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetDefaults( this->pNvnTextureView ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLevels( this->pNvnTextureView, info.GetMipLevel(), 1 ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetLayers( this->pNvnTextureView,
        info.GetArrayRange().GetBaseArrayIndex(), info.GetArrayRange().GetArrayLength() ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnTextureViewSetTarget( this->pNvnTextureView,
        Nvn::GetImageTarget( info.GetImageDimension() ) ) );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DepthStencilViewImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    this->pNvnTexture = NULL;
    this->pNvnTextureView = NULL;

    this->state = State_NotInitialized;
}

template<>
void GetImageFormatProperty< Target >( ImageFormatProperty* pOutImageFormatProperty,
    DeviceImpl< Target >* pDevice, ImageFormat imageFormat ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutImageFormatProperty );
    NN_UNUSED( pDevice );

    NVNformat nvnFormat = Nvn::GetImageFormat( imageFormat );
    Nvn::GetImageFormatProperty( pOutImageFormatProperty, nvnFormat );
}

}
}
}
