﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/util/util_BitUtil.h>
#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_TextureInfo.h>
#include <nn/gfx/detail/gfx_Texture-api.gx.2.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gx.2.h>

#include "gfx_CommonHelper.h"
#include "gfx_GxHelper.h"

#include <cafe/gx2.h>

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGx2 Target;

namespace {

GX2SurfaceUse ConvertSurfaceUse( int gpuAccessFlags )
{
    int use = 0;

    if( gpuAccessFlags & GpuAccess_ColorBuffer )
    {
        use |= GX2_SURFACE_USE_COLOR_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_DepthStencil )
    {
        use |= GX2_SURFACE_USE_DEPTH_BUFFER;
    }

    if( gpuAccessFlags & GpuAccess_Texture )
    {
        use |= GX2_SURFACE_USE_TEXTURE;
    }

    return static_cast< GX2SurfaceUse >( use );
}

void SetupGX2Surface( GX2Surface* pSurface, const TextureInfo& info)
{
    ImageStorageDimension imageStorageDimension = info.GetImageStorageDimension();
    NN_SDK_ASSERT_NOT_EQUAL( imageStorageDimension, ImageStorageDimension_Undefined );
    int arrayLength = info.GetArrayLength();

    GX2SurfaceDim dimension = Gx::GetSurfaceDimension( GetImageDimension(
        imageStorageDimension, arrayLength > 1, info.GetMultisampleCount() > 1 ) );

    memset( pSurface, 0, sizeof( GX2Surface ) );
    pSurface->dim = dimension;
    pSurface->width = info.GetWidth();
    pSurface->height = dimension == GX2_SURFACE_DIM_1D_ARRAY ? arrayLength : info.GetHeight();
    pSurface->depth = ( dimension == GX2_SURFACE_DIM_2D_ARRAY ||
        dimension == GX2_SURFACE_DIM_2D_MSAA_ARRAY ) ? arrayLength : info.GetDepth();
    pSurface->numMips = info.GetMipCount();
    pSurface->format = Gx::GetSurfaceFormat( info.GetImageFormat() );
    pSurface->aa = Gx::GetAaMode( info.GetMultisampleCount() );
    pSurface->use = ConvertSurfaceUse( info.GetGpuAccessFlags() );
    pSurface->tileMode = info.GetTileMode() == TileMode_Linear ?
        GX2_TILE_MODE_LINEAR_ALIGNED : GX2_TILE_MODE_DEFAULT_FIX2197;
    NN_GFX_CALL_GX_FUNCTION( GX2SetSurfaceSwizzle( pSurface, info.GetSwizzle() ) );
    NN_GFX_CALL_GX_FUNCTION( GX2CalcSurfaceSizeAndAlignment( pSurface ) );
}

void CalculateAuxHiZSizeAndAlignment( size_t* pOutSize,
    size_t* pOutAlignment, const TextureInfo& info, const GX2Surface& surface )
{
    if( GetTypeFormat( info.GetImageFormat() ) == TypeFormat_DepthStencil )
    {
        GX2DepthBuffer depthBuffer;

        memset( &depthBuffer, 0, sizeof( GX2DepthBuffer ) );
        depthBuffer.surface = surface;
        depthBuffer.viewNumSlices = depthBuffer.surface.depth;
        GX2CalcDepthBufferHiZInfo( &depthBuffer, pOutSize, pOutAlignment );
    }
    else if( surface.aa != GX2_AA_MODE_1X )
    {
        GX2ColorBuffer colorBuffer;

        memset( &colorBuffer, 0, sizeof( GX2ColorBuffer ) );
        colorBuffer.surface = surface;
        colorBuffer.viewNumSlices = surface.depth;

        GX2CalcColorBufferAuxInfo( &colorBuffer, pOutSize, pOutAlignment );
    }
    else
    {
        *pOutSize = 0;
        *pOutAlignment = 1;
    }
}

GX2SurfaceDim GetViewDimension( GX2SurfaceDim storageDimension, GX2SurfaceDim viewDimension ) NN_NOEXCEPT
{
    if( storageDimension == GX2_SURFACE_DIM_1D_ARRAY && viewDimension == GX2_SURFACE_DIM_1D )
    {
        return GX2_SURFACE_DIM_1D_ARRAY;
    }
    else if( storageDimension == GX2_SURFACE_DIM_2D_ARRAY && viewDimension == GX2_SURFACE_DIM_2D )
    {
        return GX2_SURFACE_DIM_2D_ARRAY;
    }
    else if( storageDimension == GX2_SURFACE_DIM_2D_MSAA_ARRAY && viewDimension == GX2_SURFACE_DIM_2D_MSAA )
    {
        return GX2_SURFACE_DIM_2D_MSAA_ARRAY;
    }
    return viewDimension;
}

}

size_t TextureImpl< Target >::CalculateMipDataAlignment(
    DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    GX2Surface surface;
    ImageFormat imageFormat = info.GetImageFormat();
    size_t size;
    size_t alignment;

    SetupGX2Surface( &surface, info );

    CalculateAuxHiZSizeAndAlignment( &size, &alignment, info, surface );

    return std::max NN_PREVENT_MACRO_FUNC ( alignment, surface.alignment );
}

size_t TextureImpl< Target >::CalculateMipDataSize(
    DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    GX2Surface surface;
    size_t totalSize;
    size_t size;
    size_t alignment;

    SetupGX2Surface( &surface, info );

    CalculateAuxHiZSizeAndAlignment( &size, &alignment, info, surface );

    alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, surface.alignment );

    totalSize  = nn::util::align_up( surface.imageSize, alignment );
    totalSize += size > 0 ? nn::util::align_up( surface.mipSize, alignment ) : surface.mipSize;
    totalSize += nn::util::align_up( size, alignment ); // For HiZ/ColorAux

    return totalSize;
}

void TextureImpl< Target >::CalculateMipDataOffsets(
    ptrdiff_t* pMipOffsets, DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    GX2Surface surface;
    size_t size;
    size_t alignment;

    SetupGX2Surface( &surface, info );

    // Add the aligned size of the HiZ/Aux buffer to the starting point of the mipmap data.
    CalculateAuxHiZSizeAndAlignment( &size, &alignment, info, surface );

    for( int idx = 0; idx < info.GetMipCount(); idx++ )
    {
        // mipOffset[0] stores the size of level 0 plus the alignment needed for level 1
        // mipOffset[1..n] stores the offset from the base of level 1 for level (n - 1)
        switch( idx )
        {
        case 0:
            {
                pMipOffsets[ 0 ] = 0;
            }
            break;
        case 1:
            {
                pMipOffsets[ 1 ] = surface.mipOffset[ 0 ] + size;
            }
            break;
        default:
            {
                pMipOffsets[ idx ] = pMipOffsets[ 1 ] + surface.mipOffset[ idx - 1 ];
            }
            break;
        }
    }
}

size_t TextureImpl< Target >::GetRowPitch( DeviceImpl< Target >*, const InfoType& info ) NN_NOEXCEPT
{
    GX2Surface surface;
    SetupGX2Surface( &surface, info );

    return surface.pitch;
}

TextureImpl< Target >::TextureImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureImpl< Target >::~TextureImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( pMemoryPool );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_ASSERT( pMemoryPool->ToData()->memorySize >= memoryPoolOffset + memoryPoolSize );
    NN_SDK_ASSERT( memoryPoolSize >= CalculateMipDataSize( pDevice, info ) );
    NN_UNUSED( memoryPoolSize );

    GX2Surface* pSurface = reinterpret_cast< GX2Surface* >( &this->gx2Surface );
    NN_STATIC_ASSERT( sizeof( this->gx2Surface ) == sizeof( GX2Surface ) );

    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    SetupGX2Surface( pSurface, info );

    pSurface->imagePtr = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();
    pSurface->mipPtr = pSurface->numMips > 1 ? nn::util::BytePtr(
        pSurface->imagePtr, pSurface->imageSize ).Get() : NULL;

    NN_SDK_REQUIRES( pSurface->imagePtr && nn::util::is_aligned(
        reinterpret_cast< uintptr_t >( pSurface->imagePtr ), pSurface->alignment ) );
    NN_SDK_ASSERT( !pSurface->mipSize || ( pSurface->mipPtr && nn::util::is_aligned(
        reinterpret_cast< uintptr_t >( pSurface->mipPtr ), pSurface->alignment ) ) );

    NN_GFX_CALL_GX_FUNCTION( GX2Invalidate( GX2_INVALIDATE_CPU, pSurface->imagePtr, pSurface->imageSize ) );
    if( pSurface->mipPtr )
    {
        NN_GFX_CALL_GX_FUNCTION(GX2Invalidate( GX2_INVALIDATE_CPU, pSurface->mipPtr, pSurface->mipSize ) );
    }
    if( GetTypeFormat( info.GetImageFormat() ) != TypeFormat_DepthStencil && pSurface->aa != GX2_AA_MODE_1X )
    {
        size_t size;
        size_t alignment;
        void* auxPtr;

        // Clear the AUX buffer to the correct value.
        CalculateAuxHiZSizeAndAlignment( &size, &alignment, info, *pSurface );
        alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, pSurface->alignment );

        auxPtr = reinterpret_cast< void* >( reinterpret_cast< ptrdiff_t >( pSurface->imagePtr ) +
             nn::util::align_up( pSurface->imageSize, alignment ) );
        memset( auxPtr, GX2_AUX_BUFFER_CLEAR_VALUE, size );
        NN_GFX_CALL_GX_FUNCTION(GX2Invalidate( GX2_INVALIDATE_CPU, auxPtr, size ) );
    }

    this->pGx2Surface = &this->gx2Surface;

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void TextureImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::TextureViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

TextureViewImpl< Target >::~TextureViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void TextureViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    GX2Texture* pGx2Texture = reinterpret_cast< GX2Texture* >( &this->gx2Texture );
    NN_STATIC_ASSERT( sizeof( this->gx2Texture ) == sizeof( GX2Texture ) );
    const TextureImpl< ApiVariationGx2 > *pTexture =  info.GetTexturePtr();

    NN_SDK_REQUIRES( pTexture != NULL );
    NN_SDK_REQUIRES( IsInitialized( *pTexture ) );

    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    pGx2Texture->surface = *reinterpret_cast< const GX2Surface* >( &pTexture->ToData()->gx2Surface );
    pGx2Texture->surface.dim = GetViewDimension(
        pGx2Texture->surface.dim, Gx::GetSurfaceDimension( info.GetImageDimension() ) );
    pGx2Texture->surface.format = Gx::GetSurfaceFormat( info.GetImageFormat() );
    pGx2Texture->viewFirstMip = info.GetSubresourceRange().GetMipRange().GetMinMipLevel();
    pGx2Texture->viewNumMips = info.GetSubresourceRange().GetMipRange().GetMipCount();
    pGx2Texture->viewFirstSlice = info.GetSubresourceRange().GetArrayRange().GetBaseArrayIndex();
    pGx2Texture->viewNumSlices = info.GetSubresourceRange().GetArrayRange().GetArrayLength();
    if( GetTypeFormat( info.GetImageFormat() ) == TypeFormat_DepthStencil )
    {
        switch( pGx2Texture->surface.format )
        {
        case GX2_SURFACE_FORMAT_D_D24_S8_UNORM:
            {
                // FIXME: This format requires tile conversion from depth to color tiling
                if( info.GetDepthStencilTextureMode() == DepthStencilFetchMode_DepthComponent )
                {
                    pGx2Texture->surface.format = GX2_SURFACE_FORMAT_T_R24_UNORM_X8;
                }
                else
                {
                    pGx2Texture->surface.format = GX2_SURFACE_FORMAT_T_X24_G8_UINT;
                }
                pGx2Texture->surface.use = GX2_SURFACE_USE_TEXTURE;
            }
            break;

        case GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24:
            {
                // This can be used directly without HiZ/
                if( info.GetDepthStencilTextureMode() == DepthStencilFetchMode_DepthComponent )
                {
                    pGx2Texture->surface.format = GX2_SURFACE_FORMAT_T_R32_FLOAT_X8_X24;
                }
                else
                {
                    pGx2Texture->surface.format = GX2_SURFACE_FORMAT_T_X32_G8_UINT_X24;
                }
                pGx2Texture->surface.use = GX2_SURFACE_USE_TEXTURE;
            }
            break;

        default:
            {
                pGx2Texture->surface.use = GX2_SURFACE_USE_DEPTH_BUFFER_TEXTURE;
            }
            break;
        }
    }

    // Setup the requested component select
    {
        GX2CompSel componentSelect =
            GX2_GET_COMP_SEL( Gx::GetComponentMapping( info.GetChannelMapping( ColorChannel_Red ) ),
                Gx::GetComponentMapping( info.GetChannelMapping( ColorChannel_Green ) ),
                Gx::GetComponentMapping( info.GetChannelMapping( ColorChannel_Blue ) ),
                Gx::GetComponentMapping( info.GetChannelMapping( ColorChannel_Alpha ) ) );
        pGx2Texture->compSel = componentSelect;
    }

    NN_GFX_CALL_GX_FUNCTION( GX2InitTextureRegs( pGx2Texture ) );

    this->pGx2Texture = &this->gx2Texture;

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void TextureViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );
    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::ColorTargetViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

ColorTargetViewImpl< Target >::~ColorTargetViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void ColorTargetViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    GX2ColorBuffer* pGx2ColorBuffer = reinterpret_cast< GX2ColorBuffer* >( &this->gx2ColorBuffer );
    NN_STATIC_ASSERT( sizeof( this->gx2ColorBuffer ) == sizeof( GX2ColorBuffer ) );
    const TextureImpl< ApiVariationGx2 >* pTexture =  info.GetTexturePtr();
    u32 size;
    u32 alignment;

    NN_SDK_REQUIRES( pTexture != NULL );
    NN_SDK_REQUIRES( IsInitialized( *pTexture ) );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    // Setup the color buffer
    pGx2ColorBuffer->surface = *reinterpret_cast< const GX2Surface* >( &pTexture->ToData()->gx2Surface );
    pGx2ColorBuffer->surface.use = GX2_SURFACE_USE_COLOR_BUFFER_TEXTURE;
    pGx2ColorBuffer->surface.dim = GetViewDimension(
        pGx2ColorBuffer->surface.dim, Gx::GetSurfaceDimension( info.GetImageDimension() ) );
    pGx2ColorBuffer->viewMip = info.GetMipLevel();
    pGx2ColorBuffer->viewFirstSlice = info.GetArrayRange().GetBaseArrayIndex();
    pGx2ColorBuffer->viewNumSlices = info.GetArrayRange().GetArrayLength();
    pGx2ColorBuffer->auxPtr = NULL;
    pGx2ColorBuffer->auxSize = 0;
    NN_GFX_CALL_GX_FUNCTION( GX2InitColorBufferRegs( pGx2ColorBuffer ) );

    NN_GFX_CALL_GX_FUNCTION( GX2CalcColorBufferAuxInfo( pGx2ColorBuffer, &size, &alignment ) );
    alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, pGx2ColorBuffer->surface.alignment );

    if( size )
    {
        // AUX buffer starts just after the image buffer ends.
        NN_GFX_CALL_GX_FUNCTION( GX2InitColorBufferAuxPtr( pGx2ColorBuffer,
             reinterpret_cast< void* >( reinterpret_cast< ptrdiff_t >( pGx2ColorBuffer->surface.imagePtr ) +
             nn::util::align_up( pGx2ColorBuffer->surface.imageSize, alignment ) ) ) );
    }

    this->pGx2ColorBuffer = &this->gx2ColorBuffer;

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void ColorTargetViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::DepthStencilViewImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DepthStencilViewImpl< Target >::~DepthStencilViewImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void DepthStencilViewImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    GX2DepthBuffer* pGx2DepthBuffer = reinterpret_cast< GX2DepthBuffer* >( &this->gx2DepthBuffer );
    NN_STATIC_ASSERT( sizeof( this->gx2DepthBuffer ) == sizeof( GX2DepthBuffer ) );
    const TextureImpl< ApiVariationGx2 > *pTexture =  info.GetTexturePtr();
    u32 size;
    u32 alignment;

    NN_SDK_REQUIRES( pTexture != NULL );
    NN_SDK_REQUIRES( IsInitialized( *pTexture ) );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    // Setup the depth buffer
    pGx2DepthBuffer->surface = *reinterpret_cast< const GX2Surface* >( &pTexture->ToData()->gx2Surface );
    pGx2DepthBuffer->surface.use = GX2_SURFACE_USE_DEPTH_BUFFER_TEXTURE;
    pGx2DepthBuffer->surface.dim = GetViewDimension(
        pGx2DepthBuffer->surface.dim, Gx::GetSurfaceDimension( info.GetImageDimension() ) );
    pGx2DepthBuffer->viewMip = info.GetMipLevel();
    pGx2DepthBuffer->viewFirstSlice = info.GetArrayRange().GetBaseArrayIndex();
    pGx2DepthBuffer->viewNumSlices = info.GetArrayRange().GetArrayLength();
    pGx2DepthBuffer->clearDepth = 1.0f; // GX2 Default
    pGx2DepthBuffer->clearStencil = 0; // GX2 Default;
    pGx2DepthBuffer->hiZPtr = NULL;
    pGx2DepthBuffer->hiZSize = 0;
    NN_GFX_CALL_GX_FUNCTION( GX2InitDepthBufferRegs( pGx2DepthBuffer ) );

    NN_GFX_CALL_GX_FUNCTION( GX2CalcDepthBufferHiZInfo( pGx2DepthBuffer, &size, &alignment ) );
    alignment = std::max NN_PREVENT_MACRO_FUNC ( alignment, pGx2DepthBuffer->surface.alignment );

    // HiZ buffer starts just after the image ends.
    NN_GFX_CALL_GX_FUNCTION( GX2InitDepthBufferHiZPtr( pGx2DepthBuffer,
         reinterpret_cast< void* >( reinterpret_cast< ptrdiff_t >( pGx2DepthBuffer->surface.imagePtr ) +
         nn::util::align_up( pGx2DepthBuffer->surface.imageSize, alignment ) ) ) );

    this->pGx2DepthBuffer = &this->gx2DepthBuffer;

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DepthStencilViewImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    this->state = State_NotInitialized;
}

template<>
void GetImageFormatProperty< Target >( ImageFormatProperty* pOutImageFormatProperty,
    DeviceImpl< Target >* pDevice, ImageFormat imageFormat ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pOutImageFormatProperty );
    NN_UNUSED( pDevice );

    GX2SurfaceFormat surfaceFormat = Gx::GetSurfaceFormat( imageFormat );
    Gx::GetImageFormatProperty( pOutImageFormatProperty, surfaceFormat );
}

}
}
}
