﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Macro.h>
#include <nn/nn_Windows.h>

#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
    #error
#endif

#include <nn/vi/vi_Layer.h>

#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>
#include <nn/gfx/gfx_Texture.h>
#include <nn/gfx/gfx_SyncInfo.h>

#include <nn/gfx/detail/gfx_SwapChain-api.vk.1-os.win32.h>
#include <nn/gfx/detail/gfx_Device-api.vk.1.h>
#include <nn/gfx/detail/gfx_Texture-api.vk.1.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.vk.1.h>
#include <nn/gfx/detail/gfx_Sync-api.vk.1.h>

#include "gfx_CommonHelper.h"
#include "gfx_VkHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationVk1 Target;

void InitializeSwapChainCommonImpl( SwapChainImpl< Target >* pThis, DeviceImpl< Target >* pDevice,
    const SwapChainInfo& info, MemoryPoolImpl< Target >* pMemoryPool,
    ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT;

void SwapChainImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pMemoryPool );
    NN_UNUSED( memoryPoolOffset );
    NN_UNUSED( memoryPoolSize );

    this->pGfxDevice = pDevice;
    this->totalScanBuffers = info.GetBufferCount();
    this->currentPresentSemaphoreIndex = 0;
    this->currentScanBufferIndex = 0;
    this->flag.SetBit( Flag_IsFirstPresentation, true );

    nn::vi::NativeWindowHandle hNativeWindow = NULL;
    nn::Result viResult;
    NN_GFX_CALL_NNVI_FUNCTION(
        viResult = nn::vi::GetNativeWindow( &hNativeWindow, info.GetLayer() ) );
    NN_SDK_ASSERT( viResult.IsSuccess() );
    NN_UNUSED( viResult );

    this->hWnd = hNativeWindow;
    this->hDc = ::GetDC( static_cast< HWND >( this->hWnd ) );
    NN_SDK_ASSERT_NOT_NULL( this->hDc );

    VkResult result;
    VkWin32SurfaceCreateInfoKHR createInfo = { };
    createInfo.sType = VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR;
    createInfo.pNext = NULL;
    createInfo.flags = 0;
    createInfo.hinstance = ::GetModuleHandle( NULL );
    createInfo.hwnd = hWnd;

    VkSurfaceKHR surface;
    NN_GFX_CALL_VK_FUNCTION( result = vkCreateWin32SurfaceKHR(
        CastToVkDispatchableObject< VkInstance >( this->pGfxDevice->ToData()->hInstance ), &createInfo,
        pDevice->ToData()->pAllocationCallback, &surface ) );
    NN_SDK_ASSERT_EQUAL( result, VK_SUCCESS );
    this->hSurfaceKHR = CastFromVkNonDispatchableObject< VkSurfaceKHR >( surface );

    InitializeSwapChainCommonImpl( this, pDevice, info, pMemoryPool, memoryPoolOffset, memoryPoolSize );

    this->state = State_Initialized;
}

void SwapChainImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( IsInitialized( *this ) );

    VkDevice device = CastToVkDispatchableObject< VkDevice >( this->pGfxDevice->ToData()->hDevice );
    VkSwapchainKHR vkSwapChainKHR = CastToVkNonDispatchableObject< VkSwapchainKHR >( this->hSwapChainKHR );
    NN_GFX_CALL_VK_FUNCTION( vkDestroySwapchainKHR( device, vkSwapChainKHR, pDevice->ToData()->pAllocationCallback ) );

    for ( uint32_t index = 0; index < this->totalScanBuffers; index++ )
    {
        SemaphoreImpl< Target >* pSemaphore = nn::gfx::DataToAccessor( this->presentSemaphore[ index ] );
        pSemaphore->Finalize( pDevice );
        ColorTargetViewImpl< Target >* pScanBufferView = nn::gfx::DataToAccessor( this->scanBufferViews[ index ] );
        pScanBufferView->Finalize( pDevice );
        TextureImpl< Target >* pScanBuffer = nn::gfx::DataToAccessor( this->scanBufferTextures[ index ] );
        pScanBuffer->Finalize( pDevice );
    }

    NN_GFX_CALL_VK_FUNCTION( vkDestroySurfaceKHR(
        CastToVkDispatchableObject< VkInstance >( this->pGfxDevice->ToData()->hInstance ),
        CastToVkNonDispatchableObject< VkSurfaceKHR >( this->hSurfaceKHR ), pDevice->ToData()->pAllocationCallback ) );

    if ( this->hDc )
    {
        ::ReleaseDC( static_cast< HWND >( this->hWnd ), static_cast< HDC >( this->hDc ) );
        this->hDc = nullptr;
    }

    this->state = State_NotInitialized;
}

}
}
}
