﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>

#include <nn/nn_Macro.h>

#include <nn/vi/vi_Layer.h>

#include <nn/gfx/gfx_SwapChainInfo.h>
#include <nn/gfx/gfx_DataAccessorConverter.h>
#include <nn/gfx/gfx_Texture.h>

#include <nn/gfx/detail/gfx_SwapChain-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Device-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Texture-api.d3d.11.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Sync-api.d3d.11.h>

#include "gfx_CommonHelper.h"
#include "gfx_D3dHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

namespace {

void SetupScanbufferTextureInfo( TextureInfo* pOutInfo, const SwapChainInfo& info ) NN_NOEXCEPT
{
    pOutInfo->SetImageStorageDimension( ImageStorageDimension_2d );
    pOutInfo->SetTileMode( TileMode_Optimal );
    pOutInfo->SetSwizzle( 0 );
    pOutInfo->SetMipCount( 1 );
    pOutInfo->SetMultiSampleCount( 1 );
    pOutInfo->SetImageFormat( info.GetFormat() );
    pOutInfo->SetGpuAccessFlags( GpuAccess_ColorBuffer );
    pOutInfo->SetWidth( info.GetWidth() );
    pOutInfo->SetHeight( info.GetHeight() );
    pOutInfo->SetDepth( 0 );
    pOutInfo->SetArrayLength( 1 );
}

}

size_t SwapChainImpl< Target >::GetScanBufferAlignment( DeviceImpl< Target >*, const SwapChainInfo& ) NN_NOEXCEPT
{
    return 1;
}

size_t SwapChainImpl< Target >::CalculateScanBufferSize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    TextureInfo textureInfo;
    SetupScanbufferTextureInfo( &textureInfo, info );

    return TextureImpl< Target >::CalculateMipDataSize( pDevice, textureInfo ) * info.GetBufferCount();
}

SwapChainImpl< Target >::SwapChainImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

SwapChainImpl< Target >::~SwapChainImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void SwapChainImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_ASSERT( this->pGfxDevice == NULL );
    NN_SDK_REQUIRES( !pMemoryPool || IsInitialized( *pMemoryPool ) );
    NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( !pMemoryPool || memoryPoolSize >= CalculateScanBufferSize( pDevice, info ) );
    NN_SDK_ASSERT( !pMemoryPool || pMemoryPool->ToData()->memorySize >= memoryPoolOffset + memoryPoolSize );

    this->pGfxDevice = pDevice;

    nn::vi::NativeWindowHandle hNativeWindow = NULL;
    nn::Result viResult;
    NN_GFX_CALL_NNVI_FUNCTION(
        viResult = nn::vi::GetNativeWindow( &hNativeWindow, info.GetLayer() ) );
    NN_SDK_ASSERT( viResult.IsSuccess() );
    NN_UNUSED( viResult );

    this->pWnd = hNativeWindow;

    // 描画用のTextureおよびColorTargetViewを生成します。
    // 表示バッファはSwapChainに内包されています。

    TextureInfo textureInfo;
    SetupScanbufferTextureInfo( &textureInfo, info );

    ColorTargetViewInfo colorTargetViewInfo;
    colorTargetViewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
    colorTargetViewInfo.SetMipLevel( 0 );
    colorTargetViewInfo.SetImageFormat( info.GetFormat() );
    colorTargetViewInfo.EditArrayRange().SetBaseArrayIndex( 0 );
    colorTargetViewInfo.EditArrayRange().SetArrayLength( 1 );

    this->scanBufferCount = static_cast< int8_t >( info.GetBufferCount() );
    for( int idxScanBuffer = 0; idxScanBuffer < this->scanBufferCount; ++idxScanBuffer )
    {
        TextureImpl< Target >* pScanBuffer =
            nn::gfx::DataToAccessor( this->scanBuffers[ idxScanBuffer ] );
        ColorTargetViewImpl< Target >* pScanBufferView =
            nn::gfx::DataToAccessor( this->scanBufferViews[ idxScanBuffer ] );
        pScanBuffer->Initialize( pDevice, textureInfo, pMemoryPool, memoryPoolOffset, memoryPoolSize );
        colorTargetViewInfo.SetTexturePtr( static_cast< TTexture< Target >* >( pScanBuffer ) );
        pScanBufferView->Initialize( pDevice, colorTargetViewInfo );
    }

    IDXGISwapChain* pDxgiSwapChain;
    D3d::CreateSwapChain( &pDxgiSwapChain, info.GetWidth(), info.GetHeight(), info.GetFormat(), info.GetBufferCount(),
        pDevice->ToData()->renderingContext.hD3dDevice, this->pWnd );
    NN_SDK_ASSERT( IsD3dHandleValid( pDxgiSwapChain ) );

    this->pSwapChain = pDxgiSwapChain;
    this->state = State_Initialized;
}

void SwapChainImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_UNUSED( pDevice );

    for( int idxScanBuffer = 0; idxScanBuffer < this->scanBufferCount; ++idxScanBuffer )
    {
        TextureImpl< Target >* pScanBuffer =
            nn::gfx::DataToAccessor( this->scanBuffers[ idxScanBuffer ] );
        ColorTargetViewImpl< Target >* pScanBufferView =
            nn::gfx::DataToAccessor( this->scanBufferViews[ idxScanBuffer ] );
        NN_SDK_ASSERT( IsInitialized( *pScanBuffer ) );
        NN_SDK_ASSERT( IsInitialized( *pScanBufferView ) );
        pScanBuffer->Finalize( pDevice );
        pScanBufferView->Finalize( pDevice );
    }

    IDXGISwapChain* pDxgiSwapChain = static_cast< IDXGISwapChain* >( this->pSwapChain );
    NN_GFX_CALL_D3D_FUNCTION( pDxgiSwapChain->Release() );

    this->pGfxDevice = NULL;
    this->state = State_NotInitialized;
}

int SwapChainImpl< Target >::GetScanBufferViews(
    TColorTargetView< Target >** ppOutScanBufferArray, int maxScanBufferCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    if( ppOutScanBufferArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferCount,
            static_cast< int >( this->scanBufferCount ) );
        for( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBufferViews[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->scanBufferCount;
    }
}

int SwapChainImpl< Target >::GetScanBuffers(
    TTexture< Target >** ppOutScanBufferTextureArray, int maxScanBufferTextureCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    if( ppOutScanBufferTextureArray )
    {
        int count = std::min NN_PREVENT_MACRO_FUNC( maxScanBufferTextureCount,
            static_cast< int >( this->scanBufferCount ) );
        for( int idxScanBuffer = 0; idxScanBuffer < count; ++idxScanBuffer )
        {
            ppOutScanBufferTextureArray[ idxScanBuffer ] =
                nn::gfx::DataToAccessor( this->scanBuffers[ idxScanBuffer ] );
        }
        return count;
    }
    else
    {
        return this->scanBufferCount;
    }
}

AcquireScanBufferResult SwapChainImpl< Target >::AcquireNextScanBufferIndex(
    int* pOutScanBufferIndex, SemaphoreImpl< Target >* pSemaphore,
    FenceImpl< Target >* pFence ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( IsInitialized( *this ) );
    NN_SDK_REQUIRES_NOT_NULL( pOutScanBufferIndex );
    NN_SDK_REQUIRES_NOT_NULL( pSemaphore );
    NN_SDK_REQUIRES( IsInitialized( *pSemaphore ) );
    NN_SDK_REQUIRES( pFence == NULL || IsInitialized( *pFence ) );

    pSemaphore->ToData()->flags.SetBit( SemaphoreImpl< Target >::DataType::Flag_Signaled, true );
    if( pFence )
    {
        pFence->ToData()->flags.SetBit( FenceImpl< Target >::DataType::Flag_Signaled, true );
    }

    *pOutScanBufferIndex = this->currentScanBufferIndex;

    this->flag.SetBit( Flag_Acquired, true );

    return AcquireScanBufferResult_Success;
}

int SwapChainImpl< Target >::AcquireNextScanBufferIndex() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->flag.SetBit( Flag_Acquired, true );
    return this->currentScanBufferIndex;
}

ColorTargetViewImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBufferView() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->flag.SetBit( Flag_Acquired, true );
    return nn::gfx::DataToAccessor( this->scanBufferViews[ this->currentScanBufferIndex ] );
}

TextureImpl< Target >* SwapChainImpl< Target >::AcquireNextScanBuffer() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->flag.SetBit( Flag_Acquired, true );
    return nn::gfx::DataToAccessor( this->scanBuffers[ this->currentScanBufferIndex ] );
}

}
}
}
