﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_PipelineInfo.h>

#include <nn/gfx/detail/gfx_Misc.h>
#include <nn/gfx/detail/gfx_Pipeline-api.d3d.11.h>
#include <nn/gfx/detail/gfx_Shader-api.d3d.11.h>
#include <nn/gfx/detail/gfx_State-api.d3d.11.h>

#include "gfx_D3dHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationD3d11 Target;

size_t PipelineImpl< Target >::GetRequiredMemorySize( const GraphicsInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( info.GetBlendStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetVertexStateInfo() );
    return BlendStateImpl< Target >::GetRequiredMemorySize( *info.GetBlendStateInfo() )
        + VertexStateImpl< Target >::GetRequiredMemorySize( *info.GetVertexStateInfo() )
        + 16; // 一時的措置としてダミーで16バイト加算します。（サイズ0を許容しない実装のTestsアプリが見られるため。）
}

size_t PipelineImpl< Target >::GetRequiredMemorySize( const ComputeInfoType& info ) NN_NOEXCEPT
{
    NN_UNUSED( info );
    return 0;
}

PipelineImpl< Target >::PipelineImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

PipelineImpl< Target >::~PipelineImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void PipelineImpl< Target >::SetMemory( void* value, size_t size ) NN_NOEXCEPT
{
    NN_UNUSED( value );
    NN_UNUSED( size );
}

void* PipelineImpl< Target >::GetMemory() NN_NOEXCEPT
{
    return NULL;
}

const void* PipelineImpl< Target >::GetMemory() const NN_NOEXCEPT
{
    return NULL;
}

void PipelineImpl< Target >::Initialize( DeviceImpl< Target >*pDevice, const GraphicsInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );

    this->flags.Clear();

    this->pipelineType = PipelineType_Graphics;

    NN_SDK_ASSERT_NOT_NULL( info.GetRasterizerStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetBlendStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetDepthStencilStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetRenderTargetStateInfo() );
    NN_SDK_ASSERT_NOT_NULL( info.GetVertexStateInfo() );

    this->pShader = static_cast< const ShaderImpl< Target >* >( info.GetShaderPtr() );
    NN_SDK_ASSERT_NOT_NULL( this->pShader.ptr );

    RasterizerStateImpl< Target >& rasterizerStateAccesssor =
        nn::gfx::DataToAccessor( this->rasterizerState );
    BlendStateImpl< Target >& blendStateAccessor =
        nn::gfx::DataToAccessor( this->blendState );
    DepthStencilStateImpl< Target >& depthStencilStateAccessor =
        nn::gfx::DataToAccessor( this->depthStencilState );
    VertexStateImpl< Target >& vertexStateAccessor =
        nn::gfx::DataToAccessor( this->vertexState );
    TessellationStateImpl< Target >& tessellationStateAccessor =
        nn::gfx::DataToAccessor( this->tessellationState );

    rasterizerStateAccesssor.Initialize( pDevice, *info.GetRasterizerStateInfo() );
    blendStateAccessor.Initialize( pDevice, *info.GetBlendStateInfo() );
    depthStencilStateAccessor.Initialize( pDevice, *info.GetDepthStencilStateInfo() );
    vertexStateAccessor.Initialize( pDevice, *info.GetVertexStateInfo(), this->pShader );
    if( info.GetTessellationStateInfo() )
    {
        tessellationStateAccessor.Initialize( pDevice, *info.GetTessellationStateInfo() );
        this->flags.SetBit( Flag_HasTessellationState, true );
    }

    this->state = State_Initialized;
}

void PipelineImpl< Target >::Initialize( DeviceImpl< Target >*pDevice, const ComputeInfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_UNUSED( pDevice );

    this->flags.Clear();

    this->pipelineType = PipelineType_Compute;

    this->pShader = static_cast< const ShaderImpl< Target >* >( info.GetShaderPtr() );
    NN_SDK_ASSERT_NOT_NULL( this->pShader.ptr );

    this->state = State_Initialized;
}

void PipelineImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_UNUSED( pDevice );
    NN_SDK_REQUIRES( this->state == State_Initialized );

    if( this->pipelineType == PipelineType_Graphics )
    {
        static_cast< RasterizerStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->rasterizerState ) ).Finalize( pDevice );
        static_cast< BlendStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->blendState ) ).Finalize( pDevice );
        static_cast< DepthStencilStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->depthStencilState ) ).Finalize( pDevice );
        static_cast< VertexStateImpl< Target >& >(
            nn::gfx::DataToAccessor( this->vertexState ) ).Finalize( pDevice );
        if( this->flags.GetBit( Flag_HasTessellationState ) )
        {
            static_cast< TessellationStateImpl< Target >& >(
                nn::gfx::DataToAccessor( this->tessellationState ) ).Finalize( pDevice );
        }
    }

    this->state = State_NotInitialized;
}

}
}
}
