﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <memory>
#include <cstring>
#include <cstdlib>

#include <nvn/nvn_FuncPtrImpl.h>

#include <nn/nn_SdkLog.h>
#include <nn/nn_StaticAssert.h>

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_Common.h>

#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Log.h>

#include "gfx_NvnHelper.h"
#include "gfx_CommonHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {

    struct ImageFormatAndNvnFormat
    {
        ImageFormat imageFormat;
        NVNformat nvnFormat;
    } s_NvnTextureFormatList[] =
    {
        // ImageFormat の昇順
        { ImageFormat_R8_Unorm, NVN_FORMAT_R8 },
        { ImageFormat_R8_Snorm, NVN_FORMAT_R8SN },
        { ImageFormat_R8_Uint, NVN_FORMAT_R8UI },
        { ImageFormat_R8_Sint, NVN_FORMAT_R8I },
        { ImageFormat_R4_G4_B4_A4_Unorm, NVN_FORMAT_RGBA4 },
        { ImageFormat_R5_G5_B5_A1_Unorm, NVN_FORMAT_RGB5A1 },
        { ImageFormat_A1_B5_G5_R5_Unorm, NVN_FORMAT_A1BGR5 },
        { ImageFormat_R5_G6_B5_Unorm, NVN_FORMAT_RGB565 },
        { ImageFormat_B5_G6_R5_Unorm, NVN_FORMAT_BGR565 },
        { ImageFormat_R8_G8_Unorm, NVN_FORMAT_RG8 },
        { ImageFormat_R8_G8_Snorm, NVN_FORMAT_RG8SN },
        { ImageFormat_R8_G8_Uint, NVN_FORMAT_RG8UI },
        { ImageFormat_R8_G8_Sint, NVN_FORMAT_RG8I },
        { ImageFormat_R16_Unorm, NVN_FORMAT_R16 },
        { ImageFormat_R16_Snorm, NVN_FORMAT_R16SN },
        { ImageFormat_R16_Uint, NVN_FORMAT_R16UI },
        { ImageFormat_R16_Sint, NVN_FORMAT_R16I },
        { ImageFormat_R16_Float, NVN_FORMAT_R16F },
        { ImageFormat_D16_Unorm, NVN_FORMAT_DEPTH16 },
        { ImageFormat_R8_G8_B8_A8_Unorm, NVN_FORMAT_RGBA8 },
        { ImageFormat_R8_G8_B8_A8_Snorm, NVN_FORMAT_RGBA8SN },
        { ImageFormat_R8_G8_B8_A8_Uint, NVN_FORMAT_RGBA8UI },
        { ImageFormat_R8_G8_B8_A8_Sint, NVN_FORMAT_RGBA8I },
        { ImageFormat_R8_G8_B8_A8_UnormSrgb, NVN_FORMAT_RGBA8_SRGB },
        { ImageFormat_B8_G8_R8_A8_Unorm, NVN_FORMAT_BGRA8 },
        { ImageFormat_B8_G8_R8_A8_UnormSrgb, NVN_FORMAT_BGRA8_SRGB },
        { ImageFormat_R9_G9_B9_E5_SharedExp, NVN_FORMAT_RGB9E5F },
        { ImageFormat_R10_G10_B10_A2_Unorm, NVN_FORMAT_RGB10A2 },
        { ImageFormat_R10_G10_B10_A2_Uint, NVN_FORMAT_RGB10A2UI },
        { ImageFormat_R11_G11_B10_Float, NVN_FORMAT_R11G11B10F },
        { ImageFormat_R16_G16_Unorm, NVN_FORMAT_RG16 },
        { ImageFormat_R16_G16_Snorm, NVN_FORMAT_RG16SN },
        { ImageFormat_R16_G16_Uint, NVN_FORMAT_RG16UI },
        { ImageFormat_R16_G16_Sint, NVN_FORMAT_RG16I },
        { ImageFormat_R16_G16_Float, NVN_FORMAT_RG16F },
        { ImageFormat_D24_Unorm_S8_Uint, NVN_FORMAT_DEPTH24_STENCIL8 },
        { ImageFormat_R32_Uint, NVN_FORMAT_R32UI },
        { ImageFormat_R32_Sint, NVN_FORMAT_R32I },
        { ImageFormat_R32_Float, NVN_FORMAT_R32F },
        { ImageFormat_D32_Float, NVN_FORMAT_DEPTH32F },
        { ImageFormat_R16_G16_B16_A16_Unorm, NVN_FORMAT_RGBA16 },
        { ImageFormat_R16_G16_B16_A16_Snorm, NVN_FORMAT_RGBA16SN },
        { ImageFormat_R16_G16_B16_A16_Uint, NVN_FORMAT_RGBA16UI },
        { ImageFormat_R16_G16_B16_A16_Sint, NVN_FORMAT_RGBA16I },
        { ImageFormat_R16_G16_B16_A16_Float, NVN_FORMAT_RGBA16F },
        { ImageFormat_D32_Float_S8_Uint_X24, NVN_FORMAT_DEPTH32F_STENCIL8 },
        { ImageFormat_R32_G32_Uint, NVN_FORMAT_RG32UI },
        { ImageFormat_R32_G32_Sint, NVN_FORMAT_RG32I },
        { ImageFormat_R32_G32_Float, NVN_FORMAT_RG32F },
        { ImageFormat_R32_G32_B32_Uint, NVN_FORMAT_RGB32UI },
        { ImageFormat_R32_G32_B32_Sint, NVN_FORMAT_RGB32I },
        { ImageFormat_R32_G32_B32_Float, NVN_FORMAT_RGB32F },
        { ImageFormat_R32_G32_B32_A32_Uint, NVN_FORMAT_RGBA32UI },
        { ImageFormat_R32_G32_B32_A32_Sint, NVN_FORMAT_RGBA32I },
        { ImageFormat_R32_G32_B32_A32_Float, NVN_FORMAT_RGBA32F },
        { ImageFormat_Bc1_Unorm, NVN_FORMAT_RGBA_DXT1 },
        { ImageFormat_Bc1_UnormSrgb, NVN_FORMAT_RGBA_DXT1_SRGB },
        { ImageFormat_Bc2_Unorm, NVN_FORMAT_RGBA_DXT3 },
        { ImageFormat_Bc2_UnormSrgb, NVN_FORMAT_RGBA_DXT3_SRGB },
        { ImageFormat_Bc3_Unorm, NVN_FORMAT_RGBA_DXT5 },
        { ImageFormat_Bc3_UnormSrgb, NVN_FORMAT_RGBA_DXT5_SRGB },
        { ImageFormat_Bc4_Unorm, NVN_FORMAT_RGTC1_UNORM },
        { ImageFormat_Bc4_Snorm, NVN_FORMAT_RGTC1_SNORM },
        { ImageFormat_Bc5_Unorm, NVN_FORMAT_RGTC2_UNORM },
        { ImageFormat_Bc5_Snorm, NVN_FORMAT_RGTC2_SNORM },
        { ImageFormat_Bc6_Float, NVN_FORMAT_BPTC_SFLOAT },
        { ImageFormat_Bc6_Ufloat, NVN_FORMAT_BPTC_UFLOAT },
        { ImageFormat_Bc7_Unorm, NVN_FORMAT_BPTC_UNORM },
        { ImageFormat_Bc7_UnormSrgb, NVN_FORMAT_BPTC_UNORM_SRGB },

        { ImageFormat_Astc_4x4_Unorm, NVN_FORMAT_RGBA_ASTC_4x4 },
        { ImageFormat_Astc_4x4_UnormSrgb, NVN_FORMAT_RGBA_ASTC_4x4_SRGB },
        { ImageFormat_Astc_5x4_Unorm, NVN_FORMAT_RGBA_ASTC_5x4 },
        { ImageFormat_Astc_5x4_UnormSrgb, NVN_FORMAT_RGBA_ASTC_5x4_SRGB },
        { ImageFormat_Astc_5x5_Unorm, NVN_FORMAT_RGBA_ASTC_5x5 },
        { ImageFormat_Astc_5x5_UnormSrgb, NVN_FORMAT_RGBA_ASTC_5x5_SRGB },
        { ImageFormat_Astc_6x5_Unorm, NVN_FORMAT_RGBA_ASTC_6x5 },
        { ImageFormat_Astc_6x5_UnormSrgb, NVN_FORMAT_RGBA_ASTC_6x5_SRGB },
        { ImageFormat_Astc_6x6_Unorm, NVN_FORMAT_RGBA_ASTC_6x6 },
        { ImageFormat_Astc_6x6_UnormSrgb, NVN_FORMAT_RGBA_ASTC_6x6_SRGB },
        { ImageFormat_Astc_8x5_Unorm, NVN_FORMAT_RGBA_ASTC_8x5 },
        { ImageFormat_Astc_8x5_UnormSrgb, NVN_FORMAT_RGBA_ASTC_8x5_SRGB },
        { ImageFormat_Astc_8x6_Unorm, NVN_FORMAT_RGBA_ASTC_8x6 },
        { ImageFormat_Astc_8x6_UnormSrgb, NVN_FORMAT_RGBA_ASTC_8x6_SRGB },
        { ImageFormat_Astc_8x8_Unorm, NVN_FORMAT_RGBA_ASTC_8x8 },
        { ImageFormat_Astc_8x8_UnormSrgb, NVN_FORMAT_RGBA_ASTC_8x8_SRGB },
        { ImageFormat_Astc_10x5_Unorm, NVN_FORMAT_RGBA_ASTC_10x5 },
        { ImageFormat_Astc_10x5_UnormSrgb, NVN_FORMAT_RGBA_ASTC_10x5_SRGB },
        { ImageFormat_Astc_10x6_Unorm, NVN_FORMAT_RGBA_ASTC_10x6 },
        { ImageFormat_Astc_10x6_UnormSrgb, NVN_FORMAT_RGBA_ASTC_10x6_SRGB },
        { ImageFormat_Astc_10x8_Unorm, NVN_FORMAT_RGBA_ASTC_10x8 },
        { ImageFormat_Astc_10x8_UnormSrgb, NVN_FORMAT_RGBA_ASTC_10x8_SRGB },
        { ImageFormat_Astc_10x10_Unorm, NVN_FORMAT_RGBA_ASTC_10x10 },
        { ImageFormat_Astc_10x10_UnormSrgb, NVN_FORMAT_RGBA_ASTC_10x10_SRGB },
        { ImageFormat_Astc_12x10_Unorm, NVN_FORMAT_RGBA_ASTC_12x10 },
        { ImageFormat_Astc_12x10_UnormSrgb, NVN_FORMAT_RGBA_ASTC_12x10_SRGB },
        { ImageFormat_Astc_12x12_Unorm, NVN_FORMAT_RGBA_ASTC_12x12 },
        { ImageFormat_Astc_12x12_UnormSrgb, NVN_FORMAT_RGBA_ASTC_12x12_SRGB },

        { ImageFormat_B5_G5_R5_A1_Unorm, NVN_FORMAT_BGR5A1 }
    };

    struct AttributeFormatAndNvnFormat
    {
        AttributeFormat attributeFormat;
        NVNformat nvnFormat;
    } s_NvnAttributeFormatList[] =
    {
        // AttributeFormat の昇順
        { AttributeFormat_4_4_Unorm, NVN_FORMAT_NONE }, // No equivalent
        { AttributeFormat_8_Unorm, NVN_FORMAT_R8 },
        { AttributeFormat_8_Snorm, NVN_FORMAT_R8SN },
        { AttributeFormat_8_Uint, NVN_FORMAT_R8UI },
        { AttributeFormat_8_Sint, NVN_FORMAT_R8I },
        { AttributeFormat_8_UintToFloat, NVN_FORMAT_R8_UI2F },
        { AttributeFormat_8_SintToFloat, NVN_FORMAT_R8_I2F },
        { AttributeFormat_8_8_Unorm, NVN_FORMAT_RG8 },
        { AttributeFormat_8_8_Snorm, NVN_FORMAT_RG8SN },
        { AttributeFormat_8_8_Uint, NVN_FORMAT_RG8UI },
        { AttributeFormat_8_8_Sint, NVN_FORMAT_RG8I },
        { AttributeFormat_8_8_UintToFloat, NVN_FORMAT_RG8_UI2F },
        { AttributeFormat_8_8_SintToFloat, NVN_FORMAT_RG8_I2F },
        { AttributeFormat_16_Unorm, NVN_FORMAT_R16 },
        { AttributeFormat_16_Snorm, NVN_FORMAT_R16SN },
        { AttributeFormat_16_Uint, NVN_FORMAT_R16UI },
        { AttributeFormat_16_Sint, NVN_FORMAT_R16I },
        { AttributeFormat_16_Float, NVN_FORMAT_R16F },
        { AttributeFormat_16_UintToFloat, NVN_FORMAT_R16_UI2F },
        { AttributeFormat_16_SintToFloat, NVN_FORMAT_R16_I2F },
        { AttributeFormat_8_8_8_8_Unorm, NVN_FORMAT_RGBA8 },
        { AttributeFormat_8_8_8_8_Snorm, NVN_FORMAT_RGBA8SN },
        { AttributeFormat_8_8_8_8_Uint, NVN_FORMAT_RGBA8UI },
        { AttributeFormat_8_8_8_8_Sint, NVN_FORMAT_RGBA8I },
        { AttributeFormat_8_8_8_8_UintToFloat, NVN_FORMAT_RGBA8_UI2F },
        { AttributeFormat_8_8_8_8_SintToFloat, NVN_FORMAT_RGBA8_I2F },
        { AttributeFormat_10_10_10_2_Unorm, NVN_FORMAT_RGB10A2 },
        { AttributeFormat_10_10_10_2_Snorm, NVN_FORMAT_RGB10A2SN },
        { AttributeFormat_10_10_10_2_Uint, NVN_FORMAT_RGB10A2UI },
        { AttributeFormat_10_10_10_2_Sint, NVN_FORMAT_RGB10A2I },
        { AttributeFormat_16_16_Unorm, NVN_FORMAT_RG16 },
        { AttributeFormat_16_16_Snorm, NVN_FORMAT_RG16SN },
        { AttributeFormat_16_16_Uint, NVN_FORMAT_RG16UI },
        { AttributeFormat_16_16_Sint, NVN_FORMAT_RG16I },
        { AttributeFormat_16_16_Float, NVN_FORMAT_RG16F },
        { AttributeFormat_16_16_UintToFloat, NVN_FORMAT_RG16_UI2F },
        { AttributeFormat_16_16_SintToFloat, NVN_FORMAT_RG16_I2F },
        { AttributeFormat_32_Uint, NVN_FORMAT_R32UI },
        { AttributeFormat_32_Sint, NVN_FORMAT_R32I },
        { AttributeFormat_32_Float, NVN_FORMAT_R32F },
        { AttributeFormat_16_16_16_16_Unorm, NVN_FORMAT_RGBA16 },
        { AttributeFormat_16_16_16_16_Snorm, NVN_FORMAT_RGBA16SN },
        { AttributeFormat_16_16_16_16_Uint, NVN_FORMAT_RGBA16UI },
        { AttributeFormat_16_16_16_16_Sint, NVN_FORMAT_RGBA16I },
        { AttributeFormat_16_16_16_16_Float, NVN_FORMAT_RGBA16F },
        { AttributeFormat_16_16_16_16_UintToFloat, NVN_FORMAT_RGBA16_UI2F },
        { AttributeFormat_16_16_16_16_SintToFloat, NVN_FORMAT_RGBA16_I2F },
        { AttributeFormat_32_32_Uint, NVN_FORMAT_RG32UI },
        { AttributeFormat_32_32_Sint, NVN_FORMAT_RG32I },
        { AttributeFormat_32_32_Float, NVN_FORMAT_RG32F },
        { AttributeFormat_32_32_32_Uint, NVN_FORMAT_RGB32UI },
        { AttributeFormat_32_32_32_Sint, NVN_FORMAT_RGB32I },
        { AttributeFormat_32_32_32_Float, NVN_FORMAT_RGB32F },
        { AttributeFormat_32_32_32_32_Uint, NVN_FORMAT_RGBA32UI },
        { AttributeFormat_32_32_32_32_Sint, NVN_FORMAT_RGBA32I },
        { AttributeFormat_32_32_32_32_Float, NVN_FORMAT_RGBA32F },
    };

    const struct ImageFormatAndProperty
    {
            ImageFormat format;
            ImageFormatProperty property;
    } g_ImageFormatAndPropetyTable[] =
    {
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_NONE
        { ImageFormat_R8_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R8
        { ImageFormat_R8_Snorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R8SN
        { ImageFormat_R8_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R8UI
        { ImageFormat_R8_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R8I
        { ImageFormat_R16_Float, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R16F
        { ImageFormat_R16_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R16
        { ImageFormat_R16_Snorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R16SN
        { ImageFormat_R16_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R16UI
        { ImageFormat_R16_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R16I
        { ImageFormat_R32_Float, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R32F
        { ImageFormat_R32_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R32UI
        { ImageFormat_R32_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R32I
        { ImageFormat_R8_G8_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG8
        { ImageFormat_R8_G8_Snorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG8SN
        { ImageFormat_R8_G8_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG8UI
        { ImageFormat_R8_G8_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG8I
        { ImageFormat_R16_G16_Float, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG16F
        { ImageFormat_R16_G16_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG16
        { ImageFormat_R16_G16_Snorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG16SN
        { ImageFormat_R16_G16_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG16UI
        { ImageFormat_R16_G16_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG16I
        { ImageFormat_R32_G32_Float, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG32F
        { ImageFormat_R32_G32_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG32UI
        { ImageFormat_R32_G32_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RG32I
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB8
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB8SN
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB8UI
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB8I
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB16F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB16
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB16SN
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB16UI
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB16I
        { ImageFormat_R32_G32_B32_Float, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB32F
        { ImageFormat_R32_G32_B32_Uint, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB32UI
        { ImageFormat_R32_G32_B32_Sint, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB32I
        { ImageFormat_R8_G8_B8_A8_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA8
        { ImageFormat_R8_G8_B8_A8_Snorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA8SN
        { ImageFormat_R8_G8_B8_A8_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA8UI
        { ImageFormat_R8_G8_B8_A8_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA8I
        { ImageFormat_R16_G16_B16_A16_Float, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA16F
        { ImageFormat_R16_G16_B16_A16_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA16
        { ImageFormat_R16_G16_B16_A16_Snorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA16SN
        { ImageFormat_R16_G16_B16_A16_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA16UI
        { ImageFormat_R16_G16_B16_A16_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA16I
        { ImageFormat_R32_G32_B32_A32_Float, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA32F
        { ImageFormat_R32_G32_B32_A32_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA32UI
        { ImageFormat_R32_G32_B32_A32_Sint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGBA32I
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_STENCIL8
        { ImageFormat_D16_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_DEPTH16
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_DEPTH24
        { ImageFormat_D32_Float, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_DEPTH32F
        { ImageFormat_D24_Unorm_S8_Uint, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_DEPTH24_STENCIL8
        { ImageFormat_D32_Float_S8_Uint_X24, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_DEPTH32F_STENCIL8
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX8_SRGB
        { ImageFormat_R8_G8_B8_A8_UnormSrgb, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBA8_SRGB
        { ImageFormat_R4_G4_B4_A4_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA4
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB5
        { ImageFormat_R5_G5_B5_A1_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB5A1
        { ImageFormat_R5_G6_B5_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB565
        { ImageFormat_R10_G10_B10_A2_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGB10A2
        { ImageFormat_R10_G10_B10_A2_Uint, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_RGB10A2UI
        { ImageFormat_R11_G11_B10_Float, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } }, // NVN_FORMAT_R11G11B10F
        { ImageFormat_R9_G9_B9_E5_SharedExp, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB9E5F
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB_DXT1
        { ImageFormat_Bc1_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_DXT1
        { ImageFormat_Bc2_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_DXT3
        { ImageFormat_Bc3_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_DXT5
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGB_DXT1_SRGB
        { ImageFormat_Bc1_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_DXT1_SRGB
        { ImageFormat_Bc2_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_DXT3_SRGB
        { ImageFormat_Bc3_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_DXT5_SRGB
        { ImageFormat_Bc4_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGTC1_UNORM
        { ImageFormat_Bc4_Snorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGTC1_SNORM
        { ImageFormat_Bc5_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGTC2_UNORM
        { ImageFormat_Bc5_Snorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGTC2_SNORM
        { ImageFormat_Bc7_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_BPTC_UNORM
        { ImageFormat_Bc7_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_BPTC_UNORM_SRGB
        { ImageFormat_Bc6_Float, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_BPTC_SFLOAT
        { ImageFormat_Bc6_Ufloat, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_BPTC_UFLOAT
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_R8_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_R8_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_R16_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_R16_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_R32_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_R32_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RG8_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RG8_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RG16_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RG16_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RG32_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RG32_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB8_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB8_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB16_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB16_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB32_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB32_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGBA8_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGBA8_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGBA16_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGBA16_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGBA32_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGBA32_I2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB10A2SN
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB10A2I
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB10A2_UI2F
        { ImageFormat_Undefined, { 0 } }, // NVN_FORMAT_RGB10A2_I2F
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX8
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX8SN
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX8UI
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX8I
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX16F
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX16
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX16SN
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX16UI
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX16I
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX32F
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX32UI
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_RGBX32I
        { ImageFormat_Astc_4x4_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_4x4
        { ImageFormat_Astc_5x4_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_5x4
        { ImageFormat_Astc_5x5_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_5x5
        { ImageFormat_Astc_6x5_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_6x5
        { ImageFormat_Astc_6x6_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_6x6
        { ImageFormat_Astc_8x5_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_8x5
        { ImageFormat_Astc_8x6_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_8x6
        { ImageFormat_Astc_8x8_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_8x8
        { ImageFormat_Astc_10x5_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x5
        { ImageFormat_Astc_10x6_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x6
        { ImageFormat_Astc_10x8_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x8
        { ImageFormat_Astc_10x10_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x10
        { ImageFormat_Astc_12x10_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_12x10
        { ImageFormat_Astc_12x12_Unorm, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_12x12
        { ImageFormat_Astc_4x4_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_4x4_SRGB
        { ImageFormat_Astc_5x4_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_5x4_SRGB
        { ImageFormat_Astc_5x5_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_5x5_SRGB
        { ImageFormat_Astc_6x5_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_6x5_SRGB
        { ImageFormat_Astc_6x6_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_6x6_SRGB
        { ImageFormat_Astc_8x5_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_8x5_SRGB
        { ImageFormat_Astc_8x6_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_8x6_SRGB
        { ImageFormat_Astc_8x8_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_8x8_SRGB
        { ImageFormat_Astc_10x5_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x5_SRGB
        { ImageFormat_Astc_10x6_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x6_SRGB
        { ImageFormat_Astc_10x8_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x8_SRGB
        { ImageFormat_Astc_10x10_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_10x10_SRGB
        { ImageFormat_Astc_12x10_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_12x10_SRGB
        { ImageFormat_Astc_12x12_UnormSrgb, { ImageFormatPropertyFlag_Texture } }, // NVN_FORMAT_RGBA_ASTC_12x12_SRGB

        { ImageFormat_B5_G6_R5_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_BGR565
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_BGR5
        { ImageFormat_B5_G5_R5_A1_Unorm, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }, // NVN_FORMAT_BGR5A1
        { ImageFormat_A1_B5_G5_R5_Unorm, { ImageFormatPropertyFlag_Texture} },// NVN_FORMAT_A1BGR5
        { ImageFormat_Undefined, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },// NVN_FORMAT_BGRX8
        { ImageFormat_B8_G8_R8_A8_Unorm,{ ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget | ImageFormatPropertyFlag_Image } },// NVN_FORMAT_BGRA8
        { ImageFormat_Undefined,{ ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },// NVN_FORMAT_BGRX8_SRGB
        { ImageFormat_B8_G8_R8_A8_UnormSrgb,{ ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }// NVN_FORMAT_BGRA8_SRGB
    };
}

NVNformat Nvn::GetImageFormat( ImageFormat format ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const ImageFormatAndNvnFormat& lhs, ImageFormat rhs ) const
        {
            return lhs.imageFormat < rhs;
        }
    };
    ImageFormatAndNvnFormat* pEnd = s_NvnTextureFormatList + NN_GFX_ARRAY_LENGTH( s_NvnTextureFormatList );
    ImageFormatAndNvnFormat* pFound = std::lower_bound( s_NvnTextureFormatList, pEnd, format, Comp() );
    if ( pFound == pEnd || format < pFound->imageFormat )
    {
        return NVN_FORMAT_NONE;
    }
    else
    {
        return pFound->nvnFormat;
    }
}

NVNformat Nvn::GetAttributeFormat( AttributeFormat format ) NN_NOEXCEPT
{
    struct Comp
    {
        bool operator()( const AttributeFormatAndNvnFormat& lhs, AttributeFormat rhs ) const
        {
            return lhs.attributeFormat < rhs;
        }
    };
    AttributeFormatAndNvnFormat* pEnd = s_NvnAttributeFormatList + NN_GFX_ARRAY_LENGTH( s_NvnAttributeFormatList );
    AttributeFormatAndNvnFormat* pFound = std::lower_bound( s_NvnAttributeFormatList, pEnd, format, Comp() );
    if ( pFound == pEnd || format < pFound->attributeFormat )
    {
        return NVN_FORMAT_NONE;
    }
    else
    {
        return pFound->nvnFormat;
    }
}

NVNtextureTarget Nvn::GetImageTarget( ImageDimension dimension ) NN_NOEXCEPT
{
    static const NVNtextureTarget s_TargetTable[] =
    {
        NVN_TEXTURE_TARGET_1D,
        NVN_TEXTURE_TARGET_2D,
        NVN_TEXTURE_TARGET_3D,
        NVN_TEXTURE_TARGET_CUBEMAP,
        NVN_TEXTURE_TARGET_1D_ARRAY,
        NVN_TEXTURE_TARGET_2D_ARRAY,
        NVN_TEXTURE_TARGET_2D_MULTISAMPLE,
        NVN_TEXTURE_TARGET_2D_MULTISAMPLE_ARRAY,
        NVN_TEXTURE_TARGET_CUBEMAP_ARRAY,
        NVN_TEXTURE_TARGET_RECTANGLE
    };

    return s_TargetTable[ dimension ];
}

NVNdepthFunc Nvn::GetDepthFunction( ComparisonFunction compare ) NN_NOEXCEPT
{
    static const NVNdepthFunc s_DepthFunction[] =
    {
        NVN_DEPTH_FUNC_NEVER,
        NVN_DEPTH_FUNC_LESS,
        NVN_DEPTH_FUNC_EQUAL,
        NVN_DEPTH_FUNC_LEQUAL,
        NVN_DEPTH_FUNC_GREATER,
        NVN_DEPTH_FUNC_NOTEQUAL,
        NVN_DEPTH_FUNC_GEQUAL,
        NVN_DEPTH_FUNC_ALWAYS,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_DepthFunction ) == ComparisonFunction_End );
    NN_SDK_ASSERT( compare < ComparisonFunction_End );

    return s_DepthFunction[ compare ];
}

NVNstencilOp Nvn::GetStencilOperation( StencilOperation operation ) NN_NOEXCEPT
{
    static const NVNstencilOp s_StencilOperation[] =
    {
        NVN_STENCIL_OP_KEEP,
        NVN_STENCIL_OP_ZERO,
        NVN_STENCIL_OP_REPLACE,
        NVN_STENCIL_OP_INCR,
        NVN_STENCIL_OP_DECR,
        NVN_STENCIL_OP_INVERT,
        NVN_STENCIL_OP_INCR_WRAP,
        NVN_STENCIL_OP_DECR_WRAP,
    };

    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_StencilOperation ) == StencilOperation_End );
    NN_SDK_ASSERT( operation < StencilOperation_End );

    return s_StencilOperation[ operation ];
}

NVNstencilFunc Nvn::GetStencilFunction( ComparisonFunction compare ) NN_NOEXCEPT
{
    static const NVNstencilFunc s_StencilFunction[] =
    {
        NVN_STENCIL_FUNC_NEVER,
        NVN_STENCIL_FUNC_LESS,
        NVN_STENCIL_FUNC_EQUAL,
        NVN_STENCIL_FUNC_LEQUAL,
        NVN_STENCIL_FUNC_GREATER,
        NVN_STENCIL_FUNC_NOTEQUAL,
        NVN_STENCIL_FUNC_GEQUAL,
        NVN_STENCIL_FUNC_ALWAYS,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_StencilFunction ) == ComparisonFunction_End );
    NN_SDK_ASSERT( compare < ComparisonFunction_End );

    return s_StencilFunction[ compare ];
}

NVNblendEquation Nvn::GetBlendEquation( BlendFunction function ) NN_NOEXCEPT
{
    static const NVNblendEquation s_BlendEquation[] =
    {
        NVN_BLEND_EQUATION_ADD,
        NVN_BLEND_EQUATION_SUB,
        NVN_BLEND_EQUATION_REVERSE_SUB,
        NVN_BLEND_EQUATION_MIN,
        NVN_BLEND_EQUATION_MAX,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_BlendEquation ) == BlendFunction_End );
    NN_SDK_ASSERT( function < BlendFunction_End );

    return s_BlendEquation[ function ];
}

NVNblendFunc Nvn::GetBlendFunction( BlendFactor factor ) NN_NOEXCEPT
{
    static const NVNblendFunc s_BlendFunction[] =
    {
        NVN_BLEND_FUNC_ZERO,
        NVN_BLEND_FUNC_ONE,
        NVN_BLEND_FUNC_SRC_COLOR,
        NVN_BLEND_FUNC_ONE_MINUS_SRC_COLOR,
        NVN_BLEND_FUNC_DST_COLOR,
        NVN_BLEND_FUNC_ONE_MINUS_DST_COLOR,
        NVN_BLEND_FUNC_SRC_ALPHA,
        NVN_BLEND_FUNC_ONE_MINUS_SRC_ALPHA,
        NVN_BLEND_FUNC_DST_ALPHA,
        NVN_BLEND_FUNC_ONE_MINUS_DST_ALPHA,
        NVN_BLEND_FUNC_CONSTANT_COLOR,
        NVN_BLEND_FUNC_ONE_MINUS_CONSTANT_COLOR,
        NVN_BLEND_FUNC_CONSTANT_ALPHA,
        NVN_BLEND_FUNC_ONE_MINUS_CONSTANT_ALPHA,
        NVN_BLEND_FUNC_SRC_ALPHA_SATURATE,
        NVN_BLEND_FUNC_SRC1_COLOR,
        NVN_BLEND_FUNC_ONE_MINUS_SRC1_COLOR,
        NVN_BLEND_FUNC_SRC1_ALPHA,
        NVN_BLEND_FUNC_ONE_MINUS_SRC1_ALPHA,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_BlendFunction ) == BlendFactor_End );
    NN_SDK_ASSERT( factor < BlendFactor_End );

    return s_BlendFunction[ factor ];
}

NVNlogicOp Nvn::GetLogicOperation( LogicOperation operation ) NN_NOEXCEPT
{
    static const NVNlogicOp s_LogicOperation[] =
    {
        NVN_LOGIC_OP_CLEAR,
        NVN_LOGIC_OP_AND,
        NVN_LOGIC_OP_AND_REVERSE,
        NVN_LOGIC_OP_COPY,
        NVN_LOGIC_OP_AND_INVERTED,
        NVN_LOGIC_OP_NOOP,
        NVN_LOGIC_OP_XOR,
        NVN_LOGIC_OP_OR,
        NVN_LOGIC_OP_NOR,
        NVN_LOGIC_OP_EQUIV,
        NVN_LOGIC_OP_INVERT,
        NVN_LOGIC_OP_OR_REVERSE,
        NVN_LOGIC_OP_COPY_INVERTED,
        NVN_LOGIC_OP_OR_INVERTED,
        NVN_LOGIC_OP_NAND,
        NVN_LOGIC_OP_SET,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_LogicOperation ) == LogicOperation_End );
    NN_SDK_ASSERT( operation < LogicOperation_End );

    return s_LogicOperation[ operation ];
}

NVNfrontFace Nvn::GetFrontFace( FrontFace face ) NN_NOEXCEPT
{
    static const NVNfrontFace s_FrontFace[] =
    {
        NVN_FRONT_FACE_CCW,
        NVN_FRONT_FACE_CW,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_FrontFace ) == FrontFace_End );
    NN_SDK_ASSERT( face < FrontFace_End );

    return s_FrontFace[ face ];
}

NVNpolygonMode Nvn::GetFillMode( FillMode mode ) NN_NOEXCEPT
{
    static const NVNpolygonMode s_FillMode[] =
    {
        NVN_POLYGON_MODE_POINT,
        NVN_POLYGON_MODE_LINE,
        NVN_POLYGON_MODE_FILL,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_FillMode ) == FillMode_End );
    NN_SDK_ASSERT( mode < FillMode_End );

    return s_FillMode[ mode ];
}

NVNdrawPrimitive Nvn::GetDrawPrimitive( PrimitiveTopology topology ) NN_NOEXCEPT
{
    static const NVNdrawPrimitive s_DrawPrimitive[] =
    {
        NVN_DRAW_PRIMITIVE_POINTS,
        NVN_DRAW_PRIMITIVE_LINES,
        NVN_DRAW_PRIMITIVE_LINE_STRIP,
        NVN_DRAW_PRIMITIVE_TRIANGLES,
        NVN_DRAW_PRIMITIVE_TRIANGLE_STRIP,
        NVN_DRAW_PRIMITIVE_LINES_ADJACENCY,
        NVN_DRAW_PRIMITIVE_LINE_STRIP_ADJACENCY,
        NVN_DRAW_PRIMITIVE_TRIANGLES_ADJACENCY,
        NVN_DRAW_PRIMITIVE_TRIANGLE_STRIP_ADJACENCY,
        NVN_DRAW_PRIMITIVE_PATCHES
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_DrawPrimitive ) == PrimitiveTopology_End );
    NN_SDK_ASSERT( topology < PrimitiveTopology_End );

    return s_DrawPrimitive[ topology ];
}

NVNminFilter Nvn::GetMinFilter( FilterMode filterMode ) NN_NOEXCEPT
{
    switch( ( filterMode & FilterModeBit_MinFilterMask ) >> FilterModeBit_MinFilterShift )
    {
    case FilterModeBit_Point:
        {
            switch( ( filterMode & FilterModeBit_MipFilterMask ) >> FilterModeBit_MipFilterShift )
            {
            case FilterModeBit_Point:
                return NVN_MIN_FILTER_NEAREST_MIPMAP_NEAREST;
            case FilterModeBit_Linear:
                return NVN_MIN_FILTER_NEAREST_MIPMAP_LINEAR;
            default:
                return NVN_MIN_FILTER_NEAREST;
            }
        }
        break;
    case FilterModeBit_Linear:
        {
            switch( ( filterMode & FilterModeBit_MipFilterMask ) >> FilterModeBit_MipFilterShift )
            {
            case FilterModeBit_Point:
                return NVN_MIN_FILTER_LINEAR_MIPMAP_NEAREST;
            case FilterModeBit_Linear:
                return NVN_MIN_FILTER_LINEAR_MIPMAP_LINEAR;
            default:
                return NVN_MIN_FILTER_LINEAR;
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

NVNmagFilter Nvn::GetMagFilter( FilterMode filterMode ) NN_NOEXCEPT
{
    switch( ( filterMode & FilterModeBit_MagFilterMask ) >> FilterModeBit_MagFilterShift )
    {
    case FilterModeBit_Point:
        return NVN_MAG_FILTER_NEAREST;
    case FilterModeBit_Linear:
        return NVN_MAG_FILTER_LINEAR;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

NVNwrapMode Nvn::GetWrapMode( TextureAddressMode wrapMode ) NN_NOEXCEPT
{
    static const NVNwrapMode s_WrapMode[] =
    {
        NVN_WRAP_MODE_REPEAT,
        NVN_WRAP_MODE_MIRRORED_REPEAT,
        NVN_WRAP_MODE_CLAMP_TO_EDGE,
        NVN_WRAP_MODE_CLAMP_TO_BORDER,
        NVN_WRAP_MODE_MIRROR_CLAMP_TO_EDGE,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_WrapMode ) == TextureAddressMode_End );
    NN_SDK_ASSERT( wrapMode < TextureAddressMode_End );

    return s_WrapMode[ wrapMode ];
}

NVNshaderStage Nvn::GetShaderStage( ShaderStage stage ) NN_NOEXCEPT
{
    static const NVNshaderStage s_Stage[] =
    {
        NVN_SHADER_STAGE_VERTEX,
        NVN_SHADER_STAGE_TESS_CONTROL,    /* hull shader */
        NVN_SHADER_STAGE_TESS_EVALUATION, /* domain shader */
        NVN_SHADER_STAGE_GEOMETRY,
        NVN_SHADER_STAGE_FRAGMENT,
        NVN_SHADER_STAGE_COMPUTE
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_Stage ) == ShaderStage_End );
    NN_SDK_ASSERT( stage < ShaderStage_End );

    return s_Stage[ stage ];
}

int Nvn::GetShaderStageBits( int shaderStageBits ) NN_NOEXCEPT
{
    static const NVNshaderStageBits s_ShaderStageBitTable[] =
    {
        NVN_SHADER_STAGE_VERTEX_BIT,
        NVN_SHADER_STAGE_TESS_CONTROL_BIT,
        NVN_SHADER_STAGE_TESS_EVALUATION_BIT,
        NVN_SHADER_STAGE_GEOMETRY_BIT,
        NVN_SHADER_STAGE_FRAGMENT_BIT,
        NVN_SHADER_STAGE_COMPUTE_BIT
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_ShaderStageBitTable ) == ShaderStage_End );

    int ret = 0;
    for( int idxStage = 0, mask = 0x01; idxStage < ShaderStage_End; ++idxStage, mask <<= 1 )
    {
        if( shaderStageBits & mask )
        {
            ret |= s_ShaderStageBitTable[ idxStage ];
        }
    }
    return ret;
}

NVNcompareFunc Nvn::GetRComparisonFunction( ComparisonFunction compare ) NN_NOEXCEPT
{
    static const NVNcompareFunc s_ComparisonFunction[] =
    {
        NVN_COMPARE_FUNC_NEVER,
        NVN_COMPARE_FUNC_LESS,
        NVN_COMPARE_FUNC_EQUAL,
        NVN_COMPARE_FUNC_LEQUAL,
        NVN_COMPARE_FUNC_GREATER,
        NVN_COMPARE_FUNC_NOTEQUAL,
        NVN_COMPARE_FUNC_GEQUAL,
        NVN_COMPARE_FUNC_ALWAYS,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_ComparisonFunction ) == ComparisonFunction_End );
    NN_SDK_ASSERT( compare < ComparisonFunction_End );

    return s_ComparisonFunction[ compare ];
}

NVNindexType Nvn::GetIndexFormat( IndexFormat indexFormat ) NN_NOEXCEPT
{
    static const NVNindexType s_IndexTable[] =
    {
        NVN_INDEX_TYPE_UNSIGNED_BYTE,
        NVN_INDEX_TYPE_UNSIGNED_SHORT,
        NVN_INDEX_TYPE_UNSIGNED_INT,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_IndexTable ) == IndexFormat_End );
    NN_SDK_ASSERT( indexFormat < IndexFormat_End );

    return s_IndexTable[ indexFormat ];
}

NVNface Nvn::GetCullMode( CullMode mode ) NN_NOEXCEPT
{
    static const NVNface s_Face[] =
    {
        NVN_FACE_NONE,
        NVN_FACE_FRONT,
        NVN_FACE_BACK,
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_Face ) == CullMode_End );
    NN_SDK_ASSERT( mode < CullMode_End );

    return s_Face[ mode ];
}

int Nvn::GetMemoryPoolFlags( int memoryPoolProperty ) NN_NOEXCEPT
{
    // nvn と同じ定義値
    return memoryPoolProperty;
}

NVNbufferAddress Nvn::GetBufferAddress(const GpuAddress gpuAddress) NN_NOEXCEPT
{
    return gpuAddress.ToData()->value;
}

void Nvn::SetupScanBufferTextureInfo( TextureInfo* pOutInfo, const SwapChainInfo& info ) NN_NOEXCEPT
{
    pOutInfo->SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    pOutInfo->SetTileMode( nn::gfx::TileMode_Optimal );
    pOutInfo->SetSwizzle( 0 );
    pOutInfo->SetMipCount( 1 );
    pOutInfo->SetMultiSampleCount( 1 );
    pOutInfo->SetImageFormat( info.GetFormat() );
    pOutInfo->SetGpuAccessFlags( nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_ScanBuffer );
    pOutInfo->SetWidth( info.GetWidth() );
    pOutInfo->SetHeight( info.GetHeight() );
    pOutInfo->SetDepth( 1 );
    pOutInfo->SetArrayLength( 1 );
}

NVNcounterType Nvn::GetCounterType( QueryTarget target ) NN_NOEXCEPT
{
    static const NVNcounterType s_CounterTypeTable[] =
    {
        NVN_COUNTER_TYPE_TIMESTAMP,
        NVN_COUNTER_TYPE_SAMPLES_PASSED,
        NVN_COUNTER_TYPE_INPUT_VERTICES,
        NVN_COUNTER_TYPE_INPUT_PRIMITIVES,
        NVN_COUNTER_TYPE_VERTEX_SHADER_INVOCATIONS,
        NVN_COUNTER_TYPE_GEOMETRY_SHADER_INVOCATIONS,
        NVN_COUNTER_TYPE_GEOMETRY_SHADER_PRIMITIVES,
        NVN_COUNTER_TYPE_CLIPPER_INPUT_PRIMITIVES,
        NVN_COUNTER_TYPE_CLIPPER_OUTPUT_PRIMITIVES,
        NVN_COUNTER_TYPE_FRAGMENT_SHADER_INVOCATIONS,
        NVN_COUNTER_TYPE_TESS_CONTROL_SHADER_INVOCATIONS,
        NVN_COUNTER_TYPE_TESS_EVALUATION_SHADER_INVOCATIONS,
        static_cast< NVNcounterType >( -1 ) // ComputeShaderInvocations
    };
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_CounterTypeTable ) == QueryTarget_End );
    NN_SDK_ASSERT( target != QueryTarget_ComputeShaderInvocations && target < QueryTarget_End );

    return s_CounterTypeTable[ target ];
}

nn::util::BitPack32 Nvn::GetDeviceFeature( const NVNdevice* pDevice ) NN_NOEXCEPT
{
    nn::util::BitPack32 ret;
    ret.Clear();

    int isSupported;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_MIN_MAX_FILTERING, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportMinMaxFiltering, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_STENCIL8_FORMAT, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportStencil8Format, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_ASTC_FORMATS, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportAstcFormat, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_CONSERVATIVE_RASTER, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportConservativeRaster, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_ZERO_FROM_UNMAPPED_VIRTUAL_POOL_PAGES, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportZeroFromUnmappedVirtualPoolPage, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_PASSTHROUGH_GEOMETRY_SHADERS, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportPassthroughGeometryShader, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_VIEWPORT_SWIZZLE, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportViewportSwizzle, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_SPARSE_TILED_PACKAGED_TEXTURES, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportSparseTiledPackagedTexture, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_ADVANCED_BLEND_MODES, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_AdvancedBlendModes, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_DRAW_TEXTURE, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_DrawTexture, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_TARGET_INDEPENDENT_RASTERIZATION, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_TargetIndependentRasterization, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_FRAGMENT_COVERAGE_TO_COLOR, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_FragmentCoverageToColor, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_POST_DEPTH_COVERAGE, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_PostDepthCoverage, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_IMAGES_USING_TEXTURE_HANDLES, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_ImagesUsingTextureHandles, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_SAMPLE_LOCATIONS, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SampleLocations, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_FRAGMENT_SHADER_INTERLOCK, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportFragmentShaderInterlock, isSupported != 0 );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger( pDevice,
        NVN_DEVICE_INFO_SUPPORTS_DEBUG_LAYER, &isSupported ) );
    ret.SetBit( NvnDeviceFeature_SupportsDebugLayer, isSupported != 0 );

    return ret;
}

void Nvn::GetImageFormatProperty( ImageFormatProperty* pOutImageFormatProperty,
    NVNformat nvnFormat ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL( pOutImageFormatProperty );

    *pOutImageFormatProperty = g_ImageFormatAndPropetyTable[ nvnFormat ].property;
}

ImageFormat Nvn::GetGfxImageFormat( NVNformat nvnFormat ) NN_NOEXCEPT
{
    return g_ImageFormatAndPropetyTable[ nvnFormat ].format;
}

void NVNAPIENTRY Nvn::DebugCallback( NVNdebugCallbackSource source, NVNdebugCallbackType type,
    int id, NVNdebugCallbackSeverity severity, const char* message, void* )
{
    static const char* s_DebugCallbackSourceStrings[] =
    {
        "API"
    };
    static const char* s_DebugCallbackSeverityStrings[] =
    {
        "High",
        "Medium",
        "Low",
        "Notification"
    };
    switch( type )
    {
    case NVN_DEBUG_CALLBACK_TYPE_API_ERROR:
    {
        NN_DETAIL_GFX_ERROR( "[Debug Callback] Error (%08x) (Source: %s) (Severity: %s):\n   %s\n",
            id,
            s_DebugCallbackSourceStrings[ source ],
            s_DebugCallbackSeverityStrings[ severity ],
            message );
    }
    break;
    case NVN_DEBUG_CALLBACK_TYPE_API_WARNING:
    {
        NN_DETAIL_GFX_WARN( "[Debug Callback] Warning (%08x) (Source: %s) (Severity: %s):\n   %s\n",
            id,
            s_DebugCallbackSourceStrings[ source ],
            s_DebugCallbackSeverityStrings[ severity ],
            message );
    }
    break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_UNUSED( s_DebugCallbackSourceStrings );
    NN_UNUSED( s_DebugCallbackSeverityStrings );
    NN_UNUSED( source );
    NN_UNUSED( type );
    NN_UNUSED( id );
    NN_UNUSED( severity );
    NN_UNUSED( message );
}

GlslcDll& GlslcDll::GetInstance() NN_NOEXCEPT
{
    static GlslcDll s_GlslcDll;

    if( !s_GlslcDll.IsInitialized() )
    {
        bool isSucceeded = s_GlslcDll.Initialize();
        if( !isSucceeded )
        {
            NN_DETAIL_GFX_ERROR( "could not load glslc.\n" );
            NN_SDK_ASSERT( 0 );
        }
    }

    return s_GlslcDll;
}

GlslcDll::GlslcDll() NN_NOEXCEPT
    : GlslcCompilePreSpecialized()
    , GlslcCompileSpecialized()
    , GlslcInitialize()
    , GlslcFinalize()
    , GlslcCompile()
    , GlslcGetVersion()
    , GlslcSetAllocator()
    , GlslcGetDefaultOptions()
    , m_hDll()
{
}

GlslcDll::~GlslcDll() NN_NOEXCEPT
{
    Finalize();
}

}
}
}
