﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/gfx/gfx_MemoryPoolInfo.h>

#include <nn/gfx/detail/gfx_MemoryPool-api.nvn.8.h>
#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

size_t MemoryPoolImpl< Target >::GetPoolMemoryAlignment(
    DeviceImpl< Target >*, const InfoType& ) NN_NOEXCEPT
{
    return NVN_MEMORY_POOL_STORAGE_ALIGNMENT;
}

size_t MemoryPoolImpl< Target >::GetPoolMemorySizeGranularity(
    DeviceImpl< Target >*, const InfoType& ) NN_NOEXCEPT
{
    return NVN_MEMORY_POOL_STORAGE_GRANULARITY;
}

MemoryPoolImpl< Target >::MemoryPoolImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

MemoryPoolImpl< Target >::~MemoryPoolImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized || this->flags.GetBit( Flag_Shared ) );
}

void MemoryPoolImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnMemoryPool ) == sizeof( NVNmemoryPool ) );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_ASSERT_NOT_NULL( pDevice->ToData()->pNvnDevice.ptr );
    NN_SDK_ASSERT_ALIGNED( info.GetPoolMemory(), GetPoolMemoryAlignment( pDevice, info ) );
    NN_SDK_ASSERT_ALIGNED( info.GetPoolMemorySize(), GetPoolMemorySizeGranularity( pDevice, info ) );

    NN_SDK_ASSERT( IsValidMemoryPoolProperty( info.GetMemoryPoolProperty() ) );

    int memoryPoolFlags = Nvn::GetMemoryPoolFlags( info.GetMemoryPoolProperty() );

    this->pNvnMemoryPool = &this->nvnMemoryPool;
    this->pMemory = info.GetPoolMemory();

    NVNmemoryPoolBuilder builder;
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetDevice(
        &builder, pDevice->ToData()->pNvnDevice ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetDefaults( &builder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetFlags( &builder, memoryPoolFlags ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolBuilderSetStorage(
        &builder, info.GetPoolMemory(), info.GetPoolMemorySize() ) );

    NVNboolean isMemoryPoolOk = NN_GFX_CALL_NVN_FUNCTION(
        nvnMemoryPoolInitialize( this->pNvnMemoryPool, &builder ) );
    NN_SDK_ASSERT( isMemoryPoolOk );
    NN_UNUSED( isMemoryPoolOk );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void MemoryPoolImpl< Target >::Finalize( DeviceImpl< Target >* ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

    NN_GFX_CALL_NVN_FUNCTION(
        nvnMemoryPoolFinalize( this->pNvnMemoryPool ) );

    this->pMemory = NULL;

    this->state = State_NotInitialized;
}

void* MemoryPoolImpl< Target >::Map() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    return NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolMap( this->pNvnMemoryPool ) );
}

void MemoryPoolImpl< Target >::Unmap() const NN_NOEXCEPT
{
    // nvn に Unmap 関数なし
}

void MemoryPoolImpl< Target >::FlushMappedRange( ptrdiff_t offset, size_t size ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolFlushMappedRange( this->pNvnMemoryPool, offset, size ) );
}

void MemoryPoolImpl< Target >::InvalidateMappedRange( ptrdiff_t offset, size_t size ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_GFX_CALL_NVN_FUNCTION( nvnMemoryPoolInvalidateMappedRange( this->pNvnMemoryPool, offset, size ) );
}

}
}
}
