﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_SdkLog.h>

#include <nn/gfx/gfx_Common.h>
#include <nn/gfx/gfx_CommandBuffer.h>

#include <nn/gfx/detail/gfx_Misc.h>

#include "gfx_GxHelper.h"
#include "gfx_CommonHelper.h"

namespace nn {
namespace gfx {
namespace detail {

namespace {
    int g_SwapInterval = 1;

    const GX2StencilFunction stencilOpTable[] =
    {
        GX2_STENCIL_KEEP,
        GX2_STENCIL_ZERO,
        GX2_STENCIL_REPLACE,
        GX2_STENCIL_INCR,
        GX2_STENCIL_DECR,
        GX2_STENCIL_INVERT,
        GX2_STENCIL_INCR_WRAP,
        GX2_STENCIL_DECR_WRAP
    };

    const GX2PolygonMode polygonModeTable[] =
    {
        GX2_POLYGON_MODE_POINT, // Not defined currently
        GX2_POLYGON_MODE_LINE,  // FillMode_Wireframe = 1
        GX2_POLYGON_MODE_TRIANGLE, // FillMode_Solid
    };

    const GX2PrimitiveType primitiveTable[] =
    {
        GX2_PRIMITIVE_POINTS,                  // PrimitiveTopology_PointList
        GX2_PRIMITIVE_LINES,                   // PrimitiveTopology_LineList
        GX2_PRIMITIVE_LINE_STRIP,              // PrimitiveTopology_LineStrip
        GX2_PRIMITIVE_TRIANGLES,               // PrimitiveTopology_TriangleList
        GX2_PRIMITIVE_TRIANGLE_STRIP,          // PrimitiveTopology_TriangleStrip
        GX2_PRIMITIVE_LINES_ADJACENCY,         // PrimitiveTopology_LineListAdjacency
        GX2_PRIMITIVE_LINE_STRIP_ADJACENCY,    // PrimitiveTopology_LineStripAdjacency
        GX2_PRIMITIVE_TRIANGLES_ADJACENCY,     // PrimitiveTopology_TriangleListAdjacency
        GX2_PRIMITIVE_TRIANGLE_STRIP_ADJACENCY, // PrimitiveTopology_TriangleStripAdjacency
        static_cast< GX2PrimitiveType >( 0 )
    };

    const GX2LogicOp logicOperationTable[] =
    {
        GX2_LOGIC_OP_CLEAR,     // LogicOperation_Clear
        GX2_LOGIC_OP_AND,       // LogicOperation_And
        GX2_LOGIC_OP_REVAND,    // LogicOperation_AndReverse
        GX2_LOGIC_OP_COPY,      // LogicOperation_Copy
        GX2_LOGIC_OP_INVAND,    // LogicOperation_AndInverted
        GX2_LOGIC_OP_NOOP,      // LogicOperation_NoOp
        GX2_LOGIC_OP_XOR,       // LogicOperation_Xor
        GX2_LOGIC_OP_OR,        // LogicOperation_Or
        GX2_LOGIC_OP_NOR,       // LogicOperation_Nor
        GX2_LOGIC_OP_EQUIV,     // LogicOperation_Equiv
        GX2_LOGIC_OP_INV,       // LogicOperation_Invert
        GX2_LOGIC_OP_REVOR,     // LogicOperation_OrReverse
        GX2_LOGIC_OP_INVCOPY,   // LogicOperation_CopyInverted
        GX2_LOGIC_OP_INVOR,     // LogicOperation_OrInverted
        GX2_LOGIC_OP_NAND,      // LogicOperation_Nand
        GX2_LOGIC_OP_SET,       // LogicOperation_Set
    };

    const GX2BlendFunction blendFactorTable[] =
    {
        GX2_BLEND_ZERO,                     // BlendFactor_Zero
        GX2_BLEND_ONE,                      // BlendFactor_One
        GX2_BLEND_SRC_COLOR,                // BlendFactor_SourceColor
        GX2_BLEND_ONE_MINUS_SRC_COLOR,      // BlendFactor_OneMinusSourceColor
        GX2_BLEND_DST_COLOR,                // BlendFactor_DestinationColor
        GX2_BLEND_ONE_MINUS_DST_COLOR,      // BlendFactor_OneMinusDestinationColor
        GX2_BLEND_SRC_ALPHA,                // BlendFactor_SourceAlpha
        GX2_BLEND_ONE_MINUS_SRC_ALPHA,      // BlendFactor_OneMinusSourceAlpha
        GX2_BLEND_DST_ALPHA,                // BlendFactor_DestinationAlpha
        GX2_BLEND_ONE_MINUS_DST_ALPHA,      // BlendFactor_OneMinusDestinationAlpha
        GX2_BLEND_CONSTANT_COLOR,           // BlendFactor_ConstantColor
        GX2_BLEND_ONE_MINUS_CONSTANT_COLOR, // BlendFactor_OneMinusConstantColor
        GX2_BLEND_CONSTANT_ALPHA,           // BlendFactor_ConstantAlpha
        GX2_BLEND_ONE_MINUS_CONSTANT_ALPHA, // BlendFactor_OneMinusConstantAlpha
        GX2_BLEND_SRC_ALPHA_SATURATE,       // BlendFactor_SourceAlphaSaturate
        GX2_BLEND_SRC1_COLOR,               // BlendFactor_Source1Color
        GX2_BLEND_ONE_MINUS_SRC1_COLOR,     // BlendFactor_OneMinusSource1Color
        GX2_BLEND_SRC1_ALPHA,               // BlendFactor_Source1Alpha
        GX2_BLEND_ONE_MINUS_SRC1_ALPHA,     // BlendFactor_OneMinusSource1Alpha
    };

    const GX2BlendCombine blendCombineTable[] =
    {
        GX2_BLEND_COMBINE_ADD,              // BlendFunction_Add
        GX2_BLEND_COMBINE_SRC_MINUS_DST,    // BlendFunction_Subtract
        GX2_BLEND_COMBINE_DST_MINUS_SRC,    // BlendFunction_ReverseSubtract
        GX2_BLEND_COMBINE_MIN,              // BlendFunction_Min
        GX2_BLEND_COMBINE_MAX,              // BlendFunction_Max
    };

    struct ImageFormatAndGxFormat
    {
        ImageFormat imageFormat;
        GX2SurfaceFormat gxFormat;
    };

    ImageFormatAndGxFormat s_GxTextureFormatList[] =
    {
        { ImageFormat_R8_Unorm, GX2_SURFACE_FORMAT_TC_R8_UNORM },
        { ImageFormat_R8_Snorm, GX2_SURFACE_FORMAT_TC_R8_SNORM },
        { ImageFormat_R8_Uint, GX2_SURFACE_FORMAT_TC_R8_UINT },
        { ImageFormat_R8_Sint, GX2_SURFACE_FORMAT_TC_R8_SINT },
        { ImageFormat_R4_G4_B4_A4_Unorm, GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM },
        { ImageFormat_R5_G5_B5_A1_Unorm, GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM },
        { ImageFormat_A1_B5_G5_R5_Unorm, GX2_SURFACE_FORMAT_TC_A1_B5_G5_R5_UNORM },
        { ImageFormat_R5_G6_B5_Unorm, GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM },
        // Not supported in Cafe { ImageFormat_B5_G6_R5_Unorm, GX2_SURFACE_FORMAT },
        { ImageFormat_R8_G8_Unorm, GX2_SURFACE_FORMAT_TC_R8_G8_UNORM },
        { ImageFormat_R8_G8_Snorm, GX2_SURFACE_FORMAT_TC_R8_G8_SNORM },
        { ImageFormat_R8_G8_Uint, GX2_SURFACE_FORMAT_TC_R8_G8_UINT },
        { ImageFormat_R8_G8_Sint, GX2_SURFACE_FORMAT_TC_R8_G8_SINT },
        { ImageFormat_R16_Unorm, GX2_SURFACE_FORMAT_TCD_R16_UNORM },
        { ImageFormat_R16_Snorm, GX2_SURFACE_FORMAT_TC_R16_SNORM },
        { ImageFormat_R16_Uint, GX2_SURFACE_FORMAT_TC_R16_UINT },
        { ImageFormat_R16_Sint, GX2_SURFACE_FORMAT_TC_R16_SINT },
        { ImageFormat_R16_Float, GX2_SURFACE_FORMAT_TC_R16_FLOAT },
        { ImageFormat_D16_Unorm, GX2_SURFACE_FORMAT_TCD_R16_UNORM },
        { ImageFormat_R8_G8_B8_A8_Unorm, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM },
        { ImageFormat_R8_G8_B8_A8_Snorm, GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM },
        { ImageFormat_R8_G8_B8_A8_Uint, GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT },
        { ImageFormat_R8_G8_B8_A8_Sint, GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT },
        { ImageFormat_R8_G8_B8_A8_UnormSrgb, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB },
        // Not supported in Cafe { ImageFormat_B8_G8_R8_A8_Unorm, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_B8_G8_R8_A8_Snorm, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_B8_G8_R8_A8_Uint, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_B8_G8_R8_A8_Sint, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_B8_G8_R8_A8_UnormSrgb, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_R9_G9_B9_E5_SharedExp, GX2_SURFACE_FORMAT },
        { ImageFormat_R10_G10_B10_A2_Unorm, GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM },
        { ImageFormat_R10_G10_B10_A2_Uint, GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_UINT },
        { ImageFormat_R11_G11_B10_Float, GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT },
        { ImageFormat_R16_G16_Unorm, GX2_SURFACE_FORMAT_TC_R16_G16_UNORM },
        { ImageFormat_R16_G16_Snorm, GX2_SURFACE_FORMAT_TC_R16_G16_SNORM },
        { ImageFormat_R16_G16_Uint, GX2_SURFACE_FORMAT_TC_R16_G16_UINT },
        { ImageFormat_R16_G16_Sint, GX2_SURFACE_FORMAT_TC_R16_G16_SINT },
        { ImageFormat_R16_G16_Float, GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT },
        { ImageFormat_D24_Unorm_S8_Uint, GX2_SURFACE_FORMAT_D_D24_S8_UNORM },
        { ImageFormat_R32_Uint, GX2_SURFACE_FORMAT_TC_R32_UINT },
        { ImageFormat_R32_Sint, GX2_SURFACE_FORMAT_TC_R32_SINT },
        { ImageFormat_R32_Float, GX2_SURFACE_FORMAT_TCD_R32_FLOAT },
        { ImageFormat_D32_Float, GX2_SURFACE_FORMAT_TCD_R32_FLOAT },
        { ImageFormat_R16_G16_B16_A16_Unorm, GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UNORM },
        { ImageFormat_R16_G16_B16_A16_Snorm, GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM },
        { ImageFormat_R16_G16_B16_A16_Uint, GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UINT },
        { ImageFormat_R16_G16_B16_A16_Sint, GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT },
        { ImageFormat_R16_G16_B16_A16_Float, GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT },
        { ImageFormat_D32_Float_S8_Uint_X24, GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24 },
        { ImageFormat_R32_G32_Uint, GX2_SURFACE_FORMAT_TC_R32_G32_UINT },
        { ImageFormat_R32_G32_Sint, GX2_SURFACE_FORMAT_TC_R32_G32_SINT },
        { ImageFormat_R32_G32_Float, GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT },
        // Not supported in Cafe { ImageFormat_R32_G32_B32_Uint, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_R32_G32_B32_Sint, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_R32_G32_B32_Float, GX2_SURFACE_FORMAT },
        { ImageFormat_R32_G32_B32_A32_Uint, GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_UINT },
        { ImageFormat_R32_G32_B32_A32_Sint, GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT },
        { ImageFormat_R32_G32_B32_A32_Float, GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT },
        { ImageFormat_Bc1_Unorm, GX2_SURFACE_FORMAT_T_BC1_UNORM },
        { ImageFormat_Bc1_UnormSrgb, GX2_SURFACE_FORMAT_T_BC1_SRGB },
        { ImageFormat_Bc2_Unorm, GX2_SURFACE_FORMAT_T_BC2_UNORM },
        { ImageFormat_Bc2_UnormSrgb, GX2_SURFACE_FORMAT_T_BC2_SRGB },
        { ImageFormat_Bc3_Unorm, GX2_SURFACE_FORMAT_T_BC3_UNORM },
        { ImageFormat_Bc3_UnormSrgb, GX2_SURFACE_FORMAT_T_BC3_SRGB },
        { ImageFormat_Bc4_Unorm, GX2_SURFACE_FORMAT_T_BC4_UNORM },
        { ImageFormat_Bc4_Snorm, GX2_SURFACE_FORMAT_T_BC4_SNORM },
        { ImageFormat_Bc5_Unorm, GX2_SURFACE_FORMAT_T_BC5_UNORM },
        { ImageFormat_Bc5_Snorm, GX2_SURFACE_FORMAT_T_BC5_SNORM },
        // Not supported in Cafe { ImageFormat_Bc6_Float, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_Bc6_Ufloat, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_Bc7_Unorm, GX2_SURFACE_FORMAT },
        // Not supported in Cafe { ImageFormat_Bc7_UnormSrgb, GX2_SURFACE_FORMAT },
    };

    struct CompImageFormatAndGxFormat
    {
        bool operator()( const ImageFormatAndGxFormat& lhs, ImageFormat rhs ) const NN_NOEXCEPT
        {
            return lhs.imageFormat < rhs;
        }
    };

    struct AttributeFormatAndGxFormat
    {
        AttributeFormat attributeFormat;
        GX2AttribFormat gxFormat;
    };

    AttributeFormatAndGxFormat s_GxAttributeFormatList[] =
    {
        { AttributeFormat_4_4_Unorm, GX2_ATTRIB_FORMAT_4_4_UNORM },
        { AttributeFormat_8_Unorm, GX2_ATTRIB_FORMAT_8_UNORM },
        { AttributeFormat_8_Snorm, GX2_ATTRIB_FORMAT_8_SNORM },
        { AttributeFormat_8_Uint, GX2_ATTRIB_FORMAT_8_UINT },
        { AttributeFormat_8_Sint, GX2_ATTRIB_FORMAT_8_SINT },
        { AttributeFormat_8_UintToFloat, GX2_ATTRIB_FORMAT_8_UINT_TO_FLOAT },
        { AttributeFormat_8_SintToFloat, GX2_ATTRIB_FORMAT_8_SINT_TO_FLOAT },
        { AttributeFormat_8_8_Unorm, GX2_ATTRIB_FORMAT_8_8_UNORM },
        { AttributeFormat_8_8_Snorm, GX2_ATTRIB_FORMAT_8_8_SNORM },
        { AttributeFormat_8_8_Uint, GX2_ATTRIB_FORMAT_8_8_UINT },
        { AttributeFormat_8_8_Sint, GX2_ATTRIB_FORMAT_8_8_SINT },
        { AttributeFormat_8_8_UintToFloat, GX2_ATTRIB_FORMAT_8_8_UINT_TO_FLOAT },
        { AttributeFormat_8_8_SintToFloat, GX2_ATTRIB_FORMAT_8_8_SINT_TO_FLOAT },
        { AttributeFormat_16_Unorm, GX2_ATTRIB_FORMAT_16_UNORM },
        { AttributeFormat_16_Snorm, GX2_ATTRIB_FORMAT_16_SNORM },
        { AttributeFormat_16_Uint, GX2_ATTRIB_FORMAT_16_UINT },
        { AttributeFormat_16_Sint, GX2_ATTRIB_FORMAT_16_SINT },
        { AttributeFormat_16_Float, GX2_ATTRIB_FORMAT_16_FLOAT },
        { AttributeFormat_16_UintToFloat, GX2_ATTRIB_FORMAT_16_UINT_TO_FLOAT },
        { AttributeFormat_16_SintToFloat, GX2_ATTRIB_FORMAT_16_SINT_TO_FLOAT },
        { AttributeFormat_8_8_8_8_Unorm, GX2_ATTRIB_FORMAT_8_8_8_8_UNORM },
        { AttributeFormat_8_8_8_8_Snorm, GX2_ATTRIB_FORMAT_8_8_8_8_SNORM },
        { AttributeFormat_8_8_8_8_Uint, GX2_ATTRIB_FORMAT_8_8_8_8_UINT },
        { AttributeFormat_8_8_8_8_Sint, GX2_ATTRIB_FORMAT_8_8_8_8_SINT },
        { AttributeFormat_8_8_8_8_UintToFloat, GX2_ATTRIB_FORMAT_8_8_8_8_UINT_TO_FLOAT },
        { AttributeFormat_8_8_8_8_SintToFloat, GX2_ATTRIB_FORMAT_8_8_8_8_SINT_TO_FLOAT },
        { AttributeFormat_10_10_10_2_Unorm, GX2_ATTRIB_FORMAT_10_10_10_2_UNORM },
        { AttributeFormat_10_10_10_2_Snorm, GX2_ATTRIB_FORMAT_10_10_10_2_SNORM },
        { AttributeFormat_10_10_10_2_Uint, GX2_ATTRIB_FORMAT_10_10_10_2_UINT },
        { AttributeFormat_10_10_10_2_Sint, GX2_ATTRIB_FORMAT_10_10_10_2_SINT },
        { AttributeFormat_16_16_Unorm, GX2_ATTRIB_FORMAT_16_16_UNORM },
        { AttributeFormat_16_16_Snorm, GX2_ATTRIB_FORMAT_16_16_SNORM },
        { AttributeFormat_16_16_Uint, GX2_ATTRIB_FORMAT_16_16_UINT },
        { AttributeFormat_16_16_Sint, GX2_ATTRIB_FORMAT_16_16_SINT },
        { AttributeFormat_16_16_Float, GX2_ATTRIB_FORMAT_16_16_FLOAT },
        { AttributeFormat_16_16_UintToFloat, GX2_ATTRIB_FORMAT_16_16_UINT_TO_FLOAT },
        { AttributeFormat_16_16_SintToFloat, GX2_ATTRIB_FORMAT_16_16_SINT_TO_FLOAT },
        { AttributeFormat_32_Uint, GX2_ATTRIB_FORMAT_32_UINT },
        { AttributeFormat_32_Sint, GX2_ATTRIB_FORMAT_32_SINT },
        { AttributeFormat_32_Float, GX2_ATTRIB_FORMAT_32_FLOAT },
        { AttributeFormat_16_16_16_16_Unorm, GX2_ATTRIB_FORMAT_16_16_16_16_UNORM },
        { AttributeFormat_16_16_16_16_Snorm, GX2_ATTRIB_FORMAT_16_16_16_16_SNORM },
        { AttributeFormat_16_16_16_16_Uint, GX2_ATTRIB_FORMAT_16_16_16_16_UINT },
        { AttributeFormat_16_16_16_16_Sint, GX2_ATTRIB_FORMAT_16_16_16_16_SINT },
        { AttributeFormat_16_16_16_16_Float, GX2_ATTRIB_FORMAT_16_16_16_16_FLOAT },
        { AttributeFormat_16_16_16_16_UintToFloat, GX2_ATTRIB_FORMAT_16_16_16_16_UINT_TO_FLOAT },
        { AttributeFormat_16_16_16_16_SintToFloat, GX2_ATTRIB_FORMAT_16_16_16_16_SINT_TO_FLOAT },
        { AttributeFormat_32_32_Uint, GX2_ATTRIB_FORMAT_32_32_UINT },
        { AttributeFormat_32_32_Sint, GX2_ATTRIB_FORMAT_32_32_SINT },
        { AttributeFormat_32_32_Float, GX2_ATTRIB_FORMAT_32_32_FLOAT },
        { AttributeFormat_32_32_32_Uint, GX2_ATTRIB_FORMAT_32_32_32_UINT },
        { AttributeFormat_32_32_32_Sint, GX2_ATTRIB_FORMAT_32_32_32_SINT },
        { AttributeFormat_32_32_32_Float, GX2_ATTRIB_FORMAT_32_32_32_FLOAT },
        { AttributeFormat_32_32_32_32_Uint, GX2_ATTRIB_FORMAT_32_32_32_32_UINT },
        { AttributeFormat_32_32_32_32_Sint, GX2_ATTRIB_FORMAT_32_32_32_32_SINT },
        { AttributeFormat_32_32_32_32_Float, GX2_ATTRIB_FORMAT_32_32_32_32_FLOAT },
    };

    struct CompAttributeFormatAndGxFormat
    {
        bool operator()( const AttributeFormatAndGxFormat& lhs, AttributeFormat rhs ) const
        {
            return lhs.attributeFormat < rhs;
        }
    };

    const struct SurfaceFormatAndProperty
    {
        GX2SurfaceFormat format;
        ImageFormatProperty property;
    } g_FormatPropertyList[] =
    {
        { GX2_SURFACE_FORMAT_INVALID, { 0 } },
        { GX2_SURFACE_FORMAT_TC_R8_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_T_R4_G4_UNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_TCD_R16_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R8_G8_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TCS_R5_G6_B5_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R5_G5_B5_A1_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R4_G4_B4_A4_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_A1_B5_G5_R5_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_D_D24_S8_UNORM, { ImageFormatPropertyFlag_Texture } }, // GX2_SURFACE_FORMAT_T_R24_UNORM_X8 と兼用
        { GX2_SURFACE_FORMAT_TCS_R10_G10_B10_A2_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TCS_A2_B10_G10_R10_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_T_BC1_UNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_BC2_UNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_BC3_UNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_BC4_UNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_BC5_UNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_NV12_UNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_TC_R8_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R8_G8_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R32_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_T_X24_G8_UINT, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_A2_B10_G10_R10_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_T_X32_G8_UINT_X24, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_TC_R32_G32_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_UINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R8_SNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_SNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R8_G8_SNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_SNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_T_R10_G10_B10_A2_SNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SNORM, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_T_BC4_SNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_BC5_SNORM, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_TC_R8_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R8_G8_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R32_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R10_G10_B10_A2_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R32_G32_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_SINT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_T_BC1_SRGB, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_BC2_SRGB, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_T_BC3_SRGB, { ImageFormatPropertyFlag_Texture } },
        { GX2_SURFACE_FORMAT_TC_R16_FLOAT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TCD_R32_FLOAT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_FLOAT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_D_D24_S8_FLOAT, { 0 } },
        { GX2_SURFACE_FORMAT_TC_R11_G11_B10_FLOAT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_D_D32_FLOAT_S8_UINT_X24, { ImageFormatPropertyFlag_Texture } }, // GX2_SURFACE_FORMAT_T_R32_FLOAT_X8_X24 と兼用
        { GX2_SURFACE_FORMAT_TC_R32_G32_FLOAT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R16_G16_B16_A16_FLOAT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } },
        { GX2_SURFACE_FORMAT_TC_R32_G32_B32_A32_FLOAT, { ImageFormatPropertyFlag_Texture | ImageFormatPropertyFlag_ColorTarget } }
    };

    struct CompSurfaceFormatAndProperty
    {
        bool operator()( const SurfaceFormatAndProperty& lhs, GX2SurfaceFormat rhs ) const
        {
            return lhs.format < rhs;
        }
    };
}

GX2DisplayListOverrun Gx::g_DisplayListOverrun = {};
CommandBufferImpl< ApiVariationGx2 >* Gx::g_pCurrentCommandBuffer[ MaxCoreCount ] = {};

bool Gx::IsArrayTarget( GX2SurfaceDim target ) NN_NOEXCEPT
{
    switch( target )
    {
    case GX2_SURFACE_DIM_1D_ARRAY:
    case GX2_SURFACE_DIM_2D_ARRAY:
    case GX2_SURFACE_DIM_2D_MSAA_ARRAY:
        {
            return true;
        }

    default:
        {
            return false;
        }
    }
}

GX2StencilFunction Gx::GetStencilFunction( StencilOperation operation ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( stencilOpTable ) == StencilOperation_End );

    return stencilOpTable[ operation ];
}

GX2CompareFunction Gx::GetCompareFunction( ComparisonFunction comparisonFunction ) NN_NOEXCEPT
{
    return static_cast< GX2CompareFunction >( comparisonFunction +
        ( GX2_COMPARE_NEVER - ComparisonFunction_Never ) );
}

GX2AttribFormat Gx::GetAttributeFormat( AttributeFormat format ) NN_NOEXCEPT
{
    AttributeFormatAndGxFormat* pEnd =
        s_GxAttributeFormatList + NN_GFX_ARRAY_LENGTH( s_GxAttributeFormatList );
    AttributeFormatAndGxFormat* pFound = std::lower_bound(
        s_GxAttributeFormatList, pEnd, format, CompAttributeFormatAndGxFormat() );
    if( pFound == pEnd || format < pFound->attributeFormat )
    {
        return static_cast< GX2AttribFormat >( -1 );
    }
    else
    {
        return pFound->gxFormat;
    }
}

GX2PolygonMode Gx::GetPolygonMode( FillMode fillMode ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( polygonModeTable ) == FillMode_End );

    return polygonModeTable[ fillMode ];
}

GX2PrimitiveType Gx::GetPrimitiveType( PrimitiveTopology topology ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( primitiveTable ) == PrimitiveTopology_End );

    return primitiveTable[ topology ];
}

GX2LogicOp Gx::GetLogicalOperation( LogicOperation logicOperation ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( logicOperationTable ) == LogicOperation_End );

    return logicOperationTable[ logicOperation ];
}

GX2BlendFunction Gx::GetBlendFunction( BlendFactor blendFactor ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( blendFactorTable ) == BlendFactor_End );

    return blendFactorTable[ blendFactor ];
}

GX2BlendCombine Gx::GetBlendCombineMode( BlendFunction blendFunction ) NN_NOEXCEPT
{
    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( blendCombineTable ) == BlendFunction_End );

    return blendCombineTable[ blendFunction ];
}

GX2SurfaceDim Gx::GetSurfaceDimension( ImageDimension imageDimension ) NN_NOEXCEPT
{
    static const GX2SurfaceDim s_DimensionTable[] =
    {
        GX2_SURFACE_DIM_1D,            // ImageDimension_1D,
        GX2_SURFACE_DIM_2D,            // ImageDimension_2D,
        GX2_SURFACE_DIM_3D,            // ImageDimension_3D,
        GX2_SURFACE_DIM_CUBE,          // ImageDimension_CUBE_MAP,
        GX2_SURFACE_DIM_1D_ARRAY,      // ImageDimension_1D_ARRAY,
        GX2_SURFACE_DIM_2D_ARRAY,      // ImageDimension_2D_ARRAY,
        GX2_SURFACE_DIM_2D_MSAA,       // ImageDimension_2D_MULTISAMPLE,
        GX2_SURFACE_DIM_2D_MSAA_ARRAY, // ImageDimension_2D_MULTISAMPLE_ARRAY,
        GX2_SURFACE_DIM_2D_MSAA        // ImageDimension_CUBE_MAP_ARRAY, - asserts on GX2
    };

    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_DimensionTable ) == ImageDimension_End );
    NN_SDK_ASSERT( imageDimension < ImageDimension_CubeMapArray ); // GX2 limitation

    return s_DimensionTable[ imageDimension ];
}

GX2AAMode Gx::GetAaMode( int numberOfSamples ) NN_NOEXCEPT
{
    static const GX2AAMode s_AaTable[] =
    {
        GX2_AA_MODE_1X, // 0
        GX2_AA_MODE_1X, // 1
        GX2_AA_MODE_2X, // 2
        GX2_AA_MODE_4X, // 3
        GX2_AA_MODE_4X, // 4
        GX2_AA_MODE_8X, // 5
        GX2_AA_MODE_8X, // 6
        GX2_AA_MODE_8X, // 7
        GX2_AA_MODE_8X  // 8
    };

    NN_SDK_ASSERT( numberOfSamples <= 8 ); // Maximum number of samples allowed for GX2

    return s_AaTable[ numberOfSamples ];
}

GX2Component Gx::GetComponentMapping( ChannelMapping mapping ) NN_NOEXCEPT
{
    static const GX2Component s_ComponentMappingTable[] =
    {
        GX2_COMPONENT_C_0, // ChannelMapping_Zero
        GX2_COMPONENT_C_1, // ChannelMapping_One
        GX2_COMPONENT_X_R, // ChannelMapping_Red
        GX2_COMPONENT_Y_G, // ChannelMapping_Green
        GX2_COMPONENT_Z_B, // ChannelMapping_Blue
        GX2_COMPONENT_W_A, // ChannelMapping_Alpha
    };

    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_ComponentMappingTable ) == ChannelMapping_End );

    return s_ComponentMappingTable[ mapping ];
}

GX2SurfaceFormat Gx::GetSurfaceFormat( ImageFormat format ) NN_NOEXCEPT
{
    ImageFormatAndGxFormat* pEnd = s_GxTextureFormatList + NN_GFX_ARRAY_LENGTH( s_GxTextureFormatList );
    ImageFormatAndGxFormat* pFound = std::lower_bound( s_GxTextureFormatList, pEnd, format, CompImageFormatAndGxFormat() );
    if( pFound == pEnd || format < pFound->imageFormat )
    {
        return GX2_SURFACE_FORMAT_INVALID;
    }
    else
    {
        return pFound->gxFormat;
    }
}

GX2IndexFormat Gx::GetIndexFormat( IndexFormat indexFormat ) NN_NOEXCEPT
{
    static const GX2IndexFormat s_IndexFormatTable[] =
    {
        GX2_INDEX_FORMAT_U16, // IndexFormat_Uint8 - asserts on GX2
        GX2_INDEX_FORMAT_U16, // IndexFormat_Uint16
        GX2_INDEX_FORMAT_U32, // IndexFormat_Uint32
    };

    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_IndexFormatTable ) == IndexFormat_End );
    NN_SDK_ASSERT( indexFormat != IndexFormat_Uint8 ); // NOT SUPPORTED IN GX2!

    return s_IndexFormatTable[ indexFormat ];
}

GX2TexBorderType Gx::GetTextureBorderType( TextureBorderColorType borderColorType ) NN_NOEXCEPT
{
    static const GX2TexBorderType s_BorderTypeTable[] =
    {
        GX2_TEX_BORDER_SOLID_WHITE,
        GX2_TEX_BORDER_CLEAR_BLACK,
        GX2_TEX_BORDER_SOLID_BLACK
    };

    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_BorderTypeTable ) == TextureBorderColorType_End );

    return s_BorderTypeTable[ borderColorType ];
}

GX2TexClamp Gx::GetTextureClamp( TextureAddressMode textureAddressMode ) NN_NOEXCEPT
{
    static const GX2TexClamp s_ClampTable[] =
    {
        GX2_TEX_CLAMP_WRAP,
        GX2_TEX_CLAMP_MIRROR,
        GX2_TEX_CLAMP_CLAMP,
        GX2_TEX_CLAMP_CLAMP_BORDER,
        GX2_TEX_CLAMP_MIRROR_ONCE
    };

    NN_STATIC_ASSERT( NN_GFX_ARRAY_LENGTH( s_ClampTable ) == TextureAddressMode_End );

    return s_ClampTable[ textureAddressMode ];
}

void Gx::SetSwapInterval( int swapInterval ) NN_NOEXCEPT
{
    if ( swapInterval != g_SwapInterval )
    {
        NN_GFX_CALL_GX_FUNCTION( GX2SetSwapInterval( swapInterval ) );
        g_SwapInterval = swapInterval;
    }
}

GX2BufferingMode Gx::GetBufferingMode( int bufferCount ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( bufferCount >= static_cast< int >( GX2_BUFFERING_FIRST )
        && bufferCount <= static_cast< int >( GX2_BUFFERING_LAST ) );
    return static_cast< GX2BufferingMode >( bufferCount );
}

void Gx::GetImageFormatProperty( ImageFormatProperty* pOutImageFormatProperty,
    GX2SurfaceFormat format ) NN_NOEXCEPT
{
    NN_SDK_ASSERT( pOutImageFormatProperty != NULL );
    const SurfaceFormatAndProperty* pEnd =
        g_FormatPropertyList + NN_GFX_ARRAY_LENGTH( g_FormatPropertyList );
    const SurfaceFormatAndProperty* pFound = std::lower_bound(
        g_FormatPropertyList, pEnd, format, CompSurfaceFormatAndProperty() );
    if( pFound == pEnd || format < pFound->format )
    {
        pOutImageFormatProperty->propertyFlags = 0;
    }
    else
    {
        *pOutImageFormatProperty = pFound->property;
    }
}

void Gx::CallbackFunction( GX2CallbackEvent callbackEvent, void* pUserData ) NN_NOEXCEPT
{
    if( callbackEvent != GX2_CB_EVENT_DL_OVERRUN )
    {
        return;
    }

    GX2DisplayListOverrun* pOverrun = static_cast< GX2DisplayListOverrun* >( pUserData );
    int coreId = static_cast< int >( OSGetCoreId() );
    NN_SDK_ASSERT( coreId < MaxCoreCount );
    CommandBufferImpl< ApiVariationGx2 >* pCommandBuffer = g_pCurrentCommandBuffer[ coreId ];
    CommandBufferImpl< ApiVariationGx2 >::DataType& commandBuffer = pCommandBuffer->ToData();

    DisplayList* pOldDisplayList = static_cast< DisplayList* >( commandBuffer.pCommandMemory );
    NN_SDK_ASSERT( pOldDisplayList->pBuffer == pOverrun->oldDisplayList );
    pOldDisplayList->usedSize = pOverrun->oldByteSize;

    Bit8 oldState = commandBuffer.state;
    commandBuffer.state = CommandBufferImpl< ApiVariationGx2 >::DataType::State_Callback;
    NN_SDK_ASSERT_NOT_NULL( commandBuffer.pOutOfCommandMemoryCallback );
    OutOfMemoryEventArg arg;
    arg.minRequiredSize = pOverrun->newByteSize;
    ( *reinterpret_cast< CommandBufferImpl< ApiVariationGx2 >::OutOfMemoryEventCallback >(
        commandBuffer.pOutOfCommandMemoryCallback.ptr ) )( static_cast<
        TCommandBuffer< ApiVariationGx2 >* >( pCommandBuffer ), arg );
    commandBuffer.state = oldState;

    DisplayList* pNewDisplayList = static_cast< DisplayList* >( commandBuffer.pCommandMemory );
    NN_SDK_ASSERT( pNewDisplayList->pBuffer != pOverrun->oldDisplayList );
    pOverrun->newDisplayList = pNewDisplayList->pBuffer;
    pOverrun->newByteSize = pNewDisplayList->bufferSize;
}

}
}
}
