﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#if !defined( NN_BUILD_CONFIG_OS_WIN32 )
#error
#endif

#include <nn/nn_SdkAssert.h>

#include <nn/hws/hws_Window.h>
#include <nn/vi/vi_Layer.h>

#include <nn/gfx/gfx_DeviceInfo.h>

#include <nn/gfx/detail/gfx_Device-api.nvn.8.h>

#include "gfx_CommonHelper.h"
#include "gfx_NvnHelper.h"
#include "gfx_WglHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationNvn8 Target;

DeviceImpl< Target >::DeviceImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DeviceImpl< Target >::~DeviceImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT( this->state == State_NotInitialized );
}

void DeviceImpl< Target >::Initialize( const InfoType& info ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_STATIC_ASSERT( sizeof( this->nvnDevice ) == sizeof( NVNdevice ) );
    NN_UNUSED( info );

    HWND hWnd = static_cast< HWND >( nn::hws::FindWindow( "GfxSystemDisplay" ).Get() );
    NN_SDK_ASSERT_NOT_NULL( hWnd );

    HDC hDc = ::GetDC( hWnd );
    NN_SDK_ASSERT_NOT_NULL( hDc );

    HGLRC hGlRc = Wgl::CreateGlRc( hDc, NULL,
        Target::Version::value, 0, info.GetDebugMode() != DebugMode_Disable );
    NN_SDK_ASSERT_NOT_NULL( hGlRc );

    ::wglMakeCurrent( hDc, hGlRc );

    // Initialize the NVN function pointer interface.
    PFNNVNBOOTSTRAPLOADERPROC bootstrapLoader = NULL;
    bootstrapLoader = reinterpret_cast< PFNNVNBOOTSTRAPLOADERPROC >(
        wglGetProcAddress( "rq34nd2ffz" ) );

    if( bootstrapLoader )
    {
        pfnc_nvnDeviceGetProcAddress = reinterpret_cast< PFNNVNDEVICEGETPROCADDRESSPROC >(
            (*bootstrapLoader)( "nvnDeviceGetProcAddress" ) );
    }

    NN_SDK_ASSERT_NOT_NULL( pfnc_nvnDeviceGetProcAddress ); // "Couldn't initialize the NVN bootstrap loader (possible version mismatch).\n" );

    NN_GFX_CALL_NVN_FUNCTION( nvnLoadCProcs( NULL, pfnc_nvnDeviceGetProcAddress ) );

    // デバイス初期化前に NULL デバイスからバージョンを取得することが可能
    int nvnMajorVersion = int();
    int nvnMinorVersion = int();
    NN_SDK_ASSERT( ( NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger(
        NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &nvnMajorVersion ) ),
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceGetInteger(
        NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &nvnMinorVersion ) ),
        nvnMajorVersion == NVN_API_MAJOR_VERSION
        && nvnMinorVersion >= NVN_API_MINOR_VERSION ) );
    NN_UNUSED( nvnMajorVersion );
    NN_UNUSED( nvnMinorVersion );

    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    if( info.GetDebugMode() == DebugMode_Enable )
    {
        deviceFlags |= NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT;
    }
    if( info.GetDebugMode() == DebugMode_Full )
    {
        deviceFlags |= NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
    }
#endif
    NVNdeviceBuilder builder;
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceBuilderSetDefaults( &builder ) );
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceBuilderSetFlags( &builder, deviceFlags ) );

    // Convenience pointer since a lot of the code already uses this.
    this->pNvnDevice = &this->nvnDevice;

    NVNboolean result = nvnDeviceInitialize( this->pNvnDevice, &builder );
    NN_SDK_REQUIRES( result ); // "Couldn't initialize the NVN device.\n"
    NN_UNUSED( result );

    // デバイス初期化後に再度関数ポインターをロード
    NN_GFX_CALL_NVN_FUNCTION( nvnLoadCProcs( this->pNvnDevice, pfnc_nvnDeviceGetProcAddress ) );

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    if( ( deviceFlags & ( NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_0_BIT |
        NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_1_BIT |
        NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT |
        NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_3_BIT |
        NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT ) ) != 0 )
    {
        NN_GFX_CALL_NVN_FUNCTION( nvnDeviceInstallDebugCallback(
            this->pNvnDevice, Nvn::DebugCallback, NULL, NVN_TRUE ) );
    }
#endif

    ::wglMakeCurrent( NULL, NULL );

    Wgl::DeleteGlRc( static_cast<HGLRC>( hGlRc ) );
    ::ReleaseDC( static_cast<HWND>( hWnd ), static_cast<HDC>( hDc ) );

    this->supportedFeatures = Nvn::GetDeviceFeature( this->pNvnDevice );

    this->flags.SetBit( Flag_Shared, false );
    this->state = State_Initialized;
}

void DeviceImpl< Target >::Finalize() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_ASSERT( !this->flags.GetBit( Flag_Shared ) );

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceInstallDebugCallback( this->pNvnDevice,
        Nvn::DebugCallback, NULL, NVN_FALSE ) );
#endif

    NN_GFX_CALL_NVN_FUNCTION( nvnDeviceFinalize( this->pNvnDevice ) );
    this->pNvnDevice = NULL;

    this->state = State_NotInitialized;
}

}
}
}
