﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_BytePtr.h>

#include <nn/gfx/gfx_DescriptorPoolInfo.h>
#include <nn/gfx/gfx_GpuAddress.h>
#include <nn/gfx/gfx_DescriptorSlot.h>

#include <nn/gfx/detail/gfx_DescriptorPool-api.gl.4.h>
#include <nn/gfx/detail/gfx_Buffer-api.gl.4.h>
#include <nn/gfx/detail/gfx_Texture-api.gl.4.h>
#include <nn/gfx/detail/gfx_Sampler-api.gl.4.h>
#include <nn/gfx/detail/gfx_MemoryPool-api.gl.4.h>

#include "gfx_GlHelper.h"

namespace nn {
namespace gfx {
namespace detail {

typedef ApiVariationGl4 Target;

namespace {

static const size_t g_DescriptorSlotIncrementSize[] =
{
    sizeof( uint32_t ) + sizeof( uint32_t ) + sizeof( uint32_t ),
    sizeof( GlHandle ) + sizeof( GlEnum ),
    sizeof( GlHandle )
};

}

size_t DescriptorPoolImpl< Target >::CalculateDescriptorPoolSize(
    DeviceImpl< Target >* pDevice, const InfoType& info ) NN_NOEXCEPT
{
    return GetDescriptorSlotIncrementSize( pDevice, info.GetDescriptorPoolType() ) * info.GetSlotCount();
}

ptrdiff_t DescriptorPoolImpl< Target >::GetDescriptorSlotIncrementSize(
    DeviceImpl< Target >*, DescriptorPoolType type ) NN_NOEXCEPT
{
    return g_DescriptorSlotIncrementSize[ type ];
}

size_t DescriptorPoolImpl< Target >::GetDescriptorPoolAlignment( DeviceImpl< Target >*, const InfoType& ) NN_NOEXCEPT
{
    // All of the data in the descriptor slot types only require 32bit alignment
    return sizeof( uint32_t );
}

DescriptorPoolImpl< Target >::DescriptorPoolImpl() NN_NOEXCEPT
{
    this->state = State_NotInitialized;
}

DescriptorPoolImpl< Target >::~DescriptorPoolImpl() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
}

void DescriptorPoolImpl< Target >::Initialize( DeviceImpl< Target >* pDevice, const InfoType& info,
    MemoryPoolImpl< Target >* pMemoryPool, ptrdiff_t memoryPoolOffset, size_t memoryPoolSize ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_SDK_REQUIRES( this->state == State_NotInitialized );
    NN_SDK_REQUIRES( memoryPoolSize >= CalculateDescriptorPoolSize( pDevice, info ) );
    NN_SDK_REQUIRES_NOT_NULL( pMemoryPool );
    NN_SDK_REQUIRES( IsInitialized( *pMemoryPool ) );
    NN_SDK_ASSERT( pMemoryPool->ToData()->pMemory );
    NN_SDK_REQUIRES( pMemoryPool->ToData()->memorySize >= memoryPoolOffset + memoryPoolSize );
    NN_UNUSED( pDevice );
    NN_UNUSED( memoryPoolSize );

    NN_SDK_ASSERT( !this->pMemory );

    this->descriptorPoolType = static_cast< Bit8 >( info.GetDescriptorPoolType() );
    this->slotCount = static_cast< uint32_t >( info.GetSlotCount() );
    this->pMemory = nn::util::BytePtr( pMemoryPool->ToData()->pMemory, memoryPoolOffset ).Get();

    this->state = State_Initialized;
}

void DescriptorPoolImpl< Target >::Finalize( DeviceImpl< Target >* pDevice ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    NN_SDK_REQUIRES_NOT_NULL( pDevice );
    NN_UNUSED( pDevice );

    this->pMemory = NULL;

    this->state = State_NotInitialized;
}

void DescriptorPoolImpl< Target >::BeginUpdate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Initialized );
    this->state = State_Begun;
}

void DescriptorPoolImpl< Target >::EndUpdate() NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    this->state = State_Initialized;
}

void DescriptorPoolImpl< Target >::SetBufferView( int indexSlot,
    const GpuAddress& gpuAddress, size_t size ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_BufferView );
    NN_SDK_ASSERT( IsValid( static_cast< GlHandle >( gpuAddress.ToData()->impl ) ) );
    uint32_t* pSlot = nn::util::BytePtr( this->pMemory, g_DescriptorSlotIncrementSize[
        DescriptorPoolType_BufferView ] * indexSlot ).Get< uint32_t >();

    pSlot[ 0 ] = static_cast< uint32_t >( gpuAddress.ToData()->impl );
    pSlot[ 1 ] = static_cast< uint32_t >( gpuAddress.ToData()->value );
    pSlot[ 2 ] = static_cast< uint32_t >( size );
}

void DescriptorPoolImpl< Target >::SetSampler( int indexSlot,
    const SamplerImpl< Target >* pSampler ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_Sampler );
    NN_SDK_REQUIRES_NOT_NULL( pSampler );
    NN_SDK_REQUIRES( IsInitialized( *pSampler ) );
    NN_SDK_ASSERT( pSampler->ToData()->hSampler );
    static_cast< GlHandle* >( this->pMemory )[ indexSlot ] = pSampler->ToData()->hSampler;
}

void DescriptorPoolImpl< Target >::SetTextureView( int indexSlot,
    const TextureViewImpl< Target >* pTextureView ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pTextureView );
    NN_SDK_REQUIRES( IsInitialized( *pTextureView ) );
    NN_SDK_ASSERT( IsValid( pTextureView->ToData()->hTexture ) );
    nn::util::BytePtr pSlot( this->pMemory, indexSlot *
        g_DescriptorSlotIncrementSize[ DescriptorPoolType_TextureView ] );
    *pSlot.Get< GlHandle >() = pTextureView->ToData()->hTexture;
    *pSlot.Advance( sizeof( GlHandle ) ).Get< GlEnum >() = pTextureView->ToData()->target;
}

void DescriptorPoolImpl< Target >::SetImage( int indexSlot,
    const TextureViewImpl< Target >* pImage ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pImage );
    NN_SDK_REQUIRES( IsInitialized( *pImage ) );
    NN_SDK_ASSERT( IsValid( pImage->ToData()->hTexture ) );
    nn::util::BytePtr pSlot( this->pMemory, indexSlot *
        g_DescriptorSlotIncrementSize[ DescriptorPoolType_TextureView ] );
    *pSlot.Get< GlHandle >() = pImage->ToData()->hTexture;
    *pSlot.Advance( sizeof( GlHandle ) ).Get< GlEnum >() = pImage->ToData()->target;
}

void DescriptorPoolImpl< Target >::SetBufferTextureView( int indexSlot,
    const BufferTextureViewImpl< Target >* pBufferTexture ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pBufferTexture );
    NN_SDK_REQUIRES( IsInitialized( *pBufferTexture ) );
    NN_SDK_ASSERT( IsValid( pBufferTexture->ToData()->hBufferTexture ) );
    nn::util::BytePtr pSlot( this->pMemory, indexSlot *
        g_DescriptorSlotIncrementSize[ DescriptorPoolType_TextureView ] );
    *pSlot.Get< GlHandle >() = pBufferTexture->ToData()->hBufferTexture;
    *pSlot.Advance( sizeof( GlHandle ) ).Get< GlEnum >() = GL_TEXTURE_BUFFER;
}

void DescriptorPoolImpl< Target >::SetBufferImage( int indexSlot,
    const BufferTextureViewImpl< Target >* pBufferTexture ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state == State_Begun );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );
    NN_SDK_REQUIRES( this->descriptorPoolType == DescriptorPoolType_TextureView );
    NN_SDK_REQUIRES_NOT_NULL( pBufferTexture );
    NN_SDK_REQUIRES( IsInitialized( *pBufferTexture ) );
    NN_SDK_ASSERT( IsValid( pBufferTexture->ToData()->hBufferTexture ) );
    nn::util::BytePtr pSlot( this->pMemory, indexSlot *
        g_DescriptorSlotIncrementSize[ DescriptorPoolType_TextureView ] );
    *pSlot.Get< GlHandle >() = pBufferTexture->ToData()->hBufferTexture;
    *pSlot.Advance( sizeof( GlHandle ) ).Get< GlEnum >() = GL_TEXTURE_BUFFER;
}

void DescriptorPoolImpl< Target >::GetDescriptorSlot(
    DescriptorSlot* pOutDescriptorSlot, int indexSlot ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_REQUIRES_RANGE( indexSlot, 0, static_cast< int >( this->slotCount ) );

    pOutDescriptorSlot->ToData()->value = reinterpret_cast< uintptr_t >( nn::util::BytePtr( this->pMemory.ptr,
        indexSlot * g_DescriptorSlotIncrementSize[ this->descriptorPoolType ] ).Get() );
}

int DescriptorPoolImpl< Target >::GetDescriptorSlotIndex(
    const DescriptorSlot& descriptorSlot ) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES( this->state != State_NotInitialized );
    NN_SDK_REQUIRES( descriptorSlot.IsValid() );

    return static_cast< int >( nn::util::ConstBytePtr( this->pMemory.ptr ).Distance(
        reinterpret_cast< const void* >( static_cast< uintptr_t >( descriptorSlot.ToData()->value ) ) ) ) /
        static_cast< int >( g_DescriptorSlotIncrementSize[ this->descriptorPoolType ] );
}

}
}
}
