﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/util/rapidjson/ens_RapidJsonInputStream.h>

namespace nn { namespace ens { namespace detail { namespace util { namespace rapidjson {

RapidJsonInputStream::RapidJsonInputStream() NN_NOEXCEPT
    : m_pString(nullptr)
    , m_StringBufferSize(0)
    , m_pStream(nullptr)
    , m_StreamBufferSize(0)
    , m_StringSize(0)
    , m_Read(0)
    , m_ReadTotal(0)
    , m_PositionInStream(0)
    , m_LastError(nn::ResultSuccess())
    , m_IsStringOverflowDetected(false)
{
}

RapidJsonInputStream::~RapidJsonInputStream() NN_NOEXCEPT
{
}

void RapidJsonInputStream::SetBuffer(Ch* pStringBuffer, size_t stringBufferSize,
    Ch* pStreamBuffer, size_t streamBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStringBuffer);
    NN_SDK_REQUIRES_GREATER(stringBufferSize, 0u);
    NN_SDK_REQUIRES_NOT_NULL(pStreamBuffer);
    NN_SDK_REQUIRES_GREATER(streamBufferSize, 0u);

    m_pString = pStringBuffer;
    m_StringBufferSize = stringBufferSize;
    m_pStream = pStreamBuffer;
    m_StreamBufferSize = streamBufferSize;
}

nn::Result RapidJsonInputStream::GetLastError() const NN_NOEXCEPT
{
    return m_LastError;
}

bool RapidJsonInputStream::IsStringOverflowDetected() const NN_NOEXCEPT
{
    return m_IsStringOverflowDetected;
}

RapidJsonInputStream::Ch RapidJsonInputStream::Peek() NN_NOEXCEPT
{
    if (m_PositionInStream >= m_Read)
    {
        NN_SDK_ASSERT_EQUAL(m_PositionInStream, m_Read);

        if (!Fill())
        {
            return '\0';
        }
    }

    return m_pStream[m_PositionInStream];
}

RapidJsonInputStream::Ch RapidJsonInputStream::Take() NN_NOEXCEPT
{
    Ch c = Peek();
    m_PositionInStream++;

    return c;
}

size_t RapidJsonInputStream::Tell() const NN_NOEXCEPT
{
    return (m_ReadTotal - m_Read) + m_PositionInStream;
}

RapidJsonInputStream::Ch* RapidJsonInputStream::PutBegin() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pString);

    m_StringSize = 0;

    return m_pString;
}

void RapidJsonInputStream::Put(Ch c) NN_NOEXCEPT
{
    if (m_StringSize >= m_StringBufferSize)
    {
        m_IsStringOverflowDetected = true;
        return;
    }

    m_pString[m_StringSize++] = c;
}

size_t RapidJsonInputStream::PutEnd(Ch* pBegin) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(pBegin, m_pString);
    NN_UNUSED(pBegin);

    return m_StringSize;
}

bool RapidJsonInputStream::Fill() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(m_pStream);

    m_PositionInStream = 0;

    nn::Result result = FillImpl(&m_Read, m_pStream, m_StreamBufferSize);

    if (result.IsFailure())
    {
        m_LastError = result;
        m_Read = 0;
    }

    m_ReadTotal += m_Read;

    return (m_Read > 0);
}

}}}}}
