﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/task/ens_TaskReport.h>
#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <nn/ens/detail/util/ens_MessagePackReader.h>
#include <nn/ens/detail/util/ens_MessagePackWriter.h>
#include <nn/ens/detail/util/ens_ResponseStructureReader.h>

namespace nn { namespace ens { namespace detail { namespace core { namespace task {

namespace
{
    const char* GetCategoryString(ReportCategory category) NN_NOEXCEPT
    {
        switch (category)
        {
        case ReportCategory_Personal:
            return "personal";
        case ReportCategory_Criminal:
            return "criminal";
        case ReportCategory_Immoral:
            return "immoral";
        case ReportCategory_Harassment:
            return "harassment";
        case ReportCategory_Commercial:
            return "commercial";
        case ReportCategory_SexuallyExplicit:
            return "sexually_explicit";
        case ReportCategory_Other:
            return "other";
/*
        case ReportCategory_PlayingInactivity:
            return "playing_inactivity";
        case ReportCategory_PlayingFrequentDisconnection:
            return "playing_frequent_disconnection";
        case ReportCategory_PlayingUnsportsmanlikeConduct:
            return "playing_unsportsmanlike_conduct";
        case ReportCategory_PlayingNetworkManipulation:
            return "playing_network_manipulation";
        case ReportCategory_Cheating:
            return "cheating";
        case ReportCategory_InappropriateNickname:
            return "inappropriate_nickname";
*/
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

TaskReport::TaskReport() NN_NOEXCEPT
    : m_pType(nullptr)
    , m_pReason(nullptr)
{
}

TaskReport::~TaskReport() NN_NOEXCEPT
{
}

void TaskReport::SetParameter(const char* pType, nn::account::NetworkServiceAccountId nsaId,
    ReportCategory category, const char* pReason,
    const SendBuffer& image,
    const ReportContentInfo& contentInfo) NN_NOEXCEPT
{
    m_pType = pType;
    m_NsaId = nsaId;

    m_Category = category;
    m_pReason = pReason;

    m_Image = image;

    m_ContentInfo = contentInfo;
}

nn::Result TaskReport::ExecuteSteps() NN_NOEXCEPT
{
    NN_RESULT_DO(StepReport());

    NN_RESULT_SUCCESS;
}

nn::Result TaskReport::StepReport() NN_NOEXCEPT
{
    NN_DETAIL_ENS_SCOPED_ACQUIRE_HTTP_CONNECTION(pCurl);

    char url[256] = {};
    detail::core::MakeUrl(url, sizeof (url), "api", "/api/v1/reports");

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_CUSTOMREQUEST, "POST"));
    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_URL, url));

    detail::core::HttpRequestHeader header;

    // 入力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), ResultInvalidState());

        detail::util::MessagePackWriter writer;

        writer.SetBuffer(GetRequestBuffer(), GetRequestBufferSize());

        NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteMap(m_Image.IsNull() ? 5 : 6));
        {
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("type"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_pType));

            char nsaId[17] = {};
            nn::util::SNPrintf(nsaId, sizeof (nsaId), "%016llx", m_NsaId.id);

            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("nsaid"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(nsaId, 16));

            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("category"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(GetCategoryString(m_Category)));

            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("reason"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_pReason));

            if (!m_Image.IsNull())
            {
                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("image"));
                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_Image));
            }

            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("content"));
            NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteMap(4));
            {
                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("id"));
                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_ContentInfo.id));

                char digest[Digest::StringLength + 1] = {};

                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("digest"));
                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_ContentInfo.digest.ToString(digest, sizeof (digest))));

                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("language"));
                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_ContentInfo.language.string));

                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteKey("created_at"));
                NN_DETAIL_ENS_REQUEST_WRITER_DO(writer.WriteValue(m_ContentInfo.createdAt.value));
            }
        }

        NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDS, writer.GetBuffer()));
        NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_POSTFIELDSIZE, writer.GetWrittenSize()));
    }

    NN_RESULT_DO(header.Add("Authorization", "Bearer %s", GetAuthToken().value));
    NN_RESULT_DO(header.Add("Content-Type", "application/x-msgpack"));

    NN_DETAIL_ENS_CURL_EASY_DO(curl_easy_setopt(pCurl, CURLOPT_HTTPHEADER, header.Get()));

    NN_RESULT_DO(Perform());

    NN_DETAIL_ENS_DUMP_RESPONSE(detail::util::MessagePackReader, GetResponse(), GetResponseSize());

    // 出力パラメータへのアクセス
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), ResultInvalidState());

        detail::util::ResponseStructureReader<1> reader;

        char reportId[64] = {};

        reader.Add("$.report_id",
            reportId, sizeof (reportId));

        NN_RESULT_THROW_UNLESS(reader.Read<detail::util::MessagePackReader>(GetResponse(), GetResponseSize()), ResultInvalidResponse());
    }

    NN_RESULT_SUCCESS;
}

}}}}}
