﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ens/detail/core/ens_HttpRequestHeader.h>
#include <cstdarg>

namespace nn { namespace ens { namespace detail { namespace core {

HttpRequestHeader::HttpRequestHeader() NN_NOEXCEPT
    : m_pHeaderList(nullptr)
{
}

HttpRequestHeader::~HttpRequestHeader() NN_NOEXCEPT
{
    Clear();
}

void HttpRequestHeader::Clear() NN_NOEXCEPT
{
    if (m_pHeaderList)
    {
        curl_slist_free_all(m_pHeaderList);
        m_pHeaderList = nullptr;
    }
}

curl_slist* HttpRequestHeader::Get() NN_NOEXCEPT
{
    return m_pHeaderList;
}

nn::Result HttpRequestHeader::Add(const char* pName, const char* pValue, ...) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pName);
    NN_SDK_REQUIRES_NOT_NULL(pValue);

    NN_SDK_REQUIRES_GREATER(nn::util::Strnlen(pName, INT_MAX), 0);

    char field[FieldSize] = {};

    size_t offset = static_cast<size_t>(nn::util::SNPrintf(field, sizeof (field), "%s: ", pName));

    NN_RESULT_THROW_UNLESS(offset < sizeof (field), ResultInvalidArgument());

    va_list args;
    va_start(args, pValue);

    size_t valueLength = static_cast<size_t>(nn::util::VSNPrintf(&field[offset], sizeof (field) - offset, pValue, args));

    va_end(args);

    NN_RESULT_THROW_UNLESS(offset + valueLength < sizeof (field), ResultInvalidArgument());

    curl_slist* pHeaderList = curl_slist_append(m_pHeaderList, field);

    NN_RESULT_THROW_UNLESS(pHeaderList, ResultHttpErrorOutOfMemory());

    m_pHeaderList = pHeaderList;

    NN_RESULT_SUCCESS;
}

}}}}
