﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/typeDef2.h>
#if EFT_IS_WIN
#include <winsock2.h>
#include <WS2tcpip.h>
#include <Windows.h>
#endif
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eftvw2_ToolConnecter.h>
#include <nw/eft/eftcom_Message.h>


// Releaseでは、devライブラリを使用しない。
#if defined( NW_RELEASE )

namespace nw {
namespace eftvw2 {

volatile bool ToolConnecter::s_isConnected = false;

bool ToolConnecter::Initialize( nw::eft2::Heap* heap, size_t buffSize, CmdReceiver* receive, CmdSender* sender, bool noUseInternalThread, int priority )
{
    EFT_UNUSED_VARIABLE( heap );
    EFT_UNUSED_VARIABLE( buffSize );
    EFT_UNUSED_VARIABLE( receive );
    EFT_UNUSED_VARIABLE( sender );
    EFT_UNUSED_VARIABLE( noUseInternalThread );
    EFT_UNUSED_VARIABLE( priority );
    return false;
}
int  ToolConnecter::threadToolReceiver(int intArg, void * ptrArg)
{
    EFT_UNUSED_VARIABLE( intArg );
    EFT_UNUSED_VARIABLE( ptrArg );
    return false;
}
void ToolConnecter::Finalize() {}
void ToolConnecter::SendPacket(){}

} // namespace eftvw
} // namespace nw


#else

#if EFT_IS_CAFE
namespace /* anonymous */ {

// 正しく処理するためにHIOのエラーコードをすべて符号なし整数型に変更する.
static const u32 EFTVW_HIO_STATUS_NEW_CONNECTION        = u32( HIO_STATUS_NEW_CONNECTION );                 // 1
static const u32 EFTVW_HIO_STATUS_OK                    = u32( HIO_STATUS_OK );                             // 0
static const u32 EFTVW_HIO_INVALID_CHANNEL_NAME         = u32( HIO_STATUS_INVALID_CHANNEL_NAME );           // -1.
static const u32 EFTVW_HIO_NO_CONNECTIONS               = u32( HIO_STATUS_NO_CONNECTIONS );                 // -2.
static const u32 EFTVW_HIO_NO_CHANNELS                  = u32( HIO_STATUS_NO_CHANNELS );                    // -3.
static const u32 EFTVW_HIO_INVALID_HANDLE               = u32( HIO_STATUS_INVALID_HANDLE );                 // -4.
static const u32 EFTVW_HIO_NO_CLIENT_TXN_BUF_AVAILABLE  = u32( HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE );    // -5.
static const u32 EFTVW_HIO_OUT_OF_MEMORY                = u32( HIO_STATUS_OUT_OF_MEMORY );                  // -6.
static const u32 EFTVW_HIO_INVALID_PARAM                = u32( HIO_STATUS_INVALID_PARAM );                  // -7.
static const u32 EFTVW_HIO_NOT_INITIALIZED              = u32( HIO_STATUS_NOT_INITIALIZED );                // -8.
static const u32 EFTVW_HIO_STATUS_INVALID_OPERATION     = u32( HIO_STATUS_INVALID_OPERATION );              // -9.

//------------------------------------------------------------------------------
//      HostIOのエラーであるかどうかチェックします.
//------------------------------------------------------------------------------
bool IsHioError( u32 status )
{
    bool error = false;
    switch( status )
    {
    case EFTVW_HIO_STATUS_NEW_CONNECTION:
        {
            nw::eft2::OutputLog( "Info : HIO_STATUS_NEW_CONNECTION\n" );
            error = false;
        }
        break;

    case EFTVW_HIO_STATUS_OK:
        {
            nw::eft2::OutputLog( "Info : HIO_STATUS_OK\n" );
            error = false;
        }
        break;

    case EFTVW_HIO_INVALID_CHANNEL_NAME:
        {
            nw::eft2::OutputLog( "Error : HIO_INVALID_CHANNEL_NAME\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_NO_CONNECTIONS:
        {
            nw::eft2::OutputLog( "Error : HIO_NO_CONNECTIONS\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_NO_CHANNELS:
        {
            nw::eft2::OutputLog( "Error : HIO_NO_CHANNELS\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_INVALID_HANDLE:
        {
            nw::eft2::OutputLog( "Error : HIO_INVALID_HANDLE\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_NO_CLIENT_TXN_BUF_AVAILABLE:
        {
            nw::eft2::OutputLog( "Error : HIO_NO_CLIENT_TXN_BUF_AVAILABLE\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_OUT_OF_MEMORY:
        {
            nw::eft2::OutputLog( "Error : HIO_OUT_OF_MEMORY\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_INVALID_PARAM:
        {
            nw::eft2::OutputLog( "Error : HIO_INVALID_PARAM\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_NOT_INITIALIZED:
        {
            nw::eft2::OutputLog( "Error : HIO_NOT_INITIALIZED\n" );
            error = true;
        }
        break;

    case EFTVW_HIO_STATUS_INVALID_OPERATION:
        {
            nw::eft2::OutputLog( "Error : HIO_STATUS_INVALID_OPERATION\n" );
            error = true;
        }
        break;

    default:
        {
            //nw::eft2::OutputLog( "Info : HIOStatus %u\n", status );
            error = false;
        }
        break;
    }

    return error;
}

} // namespace /* anonymoous */
#endif//EFT_IS_CAFE

namespace nw   {
namespace eftvw2 {

/*
    定数
*/
static const char* STR_HEADER = "NW_FE2";

static bool                s_isOK = false;
volatile bool              ToolConnecter::s_isConnected = false;
static volatile bool       s_isDetermin = true;

#if EFT_IS_CAFE
static volatile HIOStatus  s_ProcStatus;
#endif

static size_t              s_ReadBufferSize;
static nw::eft2::Heap*     s_EftHeap;
static u8*                 s_pReadBuffer;
static TaskBase*           s_taskBase;

#if EFT_IS_WIN
static u32                 s_Handle = INVALID_SOCKET;
#else
static HIOHandle           s_Handle = 0;
OSEvent event;
#endif

static nw::ut::Mutex       s_Lock;
static CmdReceiver*        s_pReceive = NULL;
static CmdSender*          s_pSender  = NULL;

#define READ_SIZE          500

//------------------------------------------------------------------------------
//　初期化
//------------------------------------------------------------------------------
bool ToolConnecter::Initialize(
    nw::eft2::Heap* heap,
    size_t bufferSize,
    CmdReceiver* receive,
    CmdSender* sender,
    bool noUseInternalThread,
    int  priority
    )
{
    s_EftHeap           = heap;
    s_ReadBufferSize    = bufferSize;
    s_pReadBuffer       = NULL;
    s_isDetermin        = true;
    s_isConnected       = false;
    s_pReceive          = static_cast<CmdReceiver*>( receive );
    s_pSender           = static_cast<CmdSender*>( sender );
    s_Lock.Initialize();


    // 読み込みバッファ確保
    s_pReadBuffer = (u8*)s_EftHeap->Alloc( s_ReadBufferSize );
    memset( s_pReadBuffer, 0, s_ReadBufferSize );

    // 通信スレッドの起動

    if ( noUseInternalThread == false )
    {
        s_taskBase = new nw::eftvw2::TaskBase();
        nw::ut::Thread::CreateArg creatArg;
        s_taskBase->SetStack( (void **)&creatArg.stackBase, &creatArg.stackSize );

        static TaskObj task;
        task.Init( threadToolReceiver, 0, NULL );
        s_taskBase->AddTask( &task );
#if EFT_IS_WIN
        creatArg.creationFlags = CREATE_SUSPENDED;
#else
        HIOInit();
        creatArg.priority = priority;
#endif

        s_taskBase->CreateThread( creatArg );

#if EFT_IS_WIN
        s_taskBase->SetPriority( priority );
#endif

        s_taskBase->Start();
    }
    s_isOK = true;

    return s_isOK;
}

//------------------------------------------------------------------------------
//　破棄
//------------------------------------------------------------------------------
void ToolConnecter::Finalize()
{
#if EFT_IS_CAFE
    // Cafeの時、Toolと繋がっていないとイベント待ちで終了しないので、シグナル状態にします。
    OSSignalEvent( &event );
#endif

    if ( s_isOK )
    {
        s_isConnected   = false;
        s_isDetermin    = false;
        s_isOK = false;
    }

#if EFT_IS_CAFE
    // Cafeの時は、OSJoinThread()が必要なので実行します。
    // Winの時は、WaitForSingleObject()がINFINITEで待つので、Toolと繋がっていないと終了しないので実行しない。
    if ( s_taskBase ) s_taskBase->Stop();
#endif
    delete s_taskBase;

#if EFT_IS_WIN
    WSACleanup();
#else
    HIOClose(s_Handle);
#endif

    if ( s_pReadBuffer )
    {
        s_EftHeap->Free( s_pReadBuffer );
        s_pReadBuffer = NULL;
    }

}

//------------------------------------------------------------------------------
//　パケットデータの送信
//------------------------------------------------------------------------------
void ToolConnecter::SendPacket()
{
#if EFT_IS_CAFE
    if (s_Handle != 0)
    {
#else
    if ( s_Handle != INVALID_SOCKET )
    {
#endif
        sendPacket( s_Handle );
    }

}

void ToolConnecter::sendPacket( u32 param )
{
    EFT_UNUSED_VARIABLE( param );
#if EFT_IS_CAFE
    HIOHandle handle = (HIOHandle)param;
#else
    SOCKET sock = (SOCKET)param;
#endif

    while (s_pSender->GetCommand() != NULL)
    {
        Cmd* cmd = s_pSender->GetCommand();

        //ProtocolEM4F::Header*  pHeader =
        //    reinterpret_cast<ProtocolEM4F::Header*>( cmd->GetCommandBuffer() );

        void* buffer = cmd->GetCommandBuffer();

#if EFT_IS_CAFE
        HIOStatus status = HIOWrite( handle, (u32)cmd->GetCommandBufferSize(), buffer );

        if (HIO_STATUS_OK > status){
            nw::eft2::OutputLog( "ERROR: HIOWrite returned %d\n", status );
        }
#else
        int len = send(sock, (const char*)buffer, (u32)cmd->GetCommandBufferSize(), 0);
        if (len == SOCKET_ERROR)
        {

            nw::eft2::OutputLog( "ERROR: Send Packet error %d\n", WSAGetLastError() );
        }
#endif
        s_pSender->PopCommand();
    }
}

bool ToolConnecter::IsHandlePacket( u8* pBuf )
{
    if (pBuf == NULL) return false;

    nw::eftcom::Message* pHeader = reinterpret_cast<nw::eftcom::Message*>( pBuf );
    nw::eftcom::Message msg = (*pHeader);

    // PINGパケットはそれはそれでよし
    if (msg.type == 0 && msg.bufferSize == 0) return true;

    if ((msg.type >= nw::eftcom::MESSAGE_TYPE_CONTROL)
        && (msg.type <= nw::eftcom::MESSAGE_TYPE_ESET_REQUEST))
    {
        if ( msg.bufferSize != 0 ) return true;
    }

    return false;
}

u8*  ToolConnecter::SkipPacket( u8* pBuf, u32& size )
{
    size = 0;
    if ( pBuf != NULL )
    {
        size = sizeof(nw::eftcom::Message);
        pBuf += size;
    }

    return pBuf;
}

u32 ToolConnecter::receivePacket( const u32 handle, void* pReceiveBuffer, const u32 receiveSize )
{
#if EFT_IS_CAFE
    const HIOHandle hioHandle = static_cast<HIOHandle>(handle);
    // HIOから読み込む.
    u32 status = HIORead(hioHandle, receiveSize, pReceiveBuffer);

    // エラーチェック.
    if (IsHioError(status))
    {
        return 0;
    }

    return status;
#elif EFT_IS_WIN
    const SOCKET sock = static_cast<SOCKET>(handle);
    int status = recv(sock, reinterpret_cast<char*>(pReceiveBuffer), receiveSize, 0);

    if ( status == SOCKET_ERROR )
    {
        nw::eft2::OutputLog("Error in recv socket %d\n", WSAGetLastError());
        return 0;
    }
    else if (status == 0)
    {
        nw::eft2::OutputLog("Connenction closed %d\n", WSAGetLastError());
        return 0;
    }
    else
    {
        return static_cast<u32>(status);
    }
#endif
}


//------------------------------------------------------------------------------
//　パケットデータの取得
//------------------------------------------------------------------------------
bool ToolConnecter::readPacket( u32 param )
{
    if ( s_pReceive == NULL )
    { return true; }

    if ( s_pReceive->gCmdReceiver == NULL )
    { return true; }

    // ヘッダ分読み込み
    u32 headerSize = receivePacket(param, s_pReadBuffer, sizeof(nw::eftcom::Message));

    // 受信データがない場合はfalseとして終了
    if (headerSize == 0) return false;

    // ヘッダ分読めない場合はパケットぶっ壊れてるから何やってもダメ
    if (headerSize != sizeof(nw::eftcom::Message))
    {
        nw::ut::Printf("[RecvErr]Imcomplete header!\n");
        return false;
    }

    // 無効なヘッダを読んじゃった場合もダメ
    if (!IsHandlePacket(s_pReadBuffer))
    {
        nw::ut::Printf("[RecvErr]Invalid message!\n");
        return false;
    }

    nw::eftcom::Message* pHeader = reinterpret_cast<nw::eftcom::Message*>(s_pReadBuffer);
    nw::eftcom::Message msg = (*pHeader);

    if ((msg.type >  nw::eftcom::MESSAGE_TYPE_CONTROL) &&
        (msg.type <= nw::eftcom::MESSAGE_TYPE_ESET_REQUEST))
    {
        // 期待サイズを算出.
        u32 size = msg.bufferSize;

        // バッファサイズ分読み込み
        u32 readSize = receivePacket(param, s_pReadBuffer + sizeof(nw::eftcom::Message), size);

        if (size == readSize)
        {
            // バッファサイズ分読めてるならコマンドとして積む
            s_pReceive->ReceiveProc(s_pReadBuffer, size + sizeof(nw::eftcom::Message));
        }
        else
        {
            // バッファサイズ分読めてないなら何やってもダメ
            nw::ut::Printf("[RecvErr]Imcomplete message!\n");
            return false;
        }
    }

    return true;
}

#if EFT_IS_CAFE
//------------------------------------------------------------------------------
//　通信ステータスの取得
//------------------------------------------------------------------------------
void ToolConnecter::procStatusCallback(HIOStatus status, void * context)
{
    s_ProcStatus = status;

    OSSignalEvent((OSEvent *)context);

    switch(status)
    {
        case HIO_STATUS_NEW_CONNECTION:
            nw::eft2::OutputLog("HIO STATUS : NEW_CONNECTION\n");
            s_isConnected = true;
            break;

        case HIO_STATUS_OK:
            nw::eft2::OutputLog("HIO STATUS : OK\n");
            break;

        case HIO_STATUS_INVALID_CHANNEL_NAME:
            nw::eft2::OutputLog("HIO STATUS : INVALID_CHANNEL_NAME\n");
            break;

        case HIO_STATUS_NO_CONNECTIONS:
            s_isConnected = false;
            nw::eft2::OutputLog("HIO STATUS : NO_CONNECTIONS\n");
            break;

        case HIO_STATUS_NO_CHANNELS:
            nw::eft2::OutputLog("HIO STATUS : NO_CHANNELS\n");
            break;

        case HIO_STATUS_INVALID_HANDLE:
            nw::eft2::OutputLog("HIO STATUS : INVALID_HANDLE\n");
            break;

        case HIO_STATUS_NO_CLIENT_TXN_BUF_AVAILABLE:
            nw::eft2::OutputLog("HIO STATUS : NO_CLIENT_TXN_BUF_AVAILABLE\n");
            break;

        default:
            nw::eft2::OutputLog("HIO STATUS : OTHER\n");
            break;
    }
}
#endif
//------------------------------------------------------------------------------
//　通信スレッド
//------------------------------------------------------------------------------

#if EFT_IS_CAFE
int ToolConnecter::threadToolReceiver(int intArg, void * ptrArg)
{
     OSInitEvent( &event, FALSE, OS_EVENT_MANUAL );

     s_Handle = HIOOpen(STR_HEADER, procStatusCallback, &event, HIO_CHANNEL_OPTION_READ_WRITE, 0);

     if (s_Handle >= 0)
     {
         // Wait for the PC to start listening.
         OSWaitEvent( &event );

         s_isConnected = true;
         while (s_isDetermin)
         {
             // 受信パケットの処理
             if (!readPacket(s_Handle)){
                 nw::eft2::OutputLog("ERROR: Failed to read %d bytes\n", s_ReadBufferSize);
             }

             OSSleepMilliseconds( 2 );
         }

         // When done with an HIO channel, call HIOClose to free up the channel.

         HIOClose(s_Handle);
     }
     else
     {
         nw::eft2::OutputLog("Initializing for EM4F : HIOOpen returned 0x%08x\n", s_Handle );
     }

    return 0;
}
#else

bool OpenSocket( SOCKET *sock0, SOCKET *sock )
{
    WSADATA wsaData;
    struct sockaddr_in addr;
    struct sockaddr_in client;
    int len;

    // winsock2の初期化
    WSAStartup(0x0202, &wsaData);

    // ソケットの作成
    *sock0 = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);

    // ソケットの設定
    addr.sin_family = AF_INET;
    addr.sin_port = htons(8201);
    inet_pton( AF_INET, "127.0.0.1", &addr.sin_addr );
    bind(*sock0, (struct sockaddr *)&addr, sizeof(addr));

    // TCPクライアントからの接続要求を待てる状態にする
    int ret = listen(*sock0, 5);

    // TCPクライアントからの接続要求を受け付ける
    len = sizeof(client);
    *sock = accept(*sock0,(struct sockaddr *)&client, &len);

    if ((*sock == INVALID_SOCKET) || (ret == SOCKET_ERROR))
    {
        nw::eft2::OutputLog("Socket open error %d\n", WSAGetLastError());
        return false;
    }

    nw::eft2::OutputLog("Server start\n");
    return true;
}

bool ReOpenSocket(SOCKET *sock0, SOCKET *sock)
{
    // TCPクライアントからの接続要求を待てる状態にする
    int ret = listen(*sock0, 5);

    // TCPクライアントからの接続要求を受け付ける
    *sock = accept(*sock0, NULL, NULL);

    if ((*sock == INVALID_SOCKET) || (ret == SOCKET_ERROR))
    {
        nw::eft2::OutputLog("Socket open error %d\n", WSAGetLastError());
        return false;
    }
    s_Handle = (u32)*sock;
    nw::eft2::OutputLog("Server Re-start \n");
    return true;
}

int ToolConnecter::threadToolReceiver(int intArg, void * ptrArg)
{
    EFT_UNUSED_VARIABLE( intArg );
    EFT_UNUSED_VARIABLE( ptrArg );

    SOCKET sock0;
    SOCKET sock;

    int status = OpenSocket( &sock0, &sock );

    if (status)
    {
        s_isConnected = true;
        s_Handle = (u32)sock;

        while( s_isDetermin )
        {
            if ( s_isConnected == false )
            {
                s_isConnected = ( ReOpenSocket(&sock0, &sock) == true ) ? true : false;
            }

            if ( s_isConnected == true )
            {
               if ( !readPacket((u32) sock) )
               {
                   if ( closesocket(sock) == SOCKET_ERROR )
                   {
                       nw::eft2::OutputLog("ERROR: Can't socket close %d\n", WSAGetLastError());
                   }
                   s_isConnected = false;
                }
            }
        }
        // TCPセッションの終了
        closesocket(sock);
    }
    else
    {
        nw::eft2::OutputLog("ERROR: Initializing for EM4F : Socket Open Error %d\n", WSAGetLastError());
    }

    shutdown(sock, SD_BOTH);

    // winsock2の終了処理
    WSACleanup();
    nw::eft2::OutputLog("Server stop\n");

    return 0;
}
#endif

} // namespace eftvw2
} // namespace nw


#endif
