﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eftvw2_EffectPreview.h>


//------------------------------------------------------------------------------
namespace nw     {
namespace eftvw2 {


//---------------------------------------------------------------------------
//  EffectPreview クラスのインスタンスを生成します。
//---------------------------------------------------------------------------
EffectPreview* EffectPreview::CreateEffectPreview(
    nw::eft2::Heap* heap, nw::eft2::System* system, eftcom::Guid guid )
{
    // プレビュー生成用のメモリを確保します。
    void* buffer = heap->Alloc( sizeof(EffectPreview) );
    if( !buffer )
    {
        nw::eft2::OutputWarning( "Memory Allocate Error!! : %d\n", sizeof(EffectPreview) );
        return NULL;
    }

    // クラスの生成を行います。
    EffectPreview* effectPreview = new ( buffer ) EffectPreview();
    if ( !effectPreview )
    {
        return NULL;
    }

    // 初期化処理を行います。
    effectPreview->Initialize( heap, EFT_VWR_PRV_EFFECT, guid );
    effectPreview->InitializeEffectPreview( system );

    // 破棄コールバックをセット
    effectPreview->SetDestroyPreviewCallback( EffectPreview::DestroyEffectPreview );

    return effectPreview;
}

//---------------------------------------------------------------------------
//  EffectPreview クラスのインスタンスを破棄します。
//---------------------------------------------------------------------------
void EffectPreview::DestroyEffectPreview( nw::eft2::Heap* heap, Preview* preview )
{
    EffectPreview* effectPreview = reinterpret_cast<EffectPreview*>( preview );

    // 再生中のエミッタセットを破棄
    effectPreview->DestroyEmitterSet();

    // プレビューの終了処理を行います。
    effectPreview->Finalize();

    // メモリを解放
    heap->Free( effectPreview );
}

//---------------------------------------------------------------------------
//  コンストラクタです。
//---------------------------------------------------------------------------
EffectPreview::EffectPreview()
: Preview()
{
    mSystem             = NULL;
    mResId              = -1;
    mWiatFrame          = 0;
    mForceCalcFrame     = 0;

    memset( &mResEffectPreview, 0, sizeof( ResEffectPreview ) );
//   mCurrentHandle.Invalidate();
    mCenter.SetIdentity();
    mOffsetMatrix.SetIdentity();
    mFunctionColor.Set( 1.0f, 1.0f, 1.0f, 1.0f );
    mViewerColor.Set( 1.0f, 1.0f, 1.0f, 1.0f );
    mEmitterSetVisible  = true;

    for ( u32 i = 0; i < nw::eft2::EFT_EMITTER_INSET_NUM; ++i )
    {
        mEmitterVisible[i] = true;
    }

    for( u32 i = 0; i < EFT_PREV_FADE_CTRL_MAX; ++i )
    {
        mFadeEmitterSet[i] = NULL;
    }
}

//---------------------------------------------------------------------------
//  エフェクトプレビューを初期化します。
//---------------------------------------------------------------------------
void EffectPreview::InitializeEffectPreview( nw::eft2::System* system )
{
    mSystem = system;
}

//---------------------------------------------------------------------------
//  計算処理です。
//---------------------------------------------------------------------------
void EffectPreview::Calc( bool bPause, f32 frameRate, const nw::math::MTX34& centerMatrix, const nw::math::MTX34& viewMatrix )
{
    // 規定クラスの計算処理を行う
    Preview::Calc( bPause, frameRate, centerMatrix, viewMatrix );

    // エフェクトハンドル取得
    if ( mCurrentHandle.IsValid() )
    {
        nw::eft2::EmitterSet* emitterSet = mCurrentHandle.GetEmitterSet();

        //------------------------------------------
        // エミッタセットに対する制御
        //------------------------------------------
        if ( mIsSetResEffectPreview )
        {
            // 描画マトリクスをセット
            // mDrawMatrixは規定クラスの計算処理で計算される。
            if ( mResPreview.constrain.matrixApplyType == 0 )
            {
                emitterSet->SetMatrix( mDrawMatrix );
            }
            else
            {
                emitterSet->SetMatrix( mDrawMatrix );
            }

            // 再生開始フレーム
            mWiatFrame      = mResEffectPreview.basicSetting.startFrame;

            // 強制開始フレーム
            mForceCalcFrame = mResEffectPreview.basicSetting.forceStartFrame;

            // エミッタセットカラー
            emitterSet->SetColor( mResEffectPreview.color.color );

            // 放出レートスケール
            emitterSet->SetEmissionRatioScale( mResEffectPreview.emission.emissionRate );

            // 放出間隔スケール
            emitterSet->SetEmissionIntervalScale( mResEffectPreview.emission.emissionInterval );

            // ライフスケール
            emitterSet->SetParticleLifeScale( mResEffectPreview.ptclControl.life );

            // パーティクル放出時スケール
            emitterSet->SetEmissionParticleScale( mResEffectPreview.ptclScale.emissionParticleScale );

            // パーティクル放出時スケール
            emitterSet->SetParticleScale( mResEffectPreview.ptclScale.particleScale );

            // エミッタ形状
            emitterSet->SetEmitterScale( mResEffectPreview.ptclScale.emitterVolumeScale );

            // 初速
            emitterSet->SetAllDirectionalVel( mResEffectPreview.ptclControl.allDirectionVel );
            emitterSet->SetDirectionalVel( mResEffectPreview.ptclControl.directionalVel );
            emitterSet->SetRandomVel( mResEffectPreview.ptclControl.randomVel );
            emitterSet->SetAddVel( mResEffectPreview.ptclControl.addVel );

            // エミッタカラー設定
            if ( mResEffectPreview.ptclControl.enableEmitterColor )
            {
                nw::math::VEC4 color;
                color.x = mResEffectPreview.ptclControl.emitterColor0.x;
                color.y = mResEffectPreview.ptclControl.emitterColor0.y;
                color.z = mResEffectPreview.ptclControl.emitterColor0.z;
                color.w = 1.0f;
                emitterSet->SetEmitterColor0( color );
                color.x = mResEffectPreview.ptclControl.emitterColor1.x;
                color.y = mResEffectPreview.ptclControl.emitterColor1.y;
                color.z = mResEffectPreview.ptclControl.emitterColor1.z;
                color.w = 1.0f;
                emitterSet->SetEmitterColor1( color );
            }
            else
            {
                nw::math::VEC4 color;
                color.x = 1.0f;
                color.y = 1.0f;
                color.z = 1.0f;
                color.w = 1.0f;
                emitterSet->SetEmitterColor0( color );
                emitterSet->SetEmitterColor1( color );
            }
        }
    }


    // エミッタカラー
    // 仕様検討中。

    // 再生開始フレームに到達した場合は再生開始する
    if ( mWiatFrame == mTime && mCurrentHandle.IsValid() )
    {
        mCurrentHandle.GetEmitterSet()->SetCalcEnable( true );
        mCurrentHandle.GetEmitterSet()->SetVisible( true );
    }

    // フェード処理が終わったエミッタセットは NULL とする
    for( u32 i = 0; i < EFT_PREV_FADE_CTRL_MAX; ++i )
    {
        if ( mFadeEmitterSet[i] )
        {
            if ( mFadeEmitterSet[i]->IsAlive() )
            {
                mFadeEmitterSet[i]->SetMatrix( mDrawMatrix );
            }
            else
            {
                mFadeEmitterSet[i] = NULL;
            }
        }
    }
}

//---------------------------------------------------------------------------
//  エミッタセットを生成します。
//---------------------------------------------------------------------------
bool EffectPreview::CreateEmitterSet( s32 resId, u32 waitFrame, u32 forceCalc )
{
    if ( mCurrentHandle.IsValid() ) return false;

    mResId              = resId;

    EFT_UNUSED_VARIABLE( waitFrame );
    EFT_UNUSED_VARIABLE( forceCalc );

    //mWiatFrame          = waitFrame;
    //mForceCalcFrame     = forceCalc;

    // エミッタセット生成します。
    mSystem->CreateEmitterSetID( &mCurrentHandle, 0, resId, EFT_VWR_USE_GROUP_ID, mHeap );
    nw::eft2::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    if ( !em )
    {
        return false;
    }

    // 強制開始フレームをセット
    if ( mForceCalcFrame != 0 )
    {
        ForceCalc( mForceCalcFrame );
    }

    // 再生開始フレームが指定されていれば、処理を停止
    if ( mWiatFrame != 0 )
    {
        mCurrentHandle.GetEmitterSet()->SetCalcEnable( false );
        mCurrentHandle.GetEmitterSet()->SetVisible( false );
    }

    return true;
}

//---------------------------------------------------------------------------
//  エミッタセットを破棄します。
//---------------------------------------------------------------------------
bool EffectPreview::DestroyEmitterSet()
{
    if ( mCurrentHandle.IsValid() )
    {
        // TODO:現状、即時KILLが無い
        mSystem->KillEmitterSet( mCurrentHandle.GetEmitterSet() );
        mCurrentHandle.Invalidate();
    }

    for( u32 i = 0; i < EFT_PREV_FADE_CTRL_MAX; ++i )
    {
        if ( mFadeEmitterSet[i] && mFadeEmitterSet[i]->IsAlive() )
        {
            mFadeEmitterSet[i]->Kill();
            mFadeEmitterSet[i] = NULL;
        }
    }

    return true;
}

//---------------------------------------------------------------------------
//  指定フレーム再生位置を進めます。
//---------------------------------------------------------------------------
bool EffectPreview::ForceCalc( u32 frame )
{
    if ( !mCurrentHandle.IsValid() ) return false;
    mCurrentHandle.GetEmitterSet()->ForceCalc( frame );
    return true;
}

//---------------------------------------------------------------------------
//  エミッタセットをフェードします。
//---------------------------------------------------------------------------
bool EffectPreview::Fade()
{
    if ( !mCurrentHandle.IsValid() ) return false;

    mCurrentHandle.GetEmitterSet()->Fade();

    // 無限寿命の場合はKill
/*
    if ( mCurrentHandle.GetEmitterSet()->IsHaveInfinityEmitter() )
    {
        mCurrentHandle.GetEmitterSet()->Kill();
    }
    else
*/
    // Fade中のエミッタセットを操作する為に保持する。
    for( u32 i = 0; i < EFT_PREV_FADE_CTRL_MAX; ++i )
    {
        if ( !mFadeEmitterSet[i] )
        {
            mFadeEmitterSet[i] = mCurrentHandle.GetEmitterSet();
            break;
        }
    }

    mCurrentHandle.Invalidate();

    return true;
}

//---------------------------------------------------------------------------
//  プレビュー再生をリセットします。
//---------------------------------------------------------------------------
void EffectPreview::ResetPreview( bool bFade )
{
    NW_UNUSED_VARIABLE( bFade );
    if ( mCurrentHandle.IsValid() )
    {
        if ( bFade )
        {
            Fade();
        }
        else
        {
            DestroyEmitterSet();
        }
        mCurrentHandle.Invalidate();
    }

    // エミッタセットを再生成
    CreateEmitterSet( mResId, mWiatFrame, mForceCalcFrame );

    // 表示・非表示を再セット
    SetVisible( mEmitterSetVisible );

    mTime = 0.0f;
}

//---------------------------------------------------------------------------
//! @brief        描画の表示/非表示を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetVisible( bool visible )
{
    if ( !mCurrentHandle.IsValid() ) return;

    nw::eft2::EmitterSet* emitterSet = mCurrentHandle.GetEmitterSet();
    if ( !emitterSet ) return;

    emitterSet->SetVisible( visible );

    // Fade中のエミッタセットにも適用
    for( u32 i = 0; i < EFT_PREV_FADE_CTRL_MAX; ++i )
    {
        if ( mFadeEmitterSet[i] && mFadeEmitterSet[i]->IsAlive() )
        {
            mFadeEmitterSet[i]->SetStopDraw( !visible );
        }
    }

    mEmitterSetVisible = visible;
}





#if 0
//---------------------------------------------------------------------------
// カラーを設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetColor( nw::ut::Color4f* color )
{
    EFT_UNUSED_VARIABLE( color );

    // TODO:後で考える
    // mFunctionColor.Set( *color );
    //
    // updateDrawColor();
}


//---------------------------------------------------------------------------
// ビューアカラーを設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetViewerColor( nw::ut::Color4f* color )
{
    EFT_UNUSED_VARIABLE( color );

    // TODO:後で考える
    // mViewerColor.Set( *color );
    //
    // updateDrawColor();
}


//---------------------------------------------------------------------------
//! @brief        描画カラーを更新する。
//---------------------------------------------------------------------------
void EffectPreview::updateDrawColor()
{
    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetAlpha( mFunctionColor.a * mViewerColor.a );
    // em->SetColor( mFunctionColor.r * mViewerColor.r,
    //     mFunctionColor.g * mViewerColor.g,
    //     mFunctionColor.b * mViewerColor.b );
}


//---------------------------------------------------------------------------
//! @brief        パーティクルスケールを設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleScale( nw::math::VEC2* particle, nw::math::VEC2* emisison )
{
    EFT_UNUSED_VARIABLE( particle );
    EFT_UNUSED_VARIABLE( emisison );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetParticleScale( particle->x, particle->y );
    // em->SetEmissionParticleScale( emisison->x, emisison->y );
}


//---------------------------------------------------------------------------
//! @brief        エミッタスケールを設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetEmitterScale( nw::math::VEC3* emitter )
{
    EFT_UNUSED_VARIABLE( emitter );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetEmitterScale( *emitter );
}


//---------------------------------------------------------------------------
//! @brief        パーティクルライフスケールを設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleLifeScale( f32 lifeScale )
{
    EFT_UNUSED_VARIABLE( lifeScale );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // for ( s32 i = 0; i < em->GetNumEmitter(); ++i )
    // {
    //     nw::eft::EmitterController* ctrl = em->GetEmitterController( i );
    //     if ( ctrl )
    //     {
    //         ctrl->SetLife( lifeScale );
    //     }
    // }
}


//---------------------------------------------------------------------------
//! @brief        パーティクル速度スケールを設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleVelocityScale( f32 allDirectionalVec, f32 directionalVec )
{
    EFT_UNUSED_VARIABLE( allDirectionalVec );
    EFT_UNUSED_VARIABLE( directionalVec );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetAllDirectionalVel( allDirectionalVec );
    // em->SetDirectionalVel( directionalVec );
}

//---------------------------------------------------------------------------
//! @brief        パーティクルワールド加算速度を設定します。(エフェクトプレビュー専用の操作メソッド)
//---------------------------------------------------------------------------
void EffectPreview::SetParticleAddWorldVelocity( nw::math::VEC3* velocity )
{
    EFT_UNUSED_VARIABLE( velocity );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetAddVel( *velocity );
}

//---------------------------------------------------------------------------
//! @brief        パーティクルワールド加算速度を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleRandomVelocity( f32 velScale )
{
    EFT_UNUSED_VARIABLE( velScale );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetRandomVel( velScale );
}


//---------------------------------------------------------------------------
//! @brief        パーティクル指定方向速度のスケール値を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleDirectionalVel( f32 velScale )
{
    EFT_UNUSED_VARIABLE( velScale );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetDirectionalVel( velScale );
}


//---------------------------------------------------------------------------
//! @brief        パーティクル放出スケールを設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleEmissionScale( f32 emissionRate, f32 emissionInterval )
{
    EFT_UNUSED_VARIABLE( emissionRate );
    EFT_UNUSED_VARIABLE( emissionInterval );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // for ( s32 i = 0; i < em->GetNumEmitter(); ++i )
    // {
    //     nw::eft::EmitterController* ctrl = em->GetEmitterController( i );
    //     if ( ctrl )
    //     {
    //         ctrl->SetEmissionRatio( emissionRate );
    //         ctrl->SetEmissionInterval( emissionInterval );
    //     }
    // }
}


//---------------------------------------------------------------------------
//! @brief        パーティクル エミッタカラー０を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleEmitterColor0( nw::ut::Color4f* color )
{
    EFT_UNUSED_VARIABLE( color );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // for ( s32 i = 0; i < em->GetNumEmitter(); ++i )
    // {
    //     nw::eft::EmitterController* ctrl = em->GetEmitterController( i );
    //     if ( ctrl )
    //     {
    //         ctrl->SetEmitterColor0( *color );
    //     }
    // }
}


//---------------------------------------------------------------------------
//! @brief        パーティクル エミッタカラー０を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetParticleEmitterColor1( nw::ut::Color4f* color )
{
    EFT_UNUSED_VARIABLE( color );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // for ( s32 i = 0; i < em->GetNumEmitter(); ++i )
    // {
    //     nw::eft::EmitterController* ctrl = em->GetEmitterController( i );
    //     if ( ctrl )
    //     {
    //         ctrl->SetEmitterColor1( *color );
    //     }
    // }
}


//---------------------------------------------------------------------------
//! @brief        描画の表示/非表示を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetVisible( bool visible )
{
    EFT_UNUSED_VARIABLE( visible );

    // TODO:後で考える
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetStopDraw( !visible );
    //
    // // Fade中のエミッタセットを操作する為に保持する
    // for( u32 i = 0; i < EFT_PREV_FADE_CTRL_MAX; ++i )
    // {
    //     if ( mFadeEmitterSet[i] && mFadeEmitterSet[i]->IsAlive() )
    //     {
    //         mFadeEmitterSet[i]->SetStopDraw( !visible );
    //     }
    // }
    //
    // mEmitterSetVisible = visible;
}


//---------------------------------------------------------------------------
//! @brief        エミッタの描画の表示/非表示を設定します。
//---------------------------------------------------------------------------
void EffectPreview::SetEmitterVisible( u32 emitterID, bool visible )
{
    EFT_UNUSED_VARIABLE( emitterID );
    EFT_UNUSED_VARIABLE( visible );

    // TODO:後で考える
    // mEmitterVisible[emitterID] = visible;
    //
    // if ( !mCurrentHandle.IsValid() ) return;
    //
    // nw::eft::EmitterSet* em = mCurrentHandle.GetEmitterSet();
    // if ( !em ) return;
    //
    // em->SetStopDraw( emitterID, !mEmitterVisible[emitterID] );
    //
    // // Fade中のエミッタセットを操作する為に保持する
    // for( u32 i = 0; i < EFT_PREV_FADE_CTRL_MAX; ++i )
    // {
    //     if ( mFadeEmitterSet[i] && mFadeEmitterSet[i]->IsAlive() )
    //     {
    //         mFadeEmitterSet[i]->SetStopDraw( emitterID, !mEmitterVisible[emitterID] );
    //     }
    // }
}
#endif

//------------------------------------------------------------------------------
} // namespace eftvw
} // namespace nw
