﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nw/eft/eft2_SuperStripe.h>
#include <nw/eft/eft2_Misc.h>
#include <nw/eft/eft2_MemUtil.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_RenderContext.h>
#include <nw/eft/eft2_Render.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_Stripe.h>
#include <nw/eft/eft2_StripeConnection.h>

#ifndef EFT_DEGRADATION_SPEC

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴毎に固定割合」x「全て貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetUniformFill( int index, int maxIndex, float indexOffset )
{
    NW_UNUSED_VARIABLE( indexOffset );
    // MEMO: CalculateTextureOffsetUniformTile() とは実装を分けておく。
    //       UV補正処理で分岐できそう。
    // (index == maxIndex-1) の時に 1.0 を返すので、分母を 1引いて計算する
    const float fIndex = static_cast< float >( index );
    return static_cast< float >( fIndex ) / static_cast< float >( maxIndex - 1 );
}
//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴距離に依存」x「全て貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetDistanceBasedFill( int index, float prevOffset, float delta, float invStripeLength )
{
    // 先頭を揃える
    if( index == 0 )
    {
        return 0.0f;
    }

    // 直前の UV 差分をに加算
    const float offset = prevOffset + ( delta * invStripeLength );
    return offset;
}

//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴毎に固定割合」x「全て貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetUniformTile( int index, int maxIndex, float indexOffset )
{
    // (index == maxIndex-1) の時に 1.0 を返すので、分母を 1引いて計算する
    const float fIndex = ( index + indexOffset );
    float offset = static_cast< float >( fIndex ) / static_cast< float >( maxIndex - 1 );
    if( offset > 1.0f )
    {
        // 1.0 を超えた分を Clamp
        offset = 1.0f;
    }
    return offset;
}

//---------------------------------------------------------------------------
//  テクスチャ座標計算：「履歴距離に依存」x「描画範囲に合わせて貼る」
//---------------------------------------------------------------------------
float CalculateTextureOffsetDistanceBasedTile( int index, float prevOffset, float delta, float length )
{
    const float UvUnitLengthOnWorld = 0.1f;
    if( index == 0 )
    {
        return length * UvUnitLengthOnWorld;
    }

    const float offset = prevOffset + ( -delta * UvUnitLengthOnWorld );
    return offset;
}

//---------------------------------------------------------------------------
//  ストライプの履歴点のなす折れ線の長さを計算する
//---------------------------------------------------------------------------
float CalculateStripeLength( const SuperStripe* stripe, int historyCount )
{
    // ストライプの全長を求める
    f32 stripeLength = 0.0f;
    for( int i = 1; i < historyCount; ++i )
    {
        stripeLength += ( stripe->hist[ i ].pos - stripe->hist[ i - 1 ].pos ).Length();
    }
    return stripeLength;
}

//---------------------------------------------------------------------------
//  UV割り当て: 「履歴毎に固定割合」
//
//  【例】
//  履歴点:   (P0)----(P1)------------(P2)----(P3)----(P4)
//  UV 座標:  0.00    0.25            0.50    0.75    1.00
//
//  要するに従来通りの仕様。
//---------------------------------------------------------------------------
void CalculateTextureOffsetUniform(
    nw::math::VEC3* pOutTexOffset,          // 【共通】出力用 テクスチャオフセット（0/1/2）
    const ResStripeSuper* pStripeData,      // 【共通】ストライプデータ
    int index,                              // 【共通】頂点インデックス
    int vertexCount,                        // 【固定割当】頂点数
    int maxVertexCount,                     // 【固定割当】頂点の最大数
    float indexOffset                       // 【固定割当】インデックスのオフセット（含む小数）
)
{
    NW_NULL_ASSERT( pOutTexOffset );
    NW_NULL_ASSERT( pStripeData );

    nw::math::VEC3 offsetBase[ EFT_STRIPE_TEXTURING_OPTION_MAX_TYPE ];

    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_FILL ].x =
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_FILL ].y =
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_FILL ].z = CalculateTextureOffsetUniformFill( index, vertexCount, indexOffset );
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_TILE ].x =
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_TILE ].y =
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_TILE ].z = CalculateTextureOffsetUniformTile( index, maxVertexCount, indexOffset );

    // テクスチャリング設定を見て必要な方を取る
    nw::math::VEC3 offset(
        offsetBase[ pStripeData->texturing0 ].x,
        offsetBase[ pStripeData->texturing1 ].y,
        offsetBase[ pStripeData->texturing2 ].z );

    *pOutTexOffset = offset;
}

//---------------------------------------------------------------------------
//  UV割り当て: 「履歴距離に依存」
//
//  【例】
//  履歴点:   (P0)----(P1)------------(P2)----(P3)----(P4)
//  UV 座標:  0.00    0.20            0.60    0.80    1.00
//
//  距離に応じてUV加算を調整する。
//  描画範囲に合わせて貼る場合は、ワールド空間に貼られたテクスチャをストライプがマスクするイメージ
//---------------------------------------------------------------------------
void CalculateTextureOffsetDistanceBased(
    nw::math::VEC3* pOutTexOffset,          // 【共通】出力用 テクスチャオフセット（0/1/2）
    const ResStripeSuper* pStripeData,      // 【共通】ストライプデータ
    int index,                              // 【共通】頂点インデックス
    float delta,                            // 【距離依存】履歴点の差分距離
    float journey,                          // 【距離依存】ストライプの通算移動距離
    float invStripeLength,                  // 【距離依存】ストライプの折れ線の長さの逆数
    const nw::math::VEC4& prevOffset        // 【距離依存】ひとつ前の頂点のオフセット
)
{
    NW_NULL_ASSERT( pOutTexOffset );
    NW_NULL_ASSERT( pStripeData );

    nw::math::VEC3 offsetBase[ EFT_STRIPE_TEXTURING_OPTION_MAX_TYPE ];

    // 直前のUVに加算
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_FILL ].x = CalculateTextureOffsetDistanceBasedFill( index, prevOffset.x, delta, invStripeLength );
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_FILL ].y = CalculateTextureOffsetDistanceBasedFill( index, prevOffset.y, delta, invStripeLength );
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_FILL ].z = CalculateTextureOffsetDistanceBasedFill( index, prevOffset.z, delta, invStripeLength );
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_TILE ].x = CalculateTextureOffsetDistanceBasedTile( index, prevOffset.x, delta, journey );
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_TILE ].y = CalculateTextureOffsetDistanceBasedTile( index, prevOffset.y, delta, journey );
    offsetBase[ EFT_STRIPE_TEXTURING_OPTION_TILE ].z = CalculateTextureOffsetDistanceBasedTile( index, prevOffset.z, delta, journey );

    // テクスチャリング設定を見て必要な方を取る
    nw::math::VEC3 offset(
        offsetBase[ pStripeData->texturing0 ].x,
        offsetBase[ pStripeData->texturing1 ].y,
        offsetBase[ pStripeData->texturing2 ].z );

    *pOutTexOffset = offset;
}

//---------------------------------------------------------------------------
// エルミート補間で曲線上の位置を求める
//---------------------------------------------------------------------------
inline void CalcHermiteInterpolatedCurveVec(
    nw::math::VEC3* curvePos,
    nw::math::VEC3* curveDir,
    const nw::eft2::SuperStripeHistory* hist,
    const u32 histPos,
    const u32 numHist,
    f32 innerPos )
{
    nw::math::VEC3 startPos, endPos;    // エルミート補間:始点・終点
    nw::math::VEC3 startDir, endDir;    // エルミート補間:開始ベクトル、終了ベクトル（点を通過する向き）
    f32 innerPosInv = 1.0f - innerPos;

    if( histPos == 0 )
    {
        // 先頭部分（3点で近似補間）
        const u32 pStart = 0;
        const u32 pStartNext = 1;
        const u32 pStartNext2 = 2;

        EFT_F32_VEC3_COPY( &startPos, &hist[pStart].pos );
        EFT_F32_VEC3_COPY( &endPos,   &hist[pStartNext].pos );

        nw::math::VEC3 end = 0.5f * ( hist[pStartNext2].pos - hist[pStart].pos );
        EFT_F32_VEC3_COPY( &endDir,   &end );
        nw::math::VEC3 start = 0.25f * (( hist[pStartNext].pos - hist[pStart].pos ) + ( hist[pStartNext].pos - hist[pStartNext2].pos ));
        EFT_F32_VEC3_COPY( &startDir, &start );

        // 進行方向
        curveDir->x = hist[ pStart ].dir.x * innerPosInv + hist[ pStartNext ].dir.x * innerPos;
        curveDir->y = hist[ pStart ].dir.y * innerPosInv + hist[ pStartNext ].dir.y * innerPos;
        curveDir->z = hist[ pStart ].dir.z * innerPosInv + hist[ pStartNext ].dir.z * innerPos;
    }
    else if( histPos >= numHist - 2 )
    {
        // ストライプの末端（3点で近似補間）
        const u32 pEnd = numHist - 1;
        const u32 pEndPrev = pEnd - 1;
        const u32 pEndPrev2 = pEnd - 2;
        if( histPos >= numHist - 1 )
        {
            innerPos = 1.0f;
            innerPosInv = 0.0f;
        }

        // 末端部分
        EFT_F32_VEC3_COPY( &startPos, &hist[pEndPrev].pos );
        EFT_F32_VEC3_COPY( &endPos,   &hist[pEnd].pos );
        nw::math::VEC3 start = 0.5f * ( hist[pEnd].pos - hist[pEndPrev2].pos );
        EFT_F32_VEC3_COPY( &startDir, &start );
        nw::math::VEC3 end = -0.25f * (( hist[pEndPrev].pos - hist[pEnd].pos ) + ( hist[pEndPrev].pos - hist[pEndPrev2].pos ));
        EFT_F32_VEC3_COPY( &endDir, &end );

        // 進行方向
        curveDir->x = hist[ pEndPrev ].dir.x * innerPosInv + hist[ pEnd ].dir.x * innerPos;
        curveDir->y = hist[ pEndPrev ].dir.y * innerPosInv + hist[ pEnd ].dir.y * innerPos;
        curveDir->z = hist[ pEndPrev ].dir.z * innerPosInv + hist[ pEnd ].dir.z * innerPos;
    }
    else
    {
        // 中間部（4点で補間）
        const u32 pNow = histPos;
        const u32 pNext = pNow + 1;
        const u32 pPrev = pNow - 1;
        const u32 pNext2 = pNow + 2;

        EFT_F32_VEC3_COPY( &startPos, &hist[pNow].pos );
        EFT_F32_VEC3_COPY( &endPos,   &hist[pNext].pos );

        nw::math::VEC3 start = ( hist[pNext].pos - hist[pPrev].pos ) * 0.5f;
        EFT_F32_VEC3_COPY( &startDir,   &start );
        nw::math::VEC3 end = ( hist[pNext2].pos - hist[pNow].pos ) * 0.5f;
        EFT_F32_VEC3_COPY( &endDir,   &end );

        // 進行方向
        curveDir->x = hist[ pNow ].dir.x * innerPosInv + hist[ pNext ].dir.x * innerPos;
        curveDir->y = hist[ pNow ].dir.y * innerPosInv + hist[ pNext ].dir.y * innerPos;
        curveDir->z = hist[ pNow ].dir.z * innerPosInv + hist[ pNext ].dir.z * innerPos;
    }
    HermiteInterpolationOnCubic( curvePos, startPos, startDir, endPos, endDir, innerPos );

    return;
}

SuperStripeSystem* sSuperStripeSystem = NULL;

//------------------------------------------------------------------------------
//  ストライプシステムの初期化処理を行います。
//------------------------------------------------------------------------------
void SuperStripeSystem::Initialize( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum )
{
    if( sSuperStripeSystem )
    {
        OutputError( "SuperStripeSystem is Initialized.\n" );
    }

    // プレビュー生成用のメモリを確保します。
    void* buffer = heap->Alloc( sizeof( SuperStripeSystem ) );
    if( !buffer )
    {
        nw::eft2::OutputWarning( "[SuperStripe]Memory Allocate Error!! : %d\n", sizeof( SuperStripeSystem ) );
        return;
    }

    sSuperStripeSystem = new ( buffer )SuperStripeSystem( heap, system, bufferMode, stripeNum );
}

//------------------------------------------------------------------------------
//  ストライプシステムの終了処理を行います。
//------------------------------------------------------------------------------
void SuperStripeSystem::Finalize( nw::eft2::Heap* heap )
{
    sSuperStripeSystem->~SuperStripeSystem();
    heap->Free( sSuperStripeSystem );
    sSuperStripeSystem = NULL;
}


//------------------------------------------------------------------------------
//  ストライプシステムで確保したワークサイズを取得します。
//------------------------------------------------------------------------------
u32 SuperStripeSystem::GetWorkSize()
{
    if( !sSuperStripeSystem ) return 0;
    return sSuperStripeSystem->_GetWorkSize();
}


//------------------------------------------------------------------------------
//  パーティクル生成コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::_ParticleEmitCallback( ParticleEmitArg& arg )
{
    nw::eft2::SuperStripe* stripe = sSuperStripeSystem->Alloc();
    arg.emPluginData = stripe;
    if( stripe )
    {
        nw::math::VEC4 random;
        arg.GetRandom( &random );
        stripe->random.x = random.x;
        stripe->random.y = random.y;
        stripe->random.z = random.z;
        stripe->random.w = random.w;
        stripe->journey = 0.0f;
        stripe->prevFrame = 0;
    }

    return true;
}

//------------------------------------------------------------------------------
//  パーティクル削除コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::_ParticleRemoveCallback( ParticleRemoveArg& arg )
{
    // 遅延ストライプに移行
    if( arg.emPluginData )
    {
        SuperStripe* stripe = reinterpret_cast< SuperStripe* >( arg.emPluginData );
        stripe->numDelayHistory = stripe->numHistory;   //現時点でのストライプ数を保存

        arg.CalcLocalColor0( &stripe->color0 );
        arg.CalcLocalColor1( &stripe->color1 );

        sSuperStripeSystem->AddDelayList( arg.emitter, stripe );
        return true;
    }
    return false;
}

//------------------------------------------------------------------------------
//  パーティクル計算コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::_ParticleCalcCallback( ParticleCalcArg& arg )
{
    SuperStripe* stripe = reinterpret_cast< SuperStripe* >( arg.emPluginData );
    sSuperStripeSystem->CalcStripe( arg, stripe );
}


//------------------------------------------------------------------------------
//  エミッタ生成後コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::_EmitterInitializeCallback( EmitterInitializeArg& arg )
{
    bool result = sSuperStripeSystem->AllocStripeVertexBuffer( arg.emitter );
    return result;
}

//------------------------------------------------------------------------------
//  エミッタ計算処理前コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::_EmitterPreCalcCallback( EmitterPreCalcArg& arg )
{
    sSuperStripeSystem->SwapBuffer( arg.emitter );
}

//------------------------------------------------------------------------------
//  エミッタ計算処理後コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::_EmitterPostCalcCallback( EmitterPostCalcArg& arg )
{
    sSuperStripeSystem->CalcDelayStripe( arg.emitter );
}

//------------------------------------------------------------------------------
//  エミッタ描画コールバック
//------------------------------------------------------------------------------
bool SuperStripeSystem::_EmitterDrawCallback( EmitterDrawArg& arg )
{
    sSuperStripeSystem->Draw( arg.emitter, arg.shaderType, arg.userParam );
    return true;
}

//------------------------------------------------------------------------------
//  エミッタ破棄後コールバック
//------------------------------------------------------------------------------
void SuperStripeSystem::_EmitterFinalizeCallback( EmitterFinalizeArg& arg )
{
    sSuperStripeSystem->FreeDelayStripeList( arg.emitter );
}

//------------------------------------------------------------------------------
// 描画設定後コールバック ダミー用
//------------------------------------------------------------------------------
bool _SuperStripeRenderStateSetCallback( nw::eft2::RenderStateSetArg& arg )
{
    EFT_UNUSED_VARIABLE( arg );
    return true;
}

//---------------------------------------------------------------------------
//  コンストラクタ
//---------------------------------------------------------------------------
SuperStripeSystem::SuperStripeSystem( nw::eft2::Heap* heap, nw::eft2::System* system, BufferMode bufferMode, u32 stripeNum )
{
    mHeap = heap;
    mSystem = system;

    mStripeArrayNum = stripeNum;
    mStripeWorkSize = sizeof( SuperStripe ) * mStripeArrayNum;
    mStripeArray = ( SuperStripe* )mHeap->Alloc( sizeof( SuperStripe ) * mStripeArrayNum );
    mStripeNum = 0;
    mBufferMode = bufferMode;

    for( u32 i = 0; i < mStripeArrayNum; i++ )
    {
        mStripeArray[ i ].used = false;
        mStripeArray[ i ].numHistory = 0;
        mStripeArray[ i ].next = NULL;
    }

    // ストライプコールバックを設定する
    CallbackSet set;
    set.emitterPreCalc = SuperStripeSystem::_EmitterPreCalcCallback;
    set.emitterInitialize = SuperStripeSystem::_EmitterInitializeCallback;
    set.particleEmit = SuperStripeSystem::_ParticleEmitCallback;
    set.particleRemove = SuperStripeSystem::_ParticleRemoveCallback;
    set.particleCalc = SuperStripeSystem::_ParticleCalcCallback;
    set.emitterPostCalc = SuperStripeSystem::_EmitterPostCalcCallback;
    set.emitterDraw = SuperStripeSystem::_EmitterDrawCallback;
    set.emitterFinalize = SuperStripeSystem::_EmitterFinalizeCallback;
    set.renderStateSet = _SuperStripeRenderStateSetCallback;
    mSystem->SetEmitterPluginCallbackSet( EFT_EMITTER_PLUGIN_CALLBACK_ID_3, set );
}

//---------------------------------------------------------------------------
//  デストラクタ
//---------------------------------------------------------------------------
SuperStripeSystem::~SuperStripeSystem()
{
    mHeap->Free( mStripeArray );
}

//---------------------------------------------------------------------------
//  空きストライプインスタンスを確保します。
//---------------------------------------------------------------------------
SuperStripe* SuperStripeSystem::Alloc()
{
    u32 i = 0;

    do
    {
        mStripeArrayIdx++;
        if( mStripeArrayIdx >= mStripeArrayNum )
        {
            mStripeArrayIdx = 0;
        }

        if( mStripeArray[ mStripeArrayIdx ].used == false )
        {
            mStripeArray[ mStripeArrayIdx ].used = true;
            mStripeArray[ mStripeArrayIdx ].numHistory = 0;
            mStripeArray[ mStripeArrayIdx ].numDelayHistory = 0;
            mStripeArray[ mStripeArrayIdx ].next = NULL;
            mStripeArray[ mStripeArrayIdx ].vertexBuffer.pos = NULL;
            mStripeArray[ mStripeArrayIdx ].vertexBuffer.dir = NULL;
            mStripeArray[ mStripeArrayIdx ].vertexBuffer.wing = NULL;
            mStripeArray[ mStripeArrayIdx ].vertexBuffer.tex = NULL;
            mStripeArray[ mStripeArrayIdx ].vertexBuffer.emat = NULL;
            mStripeNum++;

            return &mStripeArray[ mStripeArrayIdx ];
        }
    } while( ( ++i ) < mStripeArrayNum );

    OutputWarning( "SuperStripe is Empty.\n" );
    return NULL;
}

//---------------------------------------------------------------------------
//  ストライプインスタンスを返却します。
//---------------------------------------------------------------------------
void SuperStripeSystem::Free( SuperStripe* stripe )
{
    stripe->used = false;
    mStripeNum--;
}


//---------------------------------------------------------------------------
//  バッファをスワップする。
//---------------------------------------------------------------------------
void SuperStripeSystem::SwapBuffer( Emitter* emitter )
{
    emitter->stripeVertexBufferCurIdx = emitter->stripeVertexBufferStartIdx[ emitter->bufferID ];
    return;
}

//---------------------------------------------------------------------------
//  ストライプインスタンスを返却します。
//---------------------------------------------------------------------------
void SuperStripeSystem::AddDelayList( Emitter* emitter, SuperStripe* stripe )
{
    if( !stripe ) return;
    EFT_ASSERT( mStripeNum != 0 );

    if( emitter->emitterCalc )
    {
        // 遅延リストにストライプを繋ぐ
        if( !emitter->stripeDelayList )
        {
            emitter->stripeDelayList = stripe;
        }
        else
        {
            SuperStripe* temp = reinterpret_cast< SuperStripe* >( emitter->stripeDelayList );
            SuperStripe* tail = NULL;

            while( temp )
            {
                tail = temp;
                temp = temp->next;
            }

            if( tail )
            {
                tail->next = stripe;
            }
        }
    }
    else
    {
        Free( stripe );
    }
}

//---------------------------------------------------------------------------
//! @brief        ポリゴン用ワークを確保します。
//---------------------------------------------------------------------------
bool SuperStripeSystem::AllocStripeVertexBuffer( Emitter* emitter )
{
    if( !emitter )
    {
        return false;
    }

    ResStripeSuper* stripeData = reinterpret_cast< ResStripeSuper* >( emitter->emitterRes->emitterPluginData );

    // ストライプが遅延(尾が縮む時間)で破棄される為、多目にバッファを確保する
    u32 particleNum = emitter->ptclAttrFillMax;
    // 要素数
    u32 element = sizeof( SuperStripeVertexBuffer ) / sizeof( nw::math::VEC4* );

    // 最大頂点数: （履歴数 + 分割数）×Ptcl数×2（一つの履歴から生成する頂点数）×2（ダブルバッファ用）
    const u32 numDiv = static_cast<u32>( stripeData->numDivide );
    const u32 numHist = static_cast<u32>( stripeData->numHistory );
    emitter->stripeVertexBufferNum = ( numHist + ( ( numHist - 1 ) * numDiv ) ) * 2 * particleNum * mBufferMode;

    // 頂点バッファサイズ
    u32 bufferSize = emitter->stripeVertexBufferNum * element * sizeof( nw::math::VEC4 );

    // ポリゴン用バッファを確保
    if( bufferSize > 0 )
    {
        nw::math::VEC4* vec4Buffer = reinterpret_cast< nw::math::VEC4* >( AllocFromDynamicHeap( bufferSize ) );
        if( !vec4Buffer )
        {
            emitter->stripeVertexBuffer = NULL;
            return false;
        }

        emitter->stripeVertexBuffer = vec4Buffer;

        // バッファを切り分ける
        if( mBufferMode == EFT_BUFFER_MODE_DOUBLE )
        {
            emitter->stripeVertexBufferStartIdx[ EFT_BUFFER_ID_0 ] = 0;
            emitter->stripeVertexBufferEndIdx[ EFT_BUFFER_ID_0 ] = emitter->stripeVertexBufferNum / 2;
            emitter->stripeVertexBufferStartIdx[ EFT_BUFFER_ID_1 ] = emitter->stripeVertexBufferNum / 2;
            emitter->stripeVertexBufferEndIdx[ EFT_BUFFER_ID_1 ] = emitter->stripeVertexBufferNum;
            emitter->stripeVertexBufferStartIdx[ EFT_BUFFER_ID_2 ] = 0;
            emitter->stripeVertexBufferEndIdx[ EFT_BUFFER_ID_2 ] = emitter->stripeVertexBufferNum / 2;
        }
        else
        {
            emitter->stripeVertexBufferStartIdx[ EFT_BUFFER_ID_0 ] = 0;
            emitter->stripeVertexBufferEndIdx[ EFT_BUFFER_ID_0 ] = emitter->stripeVertexBufferNum / 3;
            emitter->stripeVertexBufferStartIdx[ EFT_BUFFER_ID_1 ] = emitter->stripeVertexBufferNum / 3;
            emitter->stripeVertexBufferEndIdx[ EFT_BUFFER_ID_1 ] = emitter->stripeVertexBufferNum / 3 * 2;
            emitter->stripeVertexBufferStartIdx[ EFT_BUFFER_ID_2 ] = emitter->stripeVertexBufferNum / 3 * 2;
            emitter->stripeVertexBufferEndIdx[ EFT_BUFFER_ID_2 ] = emitter->stripeVertexBufferNum;
        }
    }
    else
    {
        emitter->stripeVertexBuffer = NULL;
    }

    return true;
}

//---------------------------------------------------------------------------
//  遅延解放ストライプを破棄します。
//---------------------------------------------------------------------------
void SuperStripeSystem::FreeDelayStripeList( Emitter* emitter )
{
    if( emitter->stripeVertexBuffer )
    {
        FreeFromDynamicHeap( emitter->stripeVertexBuffer, false );
        emitter->stripeVertexBuffer = NULL;
    }

    if( !emitter->stripeDelayList ) return;
    SuperStripe* temp = reinterpret_cast< SuperStripe* >( emitter->stripeDelayList );

    while( temp )
    {
        Free( temp );
        temp = temp->next;
    }

    emitter->stripeDelayList = NULL;
}

//---------------------------------------------------------------------------
//  遅延ストライプ計算処理を行います。
//---------------------------------------------------------------------------
void SuperStripeSystem::CalcDelayStripe( Emitter* emitter )
{
    const f32 frameRate = emitter->frameRate;
    if( !emitter->stripeDelayList )
    {
        return;
    }

    SuperStripe* stripe = reinterpret_cast< SuperStripe* >( emitter->stripeDelayList );
    ResStripeSuper* stripeData = reinterpret_cast< ResStripeSuper* >( emitter->emitterRes->emitterPluginData );
    SuperStripe* preStripe = NULL;

    while( stripe )
    {
        SuperStripe* next = stripe->next;

        //---------------------------------------------------------------------
        // 履歴を動かす
        {
            const f32 histAirRegist = stripeData->historyAirRegist;
            const f32 invRate = 1.0f - frameRate;
            const f32 airRegist = histAirRegist + ( 1.0f - histAirRegist ) * invRate;

            // 履歴速度を更新（加速度）
            for( u32 i = 0; i < stripe->numHistory; ++i )
            {
                stripe->hist[ i ].vec += stripeData->historyAcceleration * frameRate;
                stripe->hist[ i ].vec *= airRegist;
            }

            // 履歴位置を更新
            for( u32 i = 0; i < stripe->numHistory; ++i )
            {
                stripe->hist[ i ].pos += stripe->hist[ i ].vec * frameRate;
            }
        }
        //---------------------------------------------------------------------
        // 履歴を取り除くかどうかチェック
        //---------------------------------------------------------------------
        stripe->time += emitter->frameRate;
        const f32 testPrevFrame = math::FFloor( stripe->prevFrame );
        const f32 testNowFrame = math::FFloor( stripe->time );
        const bool doRemoveHistory = ( testNowFrame - testPrevFrame > 0 );
        stripe->prevFrame = stripe->time; // ストライプ時間（前回更新）を記憶

        // 死亡判定
        if( stripe->numHistory > 1 )
        {
            if( doRemoveHistory )
            {
                stripe->numHistory--;
            }
            if( emitter->stripeVertexBuffer )
            {
                // スケールフェードイン／アウト
                nw::math::VEC4* vec4Buffer = reinterpret_cast< nw::math::VEC4* >( emitter->stripeVertexBuffer );
                u32 segemnet = emitter->stripeVertexBufferNum;
                stripe->vertexBuffer.pos  = &vec4Buffer[ segemnet * 0 + emitter->stripeVertexBufferCurIdx ];
                stripe->vertexBuffer.dir  = &vec4Buffer[ segemnet * 1 + emitter->stripeVertexBufferCurIdx ];
                stripe->vertexBuffer.wing = &vec4Buffer[ segemnet * 2 + emitter->stripeVertexBufferCurIdx ];
                stripe->vertexBuffer.tex  = &vec4Buffer[ segemnet * 3 + emitter->stripeVertexBufferCurIdx ];
                stripe->vertexBuffer.emat = &vec4Buffer[ segemnet * 4 + emitter->stripeVertexBufferCurIdx ];
                f32 uvIndexOffset = static_cast<f32>( stripe->numDelayHistory - stripe->numHistory );

                if( stripeData->numDivide == 0 )
                {
                    // 分割無し
                    MakeStripeVertexBuffer( &stripe->vertexBuffer, stripe, stripeData, emitter, uvIndexOffset );
                }
                else
                {
                    // 分割あり
                    uvIndexOffset *= ( stripeData->numDivide + 1 );
                    MakeStripeVertexBufferWithDivision( &stripe->vertexBuffer, stripe, stripeData, emitter, uvIndexOffset );
                }
                emitter->stripeVertexBufferCurIdx += stripe->vertexBufferNum;

                EFT_ASSERT( emitter->stripeVertexBufferNum > emitter->stripeVertexBufferCurIdx );
            }
            else
            {
                stripe->vertexBuffer.pos = NULL;
                stripe->vertexBufferNum = 0;
            }

            preStripe = stripe;
            stripe = next;
        }
        else
        {
            if( preStripe )
            {
                preStripe->next = next;
            }
            else
            {
                emitter->stripeDelayList = next;
            }
            Free( stripe );

            stripe = next;
        }
    }
}

//---------------------------------------------------------------------------
//  ストライプ計算処理を行います。
//---------------------------------------------------------------------------
void SuperStripeSystem::CalcStripe( ParticleCalcArg& arg, SuperStripe* stripe )
{
    if( !stripe || !stripe->used )
    {
        return;
    }
    //-------------------------------
    //-------------------------------
    ResStripeSuper* stripeData = reinterpret_cast< ResStripeSuper* >( arg.emitter->emitterRes->emitterPluginData );

    //-------------------------------
    // カラーの更新
    //-------------------------------
    arg.CalcLocalColor0( &stripe->color0 );     // ストライプカラー0
    arg.CalcLocalColor1( &stripe->color1 );     // ストライプカラー1

    //-------------------------------
    // 履歴を積むか積まないかのチェック
    //-------------------------------
    stripe->time = arg.GetTime();       // ストライプ時間の更新
    stripe->life = arg.GetLife();       // ストライプの寿命を設定
    const f32 testPrevFrame = math::FFloor( stripe->prevFrame );
    const f32 testNowFrame = math::FFloor( stripe->time );
    const bool doPushHistory = ( testNowFrame - testPrevFrame > 0 ) || ( testNowFrame == 0.0f );
    stripe->prevFrame = arg.GetTime();  // ストライプ時間（前回更新）を記憶

    //-------------------------------
    // 履歴の更新
    //-------------------------------
    UpdateHistory( arg, stripe, stripeData, doPushHistory );

    //-------------------------------
    // ポリゴンバッファを生成
    //-------------------------------
    if( arg.emitter->stripeVertexBuffer )
    {
        nw::math::VEC4* vec4Buffer = reinterpret_cast< nw::math::VEC4* >( arg.emitter->stripeVertexBuffer );

        u32 segemnet = arg.emitter->stripeVertexBufferNum;
        stripe->vertexBuffer.pos  = &vec4Buffer[ segemnet * 0 + arg.emitter->stripeVertexBufferCurIdx ];
        stripe->vertexBuffer.dir  = &vec4Buffer[ segemnet * 1 + arg.emitter->stripeVertexBufferCurIdx ];
        stripe->vertexBuffer.wing = &vec4Buffer[ segemnet * 2 + arg.emitter->stripeVertexBufferCurIdx ];
        stripe->vertexBuffer.tex  = &vec4Buffer[ segemnet * 3 + arg.emitter->stripeVertexBufferCurIdx ];
        stripe->vertexBuffer.emat = &vec4Buffer[ segemnet * 4 + arg.emitter->stripeVertexBufferCurIdx ];

#if EFT_IS_CAFE
        // pos, dir
        const u32 bufferNum = stripe->numHistory * 2;
        const u32 dcbzSize = bufferNum * sizeof( nw::math::VEC4 );
        MemUtil::ZeroRange( stripe->vertexBuffer.pos, dcbzSize );
        MemUtil::ZeroRange( stripe->vertexBuffer.dir, dcbzSize );
        MemUtil::ZeroRange( stripe->vertexBuffer.wing, dcbzSize );
        MemUtil::ZeroRange( stripe->vertexBuffer.tex, dcbzSize );
        if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
        {
            MemUtil::ZeroRange( stripe->vertexBuffer.emat, dcbzSize );
        }
#endif
        if( stripeData->numDivide == 0 )
        {
            // 分割無し
            MakeStripeVertexBuffer( &stripe->vertexBuffer, stripe, stripeData, arg.emitter );
        }
        else
        {
            // 分割あり
            MakeStripeVertexBufferWithDivision( &stripe->vertexBuffer, stripe, stripeData, arg.emitter );
        }
        arg.emitter->stripeVertexBufferCurIdx += stripe->vertexBufferNum;

        // メモリ領域オーバーのチェック
        EFT_ASSERT( arg.emitter->stripeVertexBufferCurIdx <= arg.emitter->stripeVertexBufferNum );
    }
    else
    {
        stripe->vertexBuffer.pos = NULL;
        stripe->vertexBufferNum = 0;
    }
}

//---------------------------------------------------------------------------
//  ストライプの履歴の更新を行います（複雑になったので分離）
//---------------------------------------------------------------------------
void SuperStripeSystem::UpdateHistory(
    ParticleCalcArg& arg,
    SuperStripe* stripe,
    ResStripeSuper* stripeData,
    const bool doPushHistory )
{
    const f32 frameRate = arg.emitter->frameRate;
    const f32 histAirRegist = stripeData->historyAirRegist;
    const f32 invRate = 1.0f - frameRate;
    const f32 airRegist = histAirRegist + ( 1.0f - histAirRegist ) * invRate;

    nw::math::VEC4 col0, col1;
    arg.CalcLocalColor0( &col0 );
    arg.CalcLocalColor1( &col1 );
    if( doPushHistory == false )
    {
        // 先頭の履歴を合わせる
        nw::math::VEC3 scale;
        arg.CalcWorldScale( &scale );

        nw::math::VEC3 prevPos;
        EFT_F32_VEC3_COPY( &prevPos, &stripe->hist[ 0 ].pos );
        if( stripeData->emitterFollow )
        {
            // エミッタに完全追従：ローカル系で保存
            arg.GetLocalPos( &stripe->hist[ 0 ].pos );
        }
        else
        {
            // エミッタに完全追従しない：ワールド系で保存
            arg.GetWorldPos( &stripe->hist[ 0 ].pos );
        }
        stripe->journey += ( stripe->hist[ 0 ].pos - prevPos ).Length();

        stripe->hist[ 0 ].scale = scale.x;
        stripe->hist[ 0 ].emitterSRT_Y.x = arg.emitter->matrixSRT.m[ 0 ][ 1 ];
        stripe->hist[ 0 ].emitterSRT_Y.y = arg.emitter->matrixSRT.m[ 1 ][ 1 ];
        stripe->hist[ 0 ].emitterSRT_Y.z = arg.emitter->matrixSRT.m[ 2 ][ 1 ];

        // 履歴を動かす
        {
            // 履歴速度を更新（加速度＆空気抵抗）
            const f32 temp = frameRate * airRegist;
            nw::math::VEC3 accel;
            accel.x = stripeData->historyAcceleration.x * temp;
            accel.y = stripeData->historyAcceleration.y * temp;
            accel.z = stripeData->historyAcceleration.z * temp;
            for( u32 i = 1; i < stripe->numHistory; ++i )
            {
                stripe->hist[ i ].vec.x = ( stripe->hist[ i ].vec.x * airRegist + accel.x );
                stripe->hist[ i ].vec.y = ( stripe->hist[ i ].vec.y * airRegist + accel.y );
                stripe->hist[ i ].vec.z = ( stripe->hist[ i ].vec.z * airRegist + accel.z );

                stripe->hist[ i ].pos.x += stripe->hist[ i ].vec.x * frameRate;
                stripe->hist[ i ].pos.y += stripe->hist[ i ].vec.y * frameRate;
                stripe->hist[ i ].pos.z += stripe->hist[ i ].vec.z * frameRate;
            }
        }
    }
    else
    {
        if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_RIBBON )
        {
            //------------------------------------------------------------------------------
            // リボン以外
            //------------------------------------------------------------------------------
            // 履歴を積む（位置差分が0以上であることが確定）

            // 履歴をずらす
            if( stripe->numHistory > 0 )
            {
                if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
                {
                    // ひとつ後ろまで延ばしてずらす（先頭に履歴が積まれるため）
                    for( u32 i = stripe->numHistory; i > 0; i-- )
                    {
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].pos, &stripe->hist[ i - 1 ].pos );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].dir, &stripe->hist[ i - 1 ].dir );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].vec, &stripe->hist[ i - 1 ].vec );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].wing, &stripe->hist[ i - 1 ].wing );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].emitterSRT_Y, &stripe->hist[ i - 1 ].emitterSRT_Y );
                        stripe->hist[ i ].scale = stripe->hist[ i - 1 ].scale;
                    }
                }
                else
                {
                    // ひとつ後ろまで延ばしてずらす（先頭に履歴が積まれるため）
                    for( u32 i = stripe->numHistory; i > 0; i-- )
                    {
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].pos, &stripe->hist[ i - 1 ].pos );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].dir, &stripe->hist[ i - 1 ].dir );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].vec, &stripe->hist[ i - 1 ].vec );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].wing, &stripe->hist[ i - 1 ].wing );
                        stripe->hist[ i ].scale = stripe->hist[ i - 1 ].scale;
                    }
                }
            }

            // 履歴を積む
            nw::math::VEC3 scale;
            arg.CalcWorldScale( &scale );

            nw::math::VEC3 prevPos;
            EFT_F32_VEC3_COPY( &prevPos, &stripe->hist[ 0 ].pos );
            if( stripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で保存
                arg.GetLocalPos( &stripe->hist[ 0 ].pos );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で保存
                arg.GetWorldPos( &stripe->hist[ 0 ].pos );
            }

            if( stripe->numHistory == 0 )
            {
                stripe->journey += stripe->hist[ 0 ].pos.Length();
            }
            else
            {
                stripe->journey += ( stripe->hist[ 0 ].pos - prevPos ).Length();
            }

            stripe->hist[ 0 ].scale = scale.x;

            if( stripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で位置を覚えるので上方向はあくまでY軸方向
                stripe->hist[0].emitterSRT_Y = nw::math::VEC3( 0, 1, 0 );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で位置を覚えるので上方向はエミッタ行列のY軸方向
                stripe->hist[0].emitterSRT_Y = nw::math::VEC3(
                    arg.emitter->matrixSRT.m[0][1],
                    arg.emitter->matrixSRT.m[1][1],
                    arg.emitter->matrixSRT.m[2][1] );
            }

            // 履歴に与える初速（仮）
            {
                nw::math::VEC4 random;
                arg.GetRandom( &random );
                const f32 hVecD = stripeData->historyVecInitSpeed;
                const f32 t = stripe->time;

                nw::math::VEC3 hVecCycRatio;
                hVecCycRatio.x = stripeData->historyInitVecRotateCycle.x * stripeData->historyVecRegulation;
                hVecCycRatio.y = stripeData->historyInitVecRotateCycle.y * stripeData->historyVecRegulation;
                hVecCycRatio.z = stripeData->historyInitVecRotateCycle.z * stripeData->historyVecRegulation;

                stripe->hist[ 0 ].vec = nw::math::VEC3(
                    nw::math::SinDeg( t * ( hVecCycRatio.x ) + 360.0f * random.x ) + nw::math::SinDeg( t * 0.5f + 360.0f * random.y ),
                    nw::math::SinDeg( t * ( hVecCycRatio.y ) + 360.0f * random.y ) + nw::math::SinDeg( t * 0.5f + 360.0f * random.z ),
                    nw::math::SinDeg( t * ( hVecCycRatio.z ) + 360.0f * random.z ) + nw::math::SinDeg( t * 0.5f + 360.0f * random.w ) ) * hVecD;
            }

            if( stripe->numHistory < stripeData->numHistory )
            {
                stripe->numHistory++;
            }

            //-------------------------------
            // 背骨の方向（hist.dir）・羽根の計算のための方向（hist.wing）を設定
            //-------------------------------
            // TODO: まだまだ改良の余地あり

            SuperStripeHistory* histPrev = &stripe->hist[ 1 ];
            SuperStripeHistory* histNow = &stripe->hist[ 0 ];
            {
                if( stripe->numHistory <= 1 )
                {
                    if( stripeData->emitterFollow )
                    {
                        // エミッタに完全追従：ローカル系で保存
                        arg.GetLocalVec( &histNow->dir );   // 最初の進行方向はPtclの初速
                    }
                    else
                    {
                        // エミッタに完全追従しない：ワールド系で保存
                        arg.GetWorldVec( &histNow->dir );   // 最初の進行方向はPtclの初速
                    }

                    if( histNow->dir.Length() > 0 )
                    {
                        histNow->dir.Normalize();
                        EFT_F32_VEC3_COPY( &histNow->wing, &histNow->dir );
                    }
                    else
                    {
                        // 初回かつ速度0（速度0＋自動移動などのケース）
                        nw::math::VEC3 emitterPos;
                        emitterPos.x = arg.emitter->matrixSRT._03;
                        emitterPos.y = arg.emitter->matrixSRT._13;
                        emitterPos.z = arg.emitter->matrixSRT._23;

                        nw::math::VEC3 worldDiff = emitterPos - arg.emitter->emitterPrevPos;
                        if( worldDiff.Length() > 0 )
                        {
                            // エミッタが動いている場合、位置差分を方向にする。
                            histNow->dir = worldDiff;
                            histNow->dir.Normalize();
                            EFT_F32_VEC3_COPY( &histNow->wing, &histNow->dir );
                        }
                        else
                        {
                            // まったく動いていない場合。仮の向きを入れておく。
                            histNow->dir = nw::math::VEC3( 0, 0.001f, 0 );
                            EFT_F32_VEC3_COPY( &histNow->wing, &histNow->dir );
                        }
                    }
                }
                else
                {
                    histNow->dir = histNow->pos - histPrev->pos;

                    if( stripe->numHistory == 2 )
                    {
                        // 履歴点が二つになった時、最初の方を今できた向きで上書き
                        EFT_F32_VEC3_COPY( &histPrev->dir, &histNow->dir );
                    }

                    // 平均をとって全体に適用（テスト）
                    nw::math::VEC3 av( 0, 0, 0 );
                    for( u32 i = 0; i < stripe->numHistory; ++i )
                    {
                        av += stripe->hist[ i ].dir * static_cast< f32 >( ( stripe->numHistory - i ) );
                    }

                    if( av.Length() > 0 )
                    {
                        av.Normalize();
                    }

                    EFT_F32_VEC3_COPY( &histNow->wing, &av );
                    if( stripe->numHistory == 2 )
                    {
                        // 履歴点が二つになった時、最初の方を今できた向きで上書き
                        EFT_F32_VEC3_COPY( &histPrev->wing, &histNow->wing );
                    }
                }

                {
                    nw::math::VEC3 wingDir; // VEC4からVEC3だけ抜き出す
                    EFT_F32_VEC3_COPY( &wingDir, &histNow->wing );
                    if( wingDir.Length() > 0 )
                    {
                        wingDir.Normalize();
                    }
                    else
                    {
                        // 向きがどうしても決定しなかった場合
                        wingDir.x = 0;
                        wingDir.y = 1;
                        wingDir.z = 0;
                    }
                    EFT_F32_VEC3_COPY( &histNow->wing, &wingDir );
                }
            }

            // 履歴を動かす
            {
                // 履歴速度を更新（加速度＆空気抵抗）
                const f32 temp = frameRate * airRegist;
                nw::math::VEC3 accel;
                accel.x = stripeData->historyAcceleration.x * temp;
                accel.y = stripeData->historyAcceleration.y * temp;
                accel.z = stripeData->historyAcceleration.z * temp;
                for( u32 i = 1; i < stripe->numHistory; ++i )
                {
                    stripe->hist[ i ].vec.x = ( stripe->hist[ i ].vec.x * airRegist + accel.x );
                    stripe->hist[ i ].vec.y = ( stripe->hist[ i ].vec.y * airRegist + accel.y );
                    stripe->hist[ i ].vec.z = ( stripe->hist[ i ].vec.z * airRegist + accel.z );

                    stripe->hist[ i ].pos.x += stripe->hist[ i ].vec.x * frameRate;
                    stripe->hist[ i ].pos.y += stripe->hist[ i ].vec.y * frameRate;
                    stripe->hist[ i ].pos.z += stripe->hist[ i ].vec.z * frameRate;
                }
            }
        }
        else
        {
            //------------------------------------------------------------------------------
            // リボンストライプ
            //------------------------------------------------------------------------------

            //------------------------------------------------------------------------------
            // 履歴を積む（位置差分が0以上であることが確定）
            //------------------------------------------------------------------------------
            // 履歴をずらす
            if( stripe->numHistory > 0 )
            {
                if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD && stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_RIBBON )
                {
                    // ひとつ後ろまで延ばしてずらす（先頭に履歴が積まれるため）
                    for( u32 i = stripe->numHistory; i > 0; i-- )
                    {
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].pos, &stripe->hist[ i - 1 ].pos );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].dir, &stripe->hist[ i - 1 ].dir );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].vec, &stripe->hist[ i - 1 ].vec );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].wing, &stripe->hist[ i - 1 ].wing );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].emitterSRT_Y, &stripe->hist[ i - 1 ].emitterSRT_Y );
                        stripe->hist[ i ].scale = stripe->hist[ i - 1 ].scale;
                    }
                }
                else
                {
                    // ひとつ後ろまで延ばしてずらす（先頭に履歴が積まれるため）
                    for( u32 i = stripe->numHistory; i > 0; i-- )
                    {
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].pos, &stripe->hist[ i - 1 ].pos );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].dir, &stripe->hist[ i - 1 ].dir );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].vec, &stripe->hist[ i - 1 ].vec );
                        EFT_F32_VEC3_COPY( &stripe->hist[ i ].wing, &stripe->hist[ i - 1 ].wing );
                        stripe->hist[ i ].scale = stripe->hist[ i - 1 ].scale;
                    }
                }
            }
            SuperStripeHistory* const pNowHistory = &stripe->hist[ 0 ];     // 新しい履歴（最後尾）
            SuperStripeHistory* const pPrevHistory = &stripe->hist[ 1 ];

            //------------------------------------------------------------------------------
            // 履歴カウンタの更新
            //------------------------------------------------------------------------------
            if( stripe->numHistory < stripeData->numHistory - 1 )
            {
                stripe->numHistory++;
            }

            //------------------------------------------------------------------------------
            // 位置を決定：既存の履歴点の更新より前に行う必要がある
            //------------------------------------------------------------------------------
            if( stripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で保存
                arg.GetLocalPos( &pNowHistory->pos );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で保存
                arg.GetWorldPos( &pNowHistory->pos );
            }

            //------------------------------------------------------------------------------
            // 背骨方向の計算の前に、既存の履歴点の位置速度を更新する必要があるのでこのタイミングで行う。
            //------------------------------------------------------------------------------
            {
                // 履歴速度を更新（加速度＆空気抵抗）
                const f32 temp = frameRate * airRegist;
                nw::math::VEC3 accel;
                accel.x = stripeData->historyAcceleration.x * temp;
                accel.y = stripeData->historyAcceleration.y * temp;
                accel.z = stripeData->historyAcceleration.z * temp;
                for( u32 i = 1; i < stripe->numHistory; ++i )
                {
                    stripe->hist[ i ].vec.x = ( stripe->hist[ i ].vec.x * airRegist + accel.x );
                    stripe->hist[ i ].vec.y = ( stripe->hist[ i ].vec.y * airRegist + accel.y );
                    stripe->hist[ i ].vec.z = ( stripe->hist[ i ].vec.z * airRegist + accel.z );

                    stripe->hist[ i ].pos.x += stripe->hist[ i ].vec.x * frameRate;
                    stripe->hist[ i ].pos.y += stripe->hist[ i ].vec.y * frameRate;
                    stripe->hist[ i ].pos.z += stripe->hist[ i ].vec.z * frameRate;
                }
            }

            //------------------------------------------------------------------------------
            // 履歴点の初速の決定
            //------------------------------------------------------------------------------
            {
                nw::math::VEC4 random;
                arg.GetRandom( &random );
                const float hVecD = stripeData->historyVecInitSpeed;
                const float t = stripe->time;

                nw::math::VEC3 hVecCycRatio(
                    stripeData->historyInitVecRotateCycle.x * stripeData->historyVecRegulation,
                    stripeData->historyInitVecRotateCycle.y * stripeData->historyVecRegulation,
                    stripeData->historyInitVecRotateCycle.z * stripeData->historyVecRegulation );

                pNowHistory->vec = nw::math::VEC3(
                    nw::math::SinDeg( t * ( hVecCycRatio.x ) + 360.0f * random.x ) + nw::math::SinDeg( t * 0.5f + 360.0f * random.y ),
                    nw::math::SinDeg( t * ( hVecCycRatio.y ) + 360.0f * random.y ) + nw::math::SinDeg( t * 0.5f + 360.0f * random.z ),
                    nw::math::SinDeg( t * ( hVecCycRatio.z ) + 360.0f * random.z ) + nw::math::SinDeg( t * 0.5f + 360.0f * random.w ) ) * hVecD;
            }

            //------------------------------------------------------------------------------
            // 履歴点の emat 方向の決定
            //------------------------------------------------------------------------------
            if( stripeData->emitterFollow )
            {
                // エミッタに完全追従：ローカル系で位置を覚えるので上方向はあくまでY軸方向
                pNowHistory->emitterSRT_Y = nw::math::VEC3( 0, 1, 0 );
            }
            else
            {
                // エミッタに完全追従しない：ワールド系で位置を覚えるので上方向はエミッタ行列のY軸方向
                pNowHistory->emitterSRT_Y = nw::math::VEC3(
                    arg.emitter->matrixSRT.m[0][1],
                    arg.emitter->matrixSRT.m[1][1],
                    arg.emitter->matrixSRT.m[2][1] );
            }

            //------------------------------------------------------------------------------
            // 履歴点のスケール値の設定
            //------------------------------------------------------------------------------
            {
                nw::math::VEC3 scale;
                arg.CalculateWorldScale( &scale );
                pNowHistory->scale = scale.x;
            }

            //------------------------------------------------------------------------------
            // 履歴点の背骨方向・羽根の展開方向の決定
            // 直前の履歴点が上段の処理で加速度による更新を受けているのに依存している
            //------------------------------------------------------------------------------
            {
                const float Epsilon = 0.00001f; // 「ほぼ」0とみなす値
                if( stripe->numHistory == 1 )
                {
                    if( stripeData->emitterFollow )
                    {
                        // エミッタに完全追従：ローカル系で保存
                        arg.GetLocalVec( &pNowHistory->dir );   // 最初の進行方向はパーティクルの初速
                    }
                    else
                    {
                        // エミッタに完全追従しない：ワールド系で保存
                        arg.GetWorldVec( &pNowHistory->dir );   // 最初の進行方向はパーティクルの初速
                    }

                    // 加速度の向きを加算しておく
                    // MEMO: 加速度＞＞速度のケースで初期のねじれを抑制できる
                    {
                        const float ratio = arg.GetLife();
                        const nw::math::VEC3 accel(
                            stripeData->historyAcceleration.x * ratio,
                            stripeData->historyAcceleration.y * ratio,
                            stripeData->historyAcceleration.z * ratio );
                        pNowHistory->dir += accel;
                    }

                    // 最初の履歴点（※前の履歴が無いので単品で向きを決める必要がある）
                    if( pNowHistory->dir.LengthSquare() < Epsilon )
                    {
                        pNowHistory->dir = nw::math::VEC3( 0, 1, 0 );
                        pNowHistory->wing.x = 1;
                        pNowHistory->wing.y = 0;
                        pNowHistory->wing.z = 0;
                    }
                    else
                    {
                        const nw::math::VEC3 axisY( 0, 1, 0 );
                        nw::math::VEC3 dir = pNowHistory->dir;
                        dir.Normalize();
                        const float d = nw::math::VEC3Dot( dir, axisY );
                        if( ::std::abs( ::std::abs( d ) - 1.0f ) < Epsilon )
                        {
                            nw::math::VEC3 wing( 1, 0, 0 );
                            wing *= d;
                            wing.Normalize();
                            EFT_F32_VEC3_COPY( &pNowHistory->wing, &wing );
                        }
                        else
                        {
                            nw::math::VEC3 wing;
                            nw::math::VEC3Cross( &wing, dir, axisY );
                            wing.Normalize();
                            EFT_F32_VEC3_COPY( &pNowHistory->wing, &wing );
                        }
                    }
                }
                else
                {
                    pNowHistory->dir = pNowHistory->pos - pPrevHistory->pos;

                    // 二つ目以降の履歴点
                    if( pNowHistory->dir.LengthSquare() < Epsilon )
                    {
                        // 前回からの位置差分を速度とみなす
                        nw::math::VEC3 prevVec = pNowHistory->pos - pPrevHistory->pos;

                        if( prevVec.LengthSquare() < Epsilon )
                        {
                            // 前回の点も動いてないなら引継ぎ
                            EFT_F32_VEC3_COPY( &pNowHistory->wing, &pPrevHistory->wing );
                            EFT_F32_VEC3_COPY( &pNowHistory->dir, &pPrevHistory->dir );
                        }
                        else
                        {
                            // ある程度動いている場合はその速度を向きとして展開方向を計算。
                            const nw::math::VEC3 axisY( 0, 1, 0 );
                            nw::math::VEC3 dir = prevVec;
                            dir.Normalize();
                            const float d = nw::math::VEC3Dot( dir, axisY );
                            if( ::std::abs( ::std::abs( d ) - 1.0f ) < Epsilon )
                            {
                                nw::math::VEC3 wing( 1, 0, 0 );
                                wing *= d;
                                wing.Normalize();
                                EFT_F32_VEC3_COPY( &pNowHistory->wing, &wing );
                            }
                            else
                            {
                                nw::math::VEC3 wing;
                                nw::math::VEC3Cross( &wing, dir, axisY );
                                wing.Normalize();
                                EFT_F32_VEC3_COPY( &pNowHistory->wing, &wing );
                            }
                        }
                    }
                    else
                    {
                        // pNowHistory->dir が 非零の場合
                        nw::math::VEC3 nowDir = pNowHistory->dir;
                        nw::math::VEC3 prevDir = pPrevHistory->dir;
                        nw::math::VEC3 prevWingDir;
                        EFT_F32_VEC3_COPY( &prevWingDir, &pPrevHistory->wing );

                        nowDir.Normalize();
                        prevDir.Normalize();
                        prevWingDir.Normalize();

                        // 前回の法線方向
                        nw::math::VEC3 prevNormal;
                        nw::math::VEC3Cross( &prevNormal, prevWingDir, prevDir );
                        prevNormal.Normalize();

                        // 法線→進行方向から展開方向を決定
                        nw::math::VEC3 wing;
                        nw::math::VEC3Cross( &wing, nowDir, prevNormal );
                        wing.Normalize();
                        EFT_F32_VEC3_COPY( &pNowHistory->wing, &wing );
                    }

                    if( stripe->numHistory == 2 )
                    {
                        if( arg.pEmitter->GetResEmitter()->ptcl.isRotateZ )
                        {
                            // Zロールで回す。
                            // pNowHistory->dir を軸に 一定量回転させる。
                            // 必ず背骨の向きが必要なので、ストライプが描画され始める直前であるこのタイミングで行う。
                            pNowHistory->dir.Normalize();
                            pNowHistory->wing.Normalize();
                            nw::math::VEC4 random;
                            arg.GetRandom( &random );

                            const float randomInitZ = arg.emitter->GetEmitterResource()->resEmitterStaticUniformBlock->rotateInitRand.z;
                            const float rz = arg.emitter->GetEmitterResource()->resEmitterStaticUniformBlock->rotateInit.z + random.x * randomInitZ;
                            const float ax = pNowHistory->dir.x;
                            const float ay = pNowHistory->dir.y;
                            const float az = pNowHistory->dir.z;

                            const float theta = rz;

                            const float c = nw::math::CosRad( theta );
                            const float s = nw::math::SinRad( theta );
                            nw::math::MTX34 mat(
                                ax*ax*( 1 - c ) + c, ax*ay*( 1 - c ) - az*s, az*ax*( 1 - c ) + ay*s,
                                ax*ay*( 1 - c ) + az*s, ay*ay*( 1 - c ) + c, ay*az*( 1 - c ) - ax*s,
                                az*ax*( 1 - c ) - ay*s, ay*az*( 1 - c ) + ax*s, az*az*( 1 - c ) + c,
                                0, 0, 0
                                );
                            nw::math::VEC3 temp;
                            EFT_F32_VEC3_COPY( &temp, &pNowHistory->wing );
                            nw::math::VEC3TransformNormal( &temp, mat, temp );
                            EFT_F32_VEC3_COPY( &pNowHistory->wing, &temp );
                        }

                        // 二つ目の履歴点（初めてストライプが見えるタイミング）の時、
                        // 初回の履歴点の wing を今回のモノと揃えておく。
                        pPrevHistory->wing = pNowHistory->wing;
                        pPrevHistory->dir = pNowHistory->dir;
                        pPrevHistory->wing.Normalize();
                        pPrevHistory->dir.Normalize();
                    }
                }

                // 最終的に決まったベクトルをここで正規化
                pNowHistory->dir.Normalize();
                pNowHistory->wing.Normalize();
            }

            //------------------------------------------------------------------------------
            // 移動距離の更新
            //------------------------------------------------------------------------------
            {
                if( stripe->numHistory == 1 )
                {
                    stripe->journey += pNowHistory->pos.Length();
                }
                else
                {
                    nw::math::VEC3 prevPos;
                    prevPos = pPrevHistory->pos;
                    nw::math::VEC3 temp = pNowHistory->pos - prevPos;
                    stripe->journey += temp.Length();
                }
            }
        }
    }
}

//---------------------------------------------------------------------------
//  ストライプの頂点バッファの共通部分を作成
//---------------------------------------------------------------------------
inline void MapDefaultVertexData(
    SuperStripeVertexBuffer* vertexBuffer,
    const u32 index,
    const nw::math::VEC3& pos,
    const nw::math::VEC3& dir,
    const nw::math::VEC3& wing,
    const nw::math::VEC3& textureOffset,
    f32 scaleFadeValue )
{
    NW_NULL_ASSERT( vertexBuffer );

    const u32 idxL = index * 2;    // 頂点（右側）Index
    const u32 idxR = idxL + 1;   // 頂点（左側）Index

    // 履歴の位置xyz / 羽根の長さ
    EFT_F32_VEC3_COPY( &vertexBuffer->pos[ idxL ], &pos );
    EFT_F32_VEC3_COPY( &vertexBuffer->pos[ idxR ], &pos );
    vertexBuffer->pos[ idxL ].w =  scaleFadeValue;
    vertexBuffer->pos[ idxR ].w = -scaleFadeValue;

    // 背骨の方向（前の履歴との位置差分）
    EFT_F32_VEC3_COPY( &vertexBuffer->dir[ idxL ], &dir );
    EFT_F32_VEC3_COPY( &vertexBuffer->dir[ idxR ], &dir );

    // 羽根の方向
    EFT_F32_VEC3_COPY( &vertexBuffer->wing[ idxL ], &wing );
    EFT_F32_VEC3_COPY( &vertexBuffer->wing[ idxR ], &wing );
    //vertexBuffer->wing[idxL].w = 0;
    //vertexBuffer->wing[idxR].w = 0;

    // テクスチャリング
    EFT_F32_VEC4_COPY( &vertexBuffer->tex[ idxL ], &textureOffset );
    EFT_F32_VEC4_COPY( &vertexBuffer->tex[ idxR ], &textureOffset );
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。
//---------------------------------------------------------------------------
void SuperStripeSystem::MakeStripeVertexBuffer(
    SuperStripeVertexBuffer* vertexBuffer,
    SuperStripe* stripe,
    const ResStripeSuper* stripeData,
    const Emitter* emitter,
    f32 indexOffset )
{
    //---------------------------------------------------------------------------
    // 履歴ポイントが2つは無いとポリゴンにならないので処理を返す。
    //---------------------------------------------------------------------------
    if( stripe->numHistory <= 1 )
    {
        stripe->vertexBuffer.pos = NULL;
        stripe->vertexBufferNum = 0;
        return;
    }

    const u32 historyCount = stripe->numHistory;
    u32 index = 0;

    //---------------------------------------------------------------------------
    // ストライプの全長を求める
    //---------------------------------------------------------------------------
    f32 stripeLengthInv = 0.0f;
    if( stripeData->textureUvMapType == EFT_TEXTURE_MAPPING_TYPE_DISTANCE_BASED )
    {
        stripeLengthInv = 1.0f / CalculateStripeLength( stripe, historyCount );
    }

    //---------------------------------------------------------------------------
    // スケールフェード値を計算
    //---------------------------------------------------------------------------
    const float scaleFadeValue = emitter->CalculateCurrentScaleFadeValue();

    //---------------------------------------------------------------------------
    //  テクスチャオフセット値の計算
    //---------------------------------------------------------------------------
    const float texOffset = fract( stripe->time ) + indexOffset;

    //---------------------------------------------------------------------------
    //  頂点バッファの要素を埋める
    //---------------------------------------------------------------------------
    if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        for( index = 0; index < historyCount; ++index )
        {
            const u32 idxL = index * 2;     // 頂点（右側）Index
            const u32 idxR = idxL + 1;      // 頂点（左側）Index
            s32 prevPos = idxL - 2;
            if( prevPos < 0 )
            {
                prevPos = 0;
            }

            nw::math::VEC3 textureUv( 0, 0, 0 );
            switch( stripeData->textureUvMapType )
            {
            case EFT_TEXTURE_MAPPING_TYPE_UNIFORM:
                {
                    const int maxVertexCount = static_cast< int >( stripeData->numHistory );
                    CalculateTextureOffsetUniform( &textureUv, stripeData, index, historyCount, maxVertexCount, texOffset );
                    break;
                }
            case EFT_TEXTURE_MAPPING_TYPE_DISTANCE_BASED:
                {
                    int prevVertexIndex = ( index * 2 ) - 2;
                    if( prevVertexIndex < 0 )
                    {
                        prevVertexIndex = 0;
                    }
                    const nw::math::VEC3 sub = stripe->hist[ index - 1 ].pos - stripe->hist[ index ].pos;
                    const float delta = sub.Length();
                    const nw::math::VEC4& preTex = vertexBuffer->tex[ prevVertexIndex ];

                    CalculateTextureOffsetDistanceBased( &textureUv, stripeData, index, delta, stripe->journey, stripeLengthInv, preTex );
                    break;
                }
            default:
                NW_ASSERT( 0 ); // ここには来ない！
                break;
            }


            // 新アトリビュートに全部まとめる
            MapDefaultVertexData(
                vertexBuffer,
                index,
                stripe->hist[index].pos,
                stripe->hist[index].dir,
                stripe->hist[index].wing,
                textureUv,
                stripe->hist[index].scale * scaleFadeValue );

            // EmitterMatrixAxisY も追加
            EFT_F32_VEC4_COPY( &vertexBuffer->emat[ idxL ], &stripe->hist[ index ].emitterSRT_Y );
            EFT_F32_VEC4_COPY( &vertexBuffer->emat[ idxR ], &stripe->hist[ index ].emitterSRT_Y );
        }
    }
    else
    {
        for( index = 0; index < historyCount; ++index )
        {
            s32 prevPos = ( index * 2 ) - 2;
            if( prevPos < 0 )
            {
                prevPos = 0;
            }

            nw::math::VEC3 textureUv( 0, 0, 0 );
            switch( stripeData->textureUvMapType )
            {
            case EFT_TEXTURE_MAPPING_TYPE_UNIFORM:
                {
                    const int maxVertexCount = static_cast< int >( stripeData->numHistory );
                    CalculateTextureOffsetUniform( &textureUv, stripeData, index, historyCount, maxVertexCount, texOffset );
                    break;
                }
            case EFT_TEXTURE_MAPPING_TYPE_DISTANCE_BASED:
                {
                    int prevVertexIndex = ( index * 2 ) - 2;
                    if( prevVertexIndex < 0 )
                    {
                        prevVertexIndex = 0;
                    }
                    const nw::math::VEC3 sub = stripe->hist[ index - 1 ].pos - stripe->hist[ index ].pos;
                    const float delta = sub.Length();
                    const nw::math::VEC4& preTex = vertexBuffer->tex[ prevVertexIndex ];

                    CalculateTextureOffsetDistanceBased( &textureUv, stripeData, index, delta, stripe->journey, stripeLengthInv, preTex );
                    break;
                }
            default:
                NW_ASSERT( 0 ); // ここには来ない！
                break;
            }

            // 新アトリビュートに全部まとめる
            MapDefaultVertexData(
                vertexBuffer,
                index,
                stripe->hist[index].pos,
                stripe->hist[index].dir,
                stripe->hist[index].wing,
                textureUv,
                stripe->hist[index].scale * scaleFadeValue );
        }
    }
    stripe->vertexBufferNum = index * 2;

#if EFT_IS_CAFE
    MemUtil::FlushCache( vertexBuffer->pos, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->dir, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->wing, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->tex, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        MemUtil::FlushCache( vertexBuffer->emat, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    }
#endif

    return;
}

//---------------------------------------------------------------------------
//  ストライプのポリゴン生成を行います。（ストライプ分割版）
//---------------------------------------------------------------------------
void SuperStripeSystem::MakeStripeVertexBufferWithDivision(
    SuperStripeVertexBuffer* vertexBuffer,
    SuperStripe* stripe,
    const ResStripeSuper* stripeData,
    const Emitter* emitter,
    f32 indexOffset )
{
    //---------------------------------------------------------------------------
    //  履歴ポイントが3つは無いとポリゴンにならないので処理を返す。
    //---------------------------------------------------------------------------
    if( stripe->numHistory <= 2 )
    {
        stripe->vertexBuffer.pos = NULL;
        stripe->vertexBufferNum = 0;
        return;
    }

    const u32 loop = stripe->numHistory;

    //---------------------------------------------------------------------------
    //  ストライプの全長を求める
    //---------------------------------------------------------------------------
    f32 stripeLengthInv = 0.0f;
    if( stripeData->textureUvMapType == EFT_TEXTURE_MAPPING_TYPE_DISTANCE_BASED )
    {
        stripeLengthInv = 1.0f / CalculateStripeLength( stripe, loop );
    }

    //---------------------------------------------------------------------------
    //  スケールフェード値を計算
    //---------------------------------------------------------------------------
    const float scaleFadeValue = emitter->CalculateCurrentScaleFadeValue();

    //---------------------------------------------------------------------------
    //  テクスチャオフセット値の計算
    //---------------------------------------------------------------------------
    const float texOffset = fract( stripe->time ) + indexOffset;

    const u32 numHist = static_cast< u32 >( stripe->numHistory );
    const SuperStripeHistory* hist = stripe->hist;
    const u32 numDiv = static_cast< u32 >( stripeData->numDivide ); // 分割数。
    const u32 numDivPoint = ( ( numHist - 1 ) * numDiv );           // 分割により発生する履歴ポイントの数
    const u32 numDrawHist = numHist + numDivPoint;                  // 分割分の頂点を追加。
    const f32 fIndexDiv = 1.0f / ( 1 + numDiv );

    u32 idxL = 0;
    u32 idxR = 0;
    u32 index = 0;

    //---------------------------------------------------------------------------
    //  頂点バッファの要素を埋める
    //---------------------------------------------------------------------------
    nw::math::VEC3 prevVertexPos( 0, 0, 0 );
    if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        // Emat が必要なストライプタイプの場合。
        for( index = 0; index < numDrawHist; ++index )
        {
            idxL  = index * 2;
            idxR  = idxL + 1;

            const f32 fIndex = index * fIndexDiv;                                   // ストライプ状の位置（履歴ポイント場が整数になる）
            const u32 histPos = static_cast<u32>( nw::math::FFloor( fIndex ) );     // 所属する履歴ポイント
            f32 innerPos = fIndex - histPos;                                        // 履歴ポイント内の位置 [0.0f, 1.0f]

            nw::math::VEC3 curvePos;            // 履歴ポイントの位置（エルミート補間後）
            nw::math::VEC3 curveDir;            // 履歴ポイントの向き（線形補間後）

            s32 prevPos = idxL - 2;
            if( prevPos < 0 )
            {
                prevPos = 0;
            }

            //-----------------------------------------------------------------
            // 履歴点の間をエルミート補間
            //-----------------------------------------------------------------
            CalcHermiteInterpolatedCurveVec( &curvePos, &curveDir, hist, histPos, numHist, innerPos );

            // テクスチャ座標
            nw::math::VEC3 textureUv( 0, 0, 0 );
            {
                switch( stripeData->textureUvMapType )
                {
                case EFT_TEXTURE_MAPPING_TYPE_UNIFORM:
                    {
                        const int numHistory = static_cast< int >( stripeData->numHistory );
                        const int maxVertexCount = ( numHistory + ( ( numHistory - 1 ) * numDiv ) );
                        CalculateTextureOffsetUniform( &textureUv, stripeData, index, numDrawHist, maxVertexCount, texOffset );
                        break;
                    }
                case EFT_TEXTURE_MAPPING_TYPE_DISTANCE_BASED:
                    {
                        float delta = 0.0f;
                        if( index > 0 )
                        {
                            nw::math::VEC3 sub = curvePos - prevVertexPos;
                            delta = sub.Length();
                        }

                        const nw::math::VEC4& preTex = vertexBuffer->tex[ prevPos ];

                        CalculateTextureOffsetDistanceBased( &textureUv, stripeData, index, delta, stripe->journey, stripeLengthInv, preTex );

                        // ひとつ前の頂点位置を記憶
                        prevVertexPos = curvePos;
                        break;
                    }
                default:
                    NW_ASSERT( 0 ); // ここには来ない！
                    break;
                }
            }

            // 新アトリビュートに全部まとめる
            MapDefaultVertexData(
                vertexBuffer,
                index,
                curvePos,
                curveDir,
                stripe->hist[ histPos ].wing,
                textureUv,
                stripe->hist[ histPos ].scale * scaleFadeValue );

            // EmitterMatrixAxisY も追加
            EFT_F32_VEC4_COPY( &vertexBuffer->emat[ idxL ], &stripe->hist[ histPos ].emitterSRT_Y );
            EFT_F32_VEC4_COPY( &vertexBuffer->emat[ idxR ], &stripe->hist[ histPos ].emitterSRT_Y );
            prevVertexPos = curvePos;
        }
    }
    else
    {
        // ビルボードストライプの時はEmatが不要！
        for( index = 0; index < numDrawHist; ++index )
        {
            idxL  = index * 2;
            idxR  = idxL + 1;

            const f32 fIndex = index * fIndexDiv;                                   // ストライプ状の位置（履歴ポイント場が整数になる）
            const u32 histPos = static_cast<u32>( nw::math::FFloor( fIndex ) );     // 所属する履歴ポイント
            f32 innerPos = fIndex - histPos;                                        // 履歴ポイント内の位置 [0.0f, 1.0f]

            nw::math::VEC3 curvePos;            // 履歴ポイントの位置（エルミート補間後）
            nw::math::VEC3 curveDir;            // 履歴ポイントの向き（線形補間後）

            s32 prevPos = idxL - 2;
            if( prevPos < 0 )
            {
                prevPos = 0;
            }

            //-----------------------------------------------------------------
            // 間を曲線補間
            //-----------------------------------------------------------------
            CalcHermiteInterpolatedCurveVec( &curvePos, &curveDir, hist, histPos, numHist, innerPos );

            nw::math::VEC3 textureUv( 0, 0, 0 );
            {
                switch( stripeData->textureUvMapType )
                {
                case EFT_TEXTURE_MAPPING_TYPE_UNIFORM:
                    {
                        const int numHistory = static_cast< int >( stripeData->numHistory );
                        const int maxVertexCount = ( numHistory + ( ( numHistory - 1 ) * numDiv ) );
                        CalculateTextureOffsetUniform( &textureUv, stripeData, index, numDrawHist, maxVertexCount, texOffset );
                        break;
                    }
                case EFT_TEXTURE_MAPPING_TYPE_DISTANCE_BASED:
                    {
                        float delta = 0.0f;
                        if( index > 0 )
                        {
                            nw::math::VEC3 sub = curvePos - prevVertexPos;
                            delta = sub.Length();
                        }

                        const nw::math::VEC4& preTex = vertexBuffer->tex[ prevPos ];

                        CalculateTextureOffsetDistanceBased( &textureUv, stripeData, index, delta, stripe->journey, stripeLengthInv, preTex );

                        // ひとつ前の頂点位置を記憶
                        prevVertexPos = curvePos;
                        break;
                    }
                default:
                    NW_ASSERT( 0 ); // ここには来ない！
                    break;
                }
            }

            // 新アトリビュートに全部まとめる
            MapDefaultVertexData(
                vertexBuffer,
                index,
                curvePos,
                curveDir,
                stripe->hist[ histPos ].wing,
                textureUv,
                stripe->hist[ histPos ].scale * scaleFadeValue );

            // EmitterMatrixAxisY も追加
            EFT_F32_VEC4_COPY( &vertexBuffer->emat[ idxL ], &stripe->hist[ histPos ].emitterSRT_Y );
            EFT_F32_VEC4_COPY( &vertexBuffer->emat[ idxR ], &stripe->hist[ histPos ].emitterSRT_Y );
            prevVertexPos = curvePos;
        }
    }
    stripe->vertexBufferNum = index * 2;

#if EFT_IS_CAFE
    MemUtil::FlushCache( vertexBuffer->pos, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->dir, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->wing, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    MemUtil::FlushCache( vertexBuffer->tex, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    if( stripeData->calcType != EFT_STRIPE_POLYGON_TYPE_BILLBOARD )
    {
        MemUtil::FlushCache( vertexBuffer->emat, sizeof( nw::math::VEC4 ) * stripe->vertexBufferNum );
    }
#endif

    return;
}

//---------------------------------------------------------------------------
//  ストライプ描画処理を行います。
//---------------------------------------------------------------------------
void SuperStripeSystem::Draw( Emitter* emitter, ShaderType shaderType, void* userParam  )
{
    System* system = emitter->emitterSet->GetSystem();

    // シェーダ切り替え
    Shader* shader = emitter->shader[shaderType];
    shader->Bind();

    struct StripeUniformBlockParam
    {
        nw::math::VEC4 random;
        nw::math::VEC4 param0;
        nw::math::VEC4 color0;
        nw::math::VEC4 color1;
        nw::math::VEC4 param1;
    };

    // 履歴ストライプデータ
    ResStripeSuper* stripeData = reinterpret_cast<ResStripeSuper*>( emitter->emitterRes->emitterPluginData );

    // 描画設定
    ResRenderState* state = &emitter->emitterData->renderState;
    Rendercontext::SetRenderStateResource( state, system->GetColorBufferEnbaleFlag() );

    // テクスチャ設定
    for ( u32 i = 0; i < EFT_TEXTURE_SLOT_MAX; i++ )
    {
        TextureSlot slot = static_cast<TextureSlot>( i );
        TextureResource* texture = emitter->emitterRes->texture[i];

        if ( texture )
        {
            u32 fragLoc = shader->GetFragmentTextureSamplerLocation( slot );
            u32 vertLoc = shader->GetVertexTextureSamplerLocation( slot );

            if ( texture->Bind( fragLoc, vertLoc, slot ) )
            {
                emitter->emitterRes->textureSampler[i].Bind( fragLoc, vertLoc, slot );
            }
        }
    }

    /*
        TODO:サンプラの間借りを辞める。
    */

    // フレームバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetFrameBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetFrameBufferTextureVertexSamplerLocation();

        if ( system->GetFrameBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB ) )
        {
            emitter->emitterRes->textureSampler[0].Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_FB );
        }
    }

    // デプスバッファテクスチャの設定
    {
        u32 fragLoc = shader->GetDepthBufferTextureFragmentSamplerLocation();
        u32 vertLoc = shader->GetDepthBufferTextureVertexSamplerLocation();

        if ( system->GetDepthBufferTexture().Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB ) )
        {
            emitter->emitterRes->textureSampler[0].Bind( fragLoc, vertLoc, EFT_TEXTURE_SLOT_EXT_DB );
        }
    }

    // エミッタ動的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_DYNAMIC, &emitter->dynamicUBO[emitter->bufferID] );

    // エミッタ静的UBOの設定
    shader->BindUniformBlock( Shader::EFT_UBO_TYPE_EMITTER_STATIC, &emitter->emitterRes->emitterStaticUbo );

    // カスタムシェーダコールバックの呼び出し
    {
        RenderStateSetArg rssArg;
        rssArg.emitter      = emitter;
        rssArg.userParam    = userParam;
        rssArg.shaderType   = shaderType;
        rssArg.isStreamOut  = false;

        mSystem->ResetCustomShaderSetting( shader );
        {
            // 描画設定後コールバックの呼び出し
            bool ret = InvokeRenderStateSetCallback( rssArg );
            if( !ret ){ return; }

            // DrawPath コールバックの呼び出し
            if ( emitter->drawPathCallback ) { emitter->drawPathCallback( rssArg ); }
        }
        mSystem->ResetCustomShaderSetting( NULL );
    }

    // 遅延リストの描画
    if ( emitter->stripeDelayList )
    {
        SuperStripe* stripe = reinterpret_cast<SuperStripe*>( emitter->stripeDelayList );
        while( stripe )
        {
            const nw::math::VEC4 uboParam0(
                stripeData->headAlpha,
                stripeData->tailAlpha,
                stripeData->headScale,
                stripeData->tailScale );

            u32 vertexNum = stripe->vertexBufferNum;

            if ( vertexNum > 0 && stripe->vertexBuffer.pos )
            {
                // ストライプUBOを設定する
                TemporaryUniformBlockBuffer stripeTempUbo;
                StripeUniformBlockParam* ubo = reinterpret_cast< StripeUniformBlockParam* >( stripeTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
                if ( !ubo )
                {
                    stripe = stripe->next;
                    continue;
                }

                EFT_F32_VEC4_COPY( &ubo->param0, &uboParam0 );
                EFT_F32_VEC4_COPY( &ubo->random, &stripe->random );
                EFT_F32_VEC4_COPY( &ubo->color0, &stripe->color0 );
                EFT_F32_VEC4_COPY( &ubo->color1, &stripe->color1 );
                ubo->param1.x = stripe->time;
                ubo->param1.y = static_cast<f32>( stripe->numHistory );
                ubo->param1.z = EFT_STRIPE_NORMAL_MODE;
                ubo->param1.w = stripe->life;

                stripeTempUbo.Validate();

                // ストライプデータのUBO設定
                shader->BindEmitterPluginUniformBlock( &stripeTempUbo );

                // 頂点を設定
                VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_0 ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.pos );
                VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_1 ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.dir );
                VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_2 ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.wing );
                VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_3 ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.tex );
                VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_4 ),
                    sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.emat );

                // 描画
                Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
                if( stripeData->option == EFT_STRIPE_CROSS_MODE )
                {
                    // クロスの場合、再度描画する！
                    // ストライプUBOを設定する
                    TemporaryUniformBlockBuffer stripeCrossTempUbo;
                    StripeUniformBlockParam* ubo2 = reinterpret_cast< StripeUniformBlockParam* >( stripeCrossTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
                    if ( !ubo2 )
                    {
                        stripe = stripe->next;
                        continue;
                    }

                    EFT_F32_VEC4_COPY( &ubo2->param0, &uboParam0 );
                    EFT_F32_VEC4_COPY( &ubo2->random, &stripe->random );
                    EFT_F32_VEC4_COPY( &ubo2->color0, &stripe->color0 );
                    EFT_F32_VEC4_COPY( &ubo2->color1, &stripe->color1 );
                    ubo2->param1.x = stripe->time;
                    ubo2->param1.y = static_cast< f32 >( stripe->numHistory );
                    ubo2->param1.z = EFT_STRIPE_CROSS_MODE;
                    ubo2->param1.w = stripe->life;

                    stripeCrossTempUbo.Validate();

                    // ストライプデータのUBO設定
                    shader->BindEmitterPluginUniformBlock( &stripeCrossTempUbo );

                    // 描画
                    Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
                }
            }
            stripe = stripe->next;
        }
    }

    // 描画
    for ( u32 i = 0; i < emitter->ptclNum; i++ )
    {
        SuperStripe* stripe = reinterpret_cast<SuperStripe*>( emitter->particleData[i].emitterPluginData );
        if ( !stripe || !stripe->used || !stripe->vertexBuffer.pos ) { continue; }

        u32 vertexNum = stripe->vertexBufferNum;
        const nw::math::VEC4 uboParam0(
            stripeData->headAlpha,
            stripeData->tailAlpha,
            stripeData->headScale,
            stripeData->tailScale );

        if ( vertexNum > 0 )
        {
            // ストライプUBOを設定する
            TemporaryUniformBlockBuffer stripeTempUbo;
            StripeUniformBlockParam* ubo = reinterpret_cast< StripeUniformBlockParam* >( stripeTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
            if ( !ubo )
            {
                stripe = stripe->next;
                continue;
            }

            EFT_F32_VEC4_COPY( &ubo->param0, &uboParam0 );
            EFT_F32_VEC4_COPY( &ubo->random, &stripe->random );
            EFT_F32_VEC4_COPY( &ubo->color0, &stripe->color0 );
            EFT_F32_VEC4_COPY( &ubo->color1, &stripe->color1 );
            ubo->param1.x = stripe->time;
            ubo->param1.y = static_cast<f32>( stripe->numHistory );
            ubo->param1.z = EFT_STRIPE_NORMAL_MODE;
            ubo->param1.w = stripe->life;


            stripeTempUbo.Validate();

            // ストライプデータのUBO設定
            shader->BindEmitterPluginUniformBlock( &stripeTempUbo );

            // 頂点を設定
            VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_0 ),
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.pos );
            VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_1 ),
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.dir );
            VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_2 ),
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.wing );
            VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_3 ),
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.tex );
            VertexBuffer::BindExtBuffer( shader->GetEmitterPluginAttribute( Shader::EFT_EMTR_PLUGIN_ATTR_IDX_4 ),
                sizeof(nw::math::VEC4) * vertexNum, 4, sizeof(nw::math::VEC4), stripe->vertexBuffer.emat );

            // 描画
            Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
            if( stripeData->option == EFT_STRIPE_CROSS_MODE )
            {
                // クロスの場合、再度描画する！
                // ストライプUBOを設定する
                TemporaryUniformBlockBuffer stripeCrossTempUbo;
                StripeUniformBlockParam* ubo2 = reinterpret_cast< StripeUniformBlockParam* >( stripeCrossTempUbo.Alloc( system, sizeof( StripeUniformBlockParam ) ) );
                if ( !ubo2 )
                {
                    stripe = stripe->next;
                    continue;
                }

                EFT_F32_VEC4_COPY( &ubo2->param0, &uboParam0 );
                EFT_F32_VEC4_COPY( &ubo2->random, &stripe->random );
                EFT_F32_VEC4_COPY( &ubo2->color0, &stripe->color0 );
                EFT_F32_VEC4_COPY( &ubo2->color1, &stripe->color1 );
                ubo2->param1.x = stripe->time;
                ubo2->param1.y = static_cast< f32 >( stripe->numHistory );
                ubo2->param1.z = EFT_STRIPE_CROSS_MODE;
                ubo2->param1.w = stripe->life;

                stripeCrossTempUbo.Validate();

                // ストライプデータのUBO設定
                shader->BindEmitterPluginUniformBlock( &stripeCrossTempUbo );

                // 描画
                Render::DrawPrimitive( Render::PRIM_TYPE_TRIANGLE_STRIP, 0, vertexNum );
            }
        }
    }
}

//------------------------------------------------------------------------------
//  ワンタイムエミッタ用の追加のエミッタ寿命を取得
//------------------------------------------------------------------------------
s32 SuperStripeSystem::GetExtendedEndTimeForOneTimeEmitter( nw::eft2::Emitter* pEmitter )
{
    ResStripeSuper* pStripeData = reinterpret_cast<ResStripeSuper*>( pEmitter->emitterRes->emitterPluginData );
    f32 extendLife = pStripeData->numHistory;
    if( pEmitter->frameRate > 1.0 )
    {
        extendLife *= pEmitter->frameRate;
    }
    return static_cast< s32 >( extendLife );
}

} // namespace eft2
} // namespace nw

#endif      // EFT_DEGRADATION_SPEC
