﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/eft/eft2_Enum.h>
#include <nw/eft/eft2_Shader.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_System.h>


namespace nw   {
namespace eft2 {

// コールバック未設定、カスタムシェーダパラメータ付加 の場合は、
// システム側で自動的にfloatの配列としてUboを設定する。
struct CustomShaderReservedUniformBlock
{
    nw::math::VEC4      param[8];
    void FlipEndian()
    {
#ifdef  EFT_ENDIAN_LITTLE
        nw::eft2::EndianUtil::Flip( &param[0] );
        nw::eft2::EndianUtil::Flip( &param[1] );
        nw::eft2::EndianUtil::Flip( &param[2] );
        nw::eft2::EndianUtil::Flip( &param[3] );
        nw::eft2::EndianUtil::Flip( &param[4] );
        nw::eft2::EndianUtil::Flip( &param[5] );
        nw::eft2::EndianUtil::Flip( &param[6] );
        nw::eft2::EndianUtil::Flip( &param[7] );
#endif//EFT_ENDIAN_LITTLE
    }
};


//---------------------------------------------------------------------------
//  無効化
//---------------------------------------------------------------------------
void Shader::Invalidate()
{
    m_Attr[EFT_ATTR_IDX_POSITION] = static_cast<u32>( EFT_INVALID_SHADER_ATTRIBUTE );
    m_EmitterResSet = NULL;
    m_ShaderRes.Invalidate();
    m_IsInitialized = false;
}


#ifdef EFT_OGL
//---------------------------------------------------------------------------
//  エミッタリソースをセットします。
//---------------------------------------------------------------------------
void Shader::Initialize( const EmitterResource* emitterResSet, const char* addShaderDef )
{
    m_EmitterResSet = emitterResSet;
    m_AddShaderDef  = addShaderDef;
    m_ShaderRes.Initialize( m_EmitterResSet, m_AddShaderDef );
}

//---------------------------------------------------------------------------
//  シェーダを初期化します。
//---------------------------------------------------------------------------
bool Shader::CompileShader()
{
    EFT_ASSERT( !m_IsInitialized );

    // カスタムシェーダ有りでコンパイル
    m_ShaderRes.Compile( m_EmitterResSet, false );

    // カスタムシェーダ有りでコンパイルできなかった場合は、
    // カスタムシェーダ無しでコンパイルを行う
    if ( m_ShaderRes.IsError() )
    {
        EFT_PRINT( "CustomShader Compile Error Occurred\n" );
        m_ShaderRes.Compile( m_EmitterResSet, true );
        if ( m_ShaderRes.IsError() )
        {
            EFT_ERR( "DefaultShader Compile Error\n" );
        }
    }

    m_IsInitialized  = true;
    return true;
}

//---------------------------------------------------------------------------
//  ストリームアウトシェーダを初期化します。
//---------------------------------------------------------------------------
void Shader::InitializeStreamOutShader()
{
    EFT_ASSERT( !m_IsInitialized );
    m_ShaderRes.InitializeStreamOutShader();
    m_IsInitialized  = true;
}
#endif

//---------------------------------------------------------------------------
//  シェーダを初期化します。
//---------------------------------------------------------------------------
#if EFT_GX2
void Shader::InitializeShader( Heap* heap, void* binary, u32 vertexIdx, u32 fragIdx )
{
    EFT_ASSERT( !m_IsInitialized );

    m_VertexIdx = vertexIdx;
    m_FragIdx   = fragIdx;
    m_ShaderRes.Initialize( heap, binary, vertexIdx, fragIdx );

    m_IsInitialized  = true;
}
#endif

//---------------------------------------------------------------------------
//  ロケーションを初期化します。
//---------------------------------------------------------------------------
void Shader::InitializeLocation()
{
    m_Attr[EFT_ATTR_IDX_TEXTURE_COORD0]            = static_cast<u32>( EFT_INVALID_SHADER_ATTRIBUTE );

    // アトリビュート
    m_Attr[EFT_ATTR_IDX_POSITION]                  = m_ShaderRes.GetAttribute( "sysPosAttr",            0, FORMAT_32_32_32_32_FLOAT,   0, false );
    m_Attr[EFT_ATTR_IDX_NORMAL]                    = m_ShaderRes.GetAttribute( "sysNormalAttr",         2, FORMAT_32_32_32_FLOAT,      0, false );
    m_Attr[EFT_ATTR_IDX_TANGENT]                   = m_ShaderRes.GetAttribute( "sysTangentAttr",        15,FORMAT_32_32_32_32_FLOAT,   0, false );
    m_Attr[EFT_ATTR_IDX_VERTEX_COLOR0]             = m_ShaderRes.GetAttribute( "sysVertexColor0Attr",   3, FORMAT_32_32_32_32_FLOAT,   0, false );
    m_Attr[EFT_ATTR_IDX_VERTEX_COLOR1]             = m_ShaderRes.GetAttribute( "sysVertexColor1Attr",   4, FORMAT_32_32_32_32_FLOAT,   0, false );
    m_Attr[EFT_ATTR_IDX_TEXTURE_COORD0]            = m_ShaderRes.GetAttribute( "sysTexCoordAttr",       5, FORMAT_32_32_32_32_FLOAT,   0, false );

#ifndef EFT_DEGRADATION_SPEC
    m_StreamAttr[EFT_STREAM_ATTR_IDX_POSITION]     = m_ShaderRes.GetAttribute( "sysInPos",              10, FORMAT_32_32_32_32_FLOAT,  0, true, true );
    m_StreamAttr[EFT_STREAM_ATTR_IDX_VECTOR]       = m_ShaderRes.GetAttribute( "sysInVec",              11, FORMAT_32_32_32_32_FLOAT,  0, true, true );
    m_EmtrPluginAttr[EFT_EMTR_PLUGIN_ATTR_IDX_0]   = m_ShaderRes.GetAttribute( "sysEmitterPluginAttr0",  6, FORMAT_32_32_32_32_FLOAT,  0, false );
    m_EmtrPluginAttr[EFT_EMTR_PLUGIN_ATTR_IDX_1]   = m_ShaderRes.GetAttribute( "sysEmitterPluginAttr1",  7, FORMAT_32_32_32_32_FLOAT,  0, false );
    m_EmtrPluginAttr[EFT_EMTR_PLUGIN_ATTR_IDX_2]   = m_ShaderRes.GetAttribute( "sysEmitterPluginAttr2", 12, FORMAT_32_32_32_32_FLOAT,  0, false );
    m_EmtrPluginAttr[EFT_EMTR_PLUGIN_ATTR_IDX_3]   = m_ShaderRes.GetAttribute( "sysEmitterPluginAttr3", 13, FORMAT_32_32_32_32_FLOAT,  0, false );
    m_EmtrPluginAttr[EFT_EMTR_PLUGIN_ATTR_IDX_4]   = m_ShaderRes.GetAttribute( "sysEmitterPluginAttr4", 14, FORMAT_32_32_32_32_FLOAT,  0, false );
#endif

#ifdef EFT_USE_UNIFORM_REGISTER
    // ビューレジスタ
    m_Register[EFT_REGISTER_LOC_VIEW_MATRIX]        = m_ShaderRes.GetUniformLocation( "VWM" );
    m_Register[EFT_REGISTER_LOC_PROJ_MATRIX]        = m_ShaderRes.GetUniformLocation( "PJM" );
    m_Register[EFT_REGISTER_LOC_VIEW_PROJ_MATRIX]   = m_ShaderRes.GetUniformLocation( "VPM" );
    m_Register[EFT_REGISTER_LOC_BILLBOARD_MATRIX]   = m_ShaderRes.GetUniformLocation( "BLM" );
    m_Register[EFT_REGISTER_LOC_CAMERA_POS]         = m_ShaderRes.GetUniformLocation( "EPS" );
    m_Register[EFT_REGISTER_LOC_CAMERA_VEC]         = m_ShaderRes.GetUniformLocation( "EVC" );
    m_Register[EFT_REGISTER_LOC_VIEW_PARAM]         = m_ShaderRes.GetUniformLocation( "VWP" );

    // エミッタレジスタ
    m_Register[EFT_REGISTER_LOC_EMITTER_PARAM0]     = m_ShaderRes.GetUniformLocation( "EP0" );
    m_Register[EFT_REGISTER_LOC_EMITTER_PARAM1]     = m_ShaderRes.GetUniformLocation( "EP1" );
    m_Register[EFT_REGISTER_LOC_EMITTER_COLOR0]     = m_ShaderRes.GetUniformLocation( "EC0" );
    m_Register[EFT_REGISTER_LOC_EMITTER_COLOR1]     = m_ShaderRes.GetUniformLocation( "EC1" );
    m_Register[EFT_REGISTER_LOC_EMITTER_SRT_MATIRX] = m_ShaderRes.GetUniformLocation( "EMX" );
    m_Register[EFT_REGISTER_LOC_EMITTER_RT_MATIRX]  = m_ShaderRes.GetUniformLocation( "EMR" );
#endif

#ifdef EFT_USE_REGISTER_DRAW
    // パーティクルレジスタ
    m_PtclAttr[EFT_PTCL_ATTR_IDX_LOCAL_POSITION]    = m_ShaderRes.GetUniformLocation( "sysLocalPosAttr"  );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_LOCAL_VECTOR]      = m_ShaderRes.GetUniformLocation( "sysLocalVecAttr"  );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_LOCAL_POS_DELTA]   = m_ShaderRes.GetUniformLocation( "sysLocalDiffAttr" );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_SCALE]             = m_ShaderRes.GetUniformLocation( "sysScaleAttr"     );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_RANDOM]            = m_ShaderRes.GetUniformLocation( "sysRandomAttr"    );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_ROTATE]            = m_ShaderRes.GetUniformLocation( "sysInitRotateAttr");
    m_PtclAttr[EFT_PTCL_ATTR_IDX_COLOR0]            = m_ShaderRes.GetUniformLocation( "sysColor0Attr"    );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_COLOR1]            = m_ShaderRes.GetUniformLocation( "sysColor1Attr"    );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX0]     = m_ShaderRes.GetUniformLocation( "sysEmtMat0Attr"   );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX1]     = m_ShaderRes.GetUniformLocation( "sysEmtMat1Attr"   );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX2]     = m_ShaderRes.GetUniformLocation( "sysEmtMat2Attr"   );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_RT_MTX0]      = m_ShaderRes.GetUniformLocation( "sysEmtRTMat0Attr" );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_RT_MTX1]      = m_ShaderRes.GetUniformLocation( "sysEmtRTMat1Attr" );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_RT_MTX2]      = m_ShaderRes.GetUniformLocation( "sysEmtRTMat2Attr" );
#endif
#ifdef EFT_USE_INSTANS_DRAW
    // パーティクルアトリビュート
    m_PtclAttr[EFT_PTCL_ATTR_IDX_LOCAL_POSITION]    = m_ShaderRes.GetAttribute( "sysLocalPosAttr",       8, FORMAT_32_32_32_32_FLOAT,   0, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_LOCAL_VECTOR]      = m_ShaderRes.GetAttribute( "sysLocalVecAttr",       8, FORMAT_32_32_32_32_FLOAT,   4, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_LOCAL_POS_DELTA]   = m_ShaderRes.GetAttribute( "sysLocalDiffAttr",      8, FORMAT_32_32_32_32_FLOAT,   8, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_SCALE]             = m_ShaderRes.GetAttribute( "sysScaleAttr" ,         9, FORMAT_32_32_32_32_FLOAT,   0, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_RANDOM]            = m_ShaderRes.GetAttribute( "sysRandomAttr",         9, FORMAT_32_32_32_32_FLOAT,   4, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_ROTATE]            = m_ShaderRes.GetAttribute( "sysInitRotateAttr",     9, FORMAT_32_32_32_32_FLOAT,   8, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_COLOR0]            = m_ShaderRes.GetAttribute( "sysColor0Attr",         9, FORMAT_32_32_32_32_FLOAT,  12, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_COLOR1]            = m_ShaderRes.GetAttribute( "sysColor1Attr",         9, FORMAT_32_32_32_32_FLOAT,  16, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX0]     = m_ShaderRes.GetAttribute( "sysEmtMat0Attr",        9, FORMAT_32_32_32_32_FLOAT,  20, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX1]     = m_ShaderRes.GetAttribute( "sysEmtMat1Attr",        9, FORMAT_32_32_32_32_FLOAT,  24, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_SRT_MTX2]     = m_ShaderRes.GetAttribute( "sysEmtMat2Attr",        9, FORMAT_32_32_32_32_FLOAT,  28, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_RT_MTX0]      = m_ShaderRes.GetAttribute( "sysEmtRTMat0Attr",      9, FORMAT_32_32_32_32_FLOAT,  32, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_RT_MTX1]      = m_ShaderRes.GetAttribute( "sysEmtRTMat1Attr",      9, FORMAT_32_32_32_32_FLOAT,  36, true );
    m_PtclAttr[EFT_PTCL_ATTR_IDX_EMTR_RT_MTX2]      = m_ShaderRes.GetAttribute( "sysEmtRTMat2Attr",      9, FORMAT_32_32_32_32_FLOAT,  40, true );
#endif

    m_TexSmpLoc[EFT_TEXTURE_SLOT_0].Initialize( &m_ShaderRes, "sysTextureSampler0" );
    m_TexSmpLoc[EFT_TEXTURE_SLOT_1].Initialize( &m_ShaderRes, "sysTextureSampler1" );
    m_TexSmpLoc[EFT_TEXTURE_SLOT_2].Initialize( &m_ShaderRes, "sysTextureSampler2" );
    m_FrameBuferTexSmpLoc.Initialize( &m_ShaderRes, "sysFrameBufferTexture" );
    m_DepthBuferTexSmpLoc.Initialize( &m_ShaderRes, "sysDepthBufferTexture" );
    m_CurlNoiseTexSmpLoc.Initialize( &m_ShaderRes, "sysCurlNoiseTextureArray" );

    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_0].Initialize(              &m_ShaderRes, "sysCustomShaderTextureSampler0" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_1].Initialize(              &m_ShaderRes, "sysCustomShaderTextureSampler1" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_2].Initialize(              &m_ShaderRes, "sysCustomShaderTextureSampler2" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_3].Initialize(              &m_ShaderRes, "sysCustomShaderTextureSampler3" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_0].Initialize(        &m_ShaderRes, "sysCustomShaderTextureArraySampler0" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_1].Initialize(        &m_ShaderRes, "sysCustomShaderTextureArraySampler1" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_2].Initialize(        &m_ShaderRes, "sysCustomShaderTextureArraySampler2" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_2D_ARRAY_3].Initialize(        &m_ShaderRes, "sysCustomShaderTextureArraySampler3" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_3D_0].Initialize(              &m_ShaderRes, "sysCustomShaderTexture3DSampler0" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_3D_1].Initialize(              &m_ShaderRes, "sysCustomShaderTexture3DSampler1" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_3D_2].Initialize(              &m_ShaderRes, "sysCustomShaderTexture3DSampler2" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_3D_3].Initialize(              &m_ShaderRes, "sysCustomShaderTexture3DSampler3" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_0].Initialize(        &m_ShaderRes, "sysCustomShaderCubeSampler0" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_1].Initialize(        &m_ShaderRes, "sysCustomShaderCubeSampler1" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_2].Initialize(        &m_ShaderRes, "sysCustomShaderCubeSampler2" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_3].Initialize(        &m_ShaderRes, "sysCustomShaderCubeSampler3" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_0].Initialize(  &m_ShaderRes, "sysCustomShaderCubeArraySampler0" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_1].Initialize(  &m_ShaderRes, "sysCustomShaderCubeArraySampler1" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_2].Initialize(  &m_ShaderRes, "sysCustomShaderCubeArraySampler2" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_CUBE_MAP_ARRAY_3].Initialize(  &m_ShaderRes, "sysCustomShaderCubeArraySampler3" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_0].Initialize(          &m_ShaderRes, "sysCustomShaderShadowSampler0" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_1].Initialize(          &m_ShaderRes, "sysCustomShaderShadowSampler1" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_2].Initialize(          &m_ShaderRes, "sysCustomShaderShadowSampler2" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_3].Initialize(          &m_ShaderRes, "sysCustomShaderShadowSampler3" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_0].Initialize(    &m_ShaderRes, "sysCustomShaderShadowArraySampler0" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_1].Initialize(    &m_ShaderRes, "sysCustomShaderShadowArraySampler1" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_2].Initialize(    &m_ShaderRes, "sysCustomShaderShadowArraySampler2" );
    m_CtmTextureLoc[EFT_CUSTOM_SHADER_TEXTURE_SHADOW_ARRAY_3].Initialize(    &m_ShaderRes, "sysCustomShaderShadowArraySampler3" );
}

//---------------------------------------------------------------------------
//  シェーダの終了処理を行います。
//---------------------------------------------------------------------------
void Shader::Finalize( Heap* heap )
{
    m_ShaderRes.Finalize( heap );
}




//---------------------------------------------------------------------------
//  ユニフォームブロック関連コード
//---------------------------------------------------------------------------
#ifdef EFT_USE_UNIFORM_BLOCK

#ifdef EFT_USE_UNIFORM_BLOCK
UniformBlockID   Shader::g_UniformBlock[Shader::EFT_UBO_TYPE_MAX];
UniformBlockID   Shader::g_CtmShaderUboID[EFT_CUSTOM_SHADER_UBO_MAX];
#endif

//---------------------------------------------------------------------------
//  ユニフォームブロックとシェーダの関連付けをします。
//---------------------------------------------------------------------------
#if EFT_OPENGL
bool _bindingUniformBlock( GLuint program, const char* name, u32 bindPoint )
{
    GLint  bufferSize;
    GLuint bufferIndex;
    bufferIndex = glGetUniformBlockIndex( program, name );
    if ( bufferIndex == GL_INVALID_INDEX ) return false;
    glGetActiveUniformBlockiv( program, bufferIndex, GL_UNIFORM_BLOCK_DATA_SIZE, &bufferSize );
    glUniformBlockBinding( program, bufferIndex, bindPoint );
    EFT_GLERR_CHECK();
    return true;
}
#endif
#if EFT_GX2
bool _bindingUniformBlockV( const GX2VertexShader* vertexShader, const char* name, u32 bindPoint )
{
    GX2UniformBlock* block = GX2GetVertexUniformBlock( vertexShader, name );
    if ( block )
    {
        EFT_ASSERT( block->location == bindPoint );
        return true;
    }
    else
    {
        return false;
    }
}
bool _bindingUniformBlockF( const GX2PixelShader* pixelShader, const char* name, u32 bindPoint )
{
    GX2UniformBlock* block = GX2GetPixelUniformBlock( pixelShader, name );
    if ( block )
    {
        EFT_ASSERT( block->location == bindPoint );
        return true;
    }
    else
    {
        return false;
    }
}
#endif

void Shader::BindingUniformBlock()
{
    m_UboEnableFlag = 0;

#if EFT_OPENGL
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysViewUniformBlock", EFT_UBO_BINDING_POINT_VIEW ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_VIEW;
        m_UboEnableFlag |= EFT_UBO_FLAG_VIEW_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysEmitterStaticUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_STATIC ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_STATIC;
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_STATIC_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysEmitterDynamicUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_DYNAMIC ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_DYNAMIC;
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_DYNAMIC_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysEmitterFieldUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_FIELD ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_FIELD;
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_FIELD_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysEmitterPluginUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_PLUGIN ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_PLUGIN;
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_PLUGIN_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysCustomShaderReservedUniformBlockParam", EFT_UBO_BINDING_POINT_RESERVED_PARAM ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_RESERVED_PARAM;
        m_UboEnableFlag |= EFT_UBO_FLAG_RESERVED_PARAM_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysCustomShaderUniformBlock0", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_0 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0;
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysCustomShaderUniformBlock1", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_1 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1;
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysCustomShaderUniformBlock2", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_2 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2;
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2_F;
    }
    if ( _bindingUniformBlock( m_ShaderRes.GetProgramID(), "sysCustmnShaderUniformBlock3", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_3 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3;
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3_F;
    }
#endif
#if EFT_GX2
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysViewUniformBlock", EFT_UBO_BINDING_POINT_VIEW ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_VIEW;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysEmitterStaticUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_STATIC ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_STATIC;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysEmitterDynamicUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_DYNAMIC ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_DYNAMIC;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysEmitterFieldUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_FIELD ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_FIELD;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysEmitterPluginUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_PLUGIN ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_PLUGIN;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysCustomShaderReservedUniformBlockParam", EFT_UBO_BINDING_POINT_RESERVED_PARAM ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_RESERVED_PARAM;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysCustomShaderUniformBlock0", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_0 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysCustomShaderUniformBlock1", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_1 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysCustomShaderUniformBlock2", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_2 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2;
    }
    if ( _bindingUniformBlockV( m_ShaderRes.GetVertexShader(), "sysCustomShaderUniformBlock3", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_3 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3;
    }

    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysViewUniformBlock", EFT_UBO_BINDING_POINT_VIEW ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_VIEW_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysEmitterStaticUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_STATIC ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_STATIC_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysEmitterDynamicUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_DYNAMIC ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_DYNAMIC_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysEmitterFieldUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_FIELD ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_FIELD_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysEmitterPluginUniformBlock", EFT_UBO_BINDING_POINT_EMITTER_PLUGIN ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_EMITTER_PLUGIN_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysCustomShaderReservedUniformBlockParam", EFT_UBO_BINDING_POINT_RESERVED_PARAM ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_RESERVED_PARAM_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysCustomShaderUniformBlock0", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_0 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysCustomShaderUniformBlock1", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_1 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysCustomShaderUniformBlock2", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_2 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2_F;
    }
    if ( _bindingUniformBlockF( m_ShaderRes.GetFragmentShader(), "sysCustomShaderUniformBlock3", EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_3 ) )
    {
        m_UboEnableFlag |= EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3_F;
    }
#endif
}

//---------------------------------------------------------------------------
//  ビューユニフォームブロックを初期化します。
//---------------------------------------------------------------------------
void Shader::InitializeUniformBlock()
{
    for ( u32 i = 0; i < EFT_UBO_TYPE_MAX; i++ )         { g_UniformBlock[i].Invalidate(); }
    for ( u32 i = 0; i < EFT_CUSTOM_SHADER_UBO_MAX; i++ ) { g_CtmShaderUboID[i].Invalidate(); }

    g_UniformBlock[EFT_UBO_TYPE_VIEW].Initialize(            EFT_UBO_BINDING_POINT_VIEW,            "sysViewUniformBlock" );
    g_UniformBlock[EFT_UBO_TYPE_EMITTER_STATIC].Initialize(  EFT_UBO_BINDING_POINT_EMITTER_STATIC,  "sysEmitterStaticUniformBlock" );
    g_UniformBlock[EFT_UBO_TYPE_EMITTER_DYNAMIC].Initialize( EFT_UBO_BINDING_POINT_EMITTER_DYNAMIC, "sysEmitterDynamicUniformBlock" );
    g_UniformBlock[EFT_UBO_TYPE_EMITTER_FIELD].Initialize(   EFT_UBO_BINDING_POINT_EMITTER_FIELD,   "sysEmitterFieldUniformBlock" );
    g_UniformBlock[EFT_UBO_TYPE_EMITTER_PLUGIN].Initialize(  EFT_UBO_BINDING_POINT_EMITTER_PLUGIN,  "sysEmitterPluginUniformBlock" );
    g_UniformBlock[EFT_UBO_TYPE_RESERVED_PARAM].Initialize(  EFT_UBO_BINDING_POINT_RESERVED_PARAM,  "sysCustomShaderReservedUniformBlockParam" );
    g_CtmShaderUboID[EFT_CUSTOM_SHADER_UBO_0].Initialize(    EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_0, "sysCustomShaderUniformBlock0" );
    g_CtmShaderUboID[EFT_CUSTOM_SHADER_UBO_1].Initialize(    EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_1, "sysCustomShaderUniformBlock1" );
    g_CtmShaderUboID[EFT_CUSTOM_SHADER_UBO_2].Initialize(    EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_2, "sysCustomShaderUniformBlock2" );
    g_CtmShaderUboID[EFT_CUSTOM_SHADER_UBO_3].Initialize(    EFT_UBO_BINDING_POINT_CUSTOM_SHADER_UBO_3, "sysCustomShaderUniformBlock3" );
}

//---------------------------------------------------------------------------
//  ビューユニフォームブロックを終了処理をします。
//---------------------------------------------------------------------------
void Shader::FinalizeUniformBlock()
{
    for ( u32 i = 0; i < EFT_UBO_TYPE_MAX; i++ )         { g_UniformBlock[i].Finalize(); }
    for ( u32 i = 0; i < EFT_CUSTOM_SHADER_UBO_MAX; i++ ) { g_CtmShaderUboID[i].Finalize(); }
}

//---------------------------------------------------------------------------
//! @brief  ビューユニフォームブロックをバインドします。
//---------------------------------------------------------------------------
void Shader::BindViewUniformBlock( const UniformBlockBuffer* viewUniformBlockBuffer )
{
    g_UniformBlock[EFT_UBO_TYPE_VIEW].BindUniformBlock( viewUniformBlockBuffer, true, true );
}

//---------------------------------------------------------------------------
//  ユニフォームブロックをバインドします。
//---------------------------------------------------------------------------
void Shader::BindUniformBlock( UniformBlockType type, const UniformBlockBuffer* uniformBuffer )
{
    NW_NULL_ASSERT( uniformBuffer );
    NW_ASSERT( uniformBuffer->IsValidate() );
    g_UniformBlock[type].BindUniformBlock( uniformBuffer, true, true );
}

//---------------------------------------------------------------------------
//  エミッタプラグインユニフォームブロックをバインドします。
//---------------------------------------------------------------------------
void Shader::BindEmitterPluginUniformBlock( const UniformBlockBuffer* uniformBuffer )
{
    NW_NULL_ASSERT( uniformBuffer );
    NW_ASSERT( uniformBuffer->IsValidate() );
    bool vertex   = m_UboEnableFlag & EFT_UBO_FLAG_EMITTER_PLUGIN ? true : false;
    bool fragment = m_UboEnableFlag & EFT_UBO_FLAG_EMITTER_PLUGIN_F ? true : false;
    g_UniformBlock[EFT_UBO_TYPE_EMITTER_PLUGIN].BindUniformBlock( uniformBuffer, vertex, fragment );
}

//---------------------------------------------------------------------------
//  バイナリ化されているエミッタプラグインデータをユニフォームブロックにバインドします。
//---------------------------------------------------------------------------
bool Shader::BindEmitterPluginUniformBlockAuto( Emitter* emitter )
{
    if ( !emitter->emitterRes->emitterPluginData ) return false;
    System* system  = emitter->emitterSet->GetSystem();

    TemporaryUniformBlockBuffer tempUbo;
    void* ubo = static_cast<CustomShaderReservedUniformBlock *>( tempUbo.Alloc( system, sizeof(nw::math::VEC4) * 8 ) );
    if ( !ubo ) return false;
    MemUtil::Copy( ubo, emitter->emitterRes->emitterPluginData, sizeof(nw::math::VEC4) * 8 );
    tempUbo.Validate();

    bool vertex   = m_UboEnableFlag & EFT_UBO_FLAG_EMITTER_PLUGIN ? true : false;
    bool fragment = m_UboEnableFlag & EFT_UBO_FLAG_EMITTER_PLUGIN_F ? true : false;
    g_UniformBlock[EFT_UBO_TYPE_EMITTER_PLUGIN].BindUniformBlock( &tempUbo, vertex, fragment );
    return true;
}

//---------------------------------------------------------------------------
//  ユーザーパラメータユニフォームブロックをバインドします。
//---------------------------------------------------------------------------
bool Shader::BindReservedCustomShaderUniformBlock( Emitter* emitter )
{
    if ( !emitter->emitterRes->customShaderParam ) return false;
    System* system  = emitter->emitterSet->GetSystem();

    TemporaryUniformBlockBuffer tempUbo;
    void* ubo = static_cast<CustomShaderReservedUniformBlock *>( tempUbo.Alloc( system, sizeof(CustomShaderReservedUniformBlock) ) );
    if ( !ubo ) return false;
    MemUtil::Copy( ubo, emitter->emitterRes->customShaderParam, sizeof(CustomShaderReservedUniformBlock) );
    tempUbo.Validate();

    bool vertex   = m_UboEnableFlag & EFT_UBO_FLAG_RESERVED_PARAM ? true : false;
    bool fragment = m_UboEnableFlag & EFT_UBO_FLAG_RESERVED_PARAM_F ? true : false;
    g_UniformBlock[EFT_UBO_TYPE_RESERVED_PARAM].BindUniformBlock( &tempUbo, vertex, fragment );
    return true;
}

//---------------------------------------------------------------------------
//  各エミッタ共通ユニフォームブロックをバインドします。
//---------------------------------------------------------------------------
void Shader::BindCommonCustomShaderUniformBlock( CustomShaderUboID id, void* uniformBlock, u32 uniformBlockSize, bool endianSwap )
{
    TemporaryUniformBlockBuffer tempUbo;
    tempUbo.Set( uniformBlock, uniformBlockSize );
    tempUbo.Validate( endianSwap );
    g_CtmShaderUboID[id].BindUniformBlock( &tempUbo, true, true );
}

//---------------------------------------------------------------------------
//  カスタムシェーダユニフォームブロックをバインドします。
//---------------------------------------------------------------------------
void Shader::BindCustomShaderUniformBlock( CustomShaderUboID id, void* uniformBlock, u32 uniformBlockSize, bool endianSwap )
{
    TemporaryUniformBlockBuffer tempUbo;
    tempUbo.Set( uniformBlock, uniformBlockSize );
    tempUbo.Validate( endianSwap );

    bool vertex   = false;
    bool fragment = false;

    if ( id == EFT_CUSTOM_SHADER_UBO_0 )
    {
        vertex   = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0 ? true : false;
        fragment = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_0_F ? true : false;
    }
    if ( id == EFT_CUSTOM_SHADER_UBO_1 )
    {
        vertex   = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1 ? true : false;
        fragment = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_1_F ? true : false;
    }
    if ( id == EFT_CUSTOM_SHADER_UBO_2 )
    {
        vertex   = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2 ? true : false;
        fragment = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_2_F ? true : false;
    }
    if ( id == EFT_CUSTOM_SHADER_UBO_3 )
    {
        vertex   = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3 ? true : false;
        fragment = m_UboEnableFlag & EFT_UBO_FLAG_CUSTOM_SHADER_UBO_3_F ? true : false;
    }
    g_CtmShaderUboID[id].BindUniformBlock( &tempUbo, vertex, fragment );
}
#endif      // EFT_USE_UNIFORM_BLOCK



//---------------------------------------------------------------------------
//  インスタンシングを有効にします。
//---------------------------------------------------------------------------
void Shader::EnableInstanced()
{
#ifdef EFT_OGL
#ifdef EFT_USE_INSTANS_DRAW
    for( u32 i = 0; i < EFT_PTCL_ATTR_IDX_AMX; i++ )
    {
        if ( m_PtclAttr[i]  != EFT_INVALID_SHADER_ATTRIBUTE ) { glVertexAttribDivisor( m_PtclAttr[i], 1 ); }
    }
#ifndef EFT_DEGRADATION_SPEC
    if ( m_StreamAttr[EFT_STREAM_ATTR_IDX_POSITION]  != EFT_INVALID_SHADER_ATTRIBUTE )
    { glVertexAttribDivisor( m_StreamAttr[EFT_STREAM_ATTR_IDX_POSITION], 1 ); }
    if ( m_StreamAttr[EFT_STREAM_ATTR_IDX_VECTOR]  != EFT_INVALID_SHADER_ATTRIBUTE )
    { glVertexAttribDivisor( m_StreamAttr[EFT_STREAM_ATTR_IDX_VECTOR], 1 ); }
#endif
#endif
#endif
}

//---------------------------------------------------------------------------
//  インスタンシングを無効にします。
//---------------------------------------------------------------------------
void Shader::DisableInstanced()
{
#ifdef EFT_OGL
#ifdef EFT_USE_INSTANS_DRAW
    for( u32 i = 0; i < EFT_PTCL_ATTR_IDX_AMX; i++ )
    {
        if ( m_PtclAttr[i]  != EFT_INVALID_SHADER_ATTRIBUTE ) { glVertexAttribDivisor( m_PtclAttr[i], 0 ); }
    }
#ifndef EFT_DEGRADATION_SPEC
    if ( m_StreamAttr[EFT_STREAM_ATTR_IDX_POSITION]  != EFT_INVALID_SHADER_ATTRIBUTE )
    { glVertexAttribDivisor( m_StreamAttr[EFT_STREAM_ATTR_IDX_POSITION], 0 ); }
    if ( m_StreamAttr[EFT_STREAM_ATTR_IDX_VECTOR]  != EFT_INVALID_SHADER_ATTRIBUTE )
    { glVertexAttribDivisor( m_StreamAttr[EFT_STREAM_ATTR_IDX_VECTOR], 0 ); }
#endif
#endif
#endif
}


} // namespace eft2
} // namespace nw

