﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nw/eft/eft2_ShaderManager.h>
#include <nw/eft/eft2_Data.h>
#include <nw/eft/eft2_MemUtil.h>

namespace nw   {
namespace eft2 {


//---------------------------------------------------------------------------
//  エミッタ時間 キーフレームアニメを計算します。
//---------------------------------------------------------------------------
void CalcEmitterKeyFrameAnim( nw::math::VEC3_* animV, bool* isDstEnd, nw::eft2::ResAnimEmitterKeyParamSet* animParam, const f32 time )
{
    // エミッタ時間アニメ用キーフレーム処理。
    // 寿命ではなくフレーム数依存であることに注意！
    s32 term = 0;
    f32 dis = 0;
    const s32 lastKeyIndex = animParam->keyNum - 1;
    const s32 keyNum = animParam->keyNum;

    const f32 maxKeyFrameNum = animParam->keyValue[lastKeyIndex].w;

    // キー無し（デフォルト値を用いるので処理を返す）
    if ( animParam->keyNum == 0 )
    {
        return;
    }

    // キーが一つのみ（固定値なので値をそのまま返す）
    if ( animParam->keyNum == 1 )
    {
        EFT_F32_VEC3_COPY( animV, &animParam->keyValue[ 0 ] );
        //animV->x = animParam->keyValue[ 0 ].x;
        //animV->y = animParam->keyValue[ 0 ].y;
        //animV->z = animParam->keyValue[ 0 ].z;
        return;
    }

    // ここからアニメ計算

    //[ 0, 最終フレーム数 ]の間の時間（ループ加味）
    f32 localTime = time;
    if( animParam->loop ){ localTime = fmod( localTime, maxKeyFrameNum ); }

    // まだ最初のキーの時間に到達していない（最初のキーのカラーでClampする）
    if ( localTime < animParam->keyValue[0].w )
    {
        EFT_F32_VEC3_COPY( animV, &animParam->keyValue[ 0 ] );
        //animV->x = animParam->keyValue[ 0 ].x;
        //animV->y = animParam->keyValue[ 0 ].y;
        //animV->z = animParam->keyValue[ 0 ].z;
        return;
    }
    // 既に最終キーの時間を過ぎている（最終キーのカラーでClampする）
    if ( animParam->keyValue[ lastKeyIndex ].w <= localTime )
    {
        EFT_F32_VEC3_COPY( animV, &animParam->keyValue[ lastKeyIndex ] );
        //animV->x = animParam->keyValue[ lastKeyIndex ].x;
        //animV->y = animParam->keyValue[ lastKeyIndex ].y;
        //animV->z = animParam->keyValue[ lastKeyIndex ].z;
        *isDstEnd = true;
        return;
    }

    // TODO：平たく書いた方がよいかも。
    nw::math::VEC3 start, end;
    for ( term = 0; term < keyNum; term++ )
    {
        s32 index1 = term;
        s32 index2 = term + 1;
        f32 t1 = animParam->keyValue[index1].w;
        f32 t2 = animParam->keyValue[index2].w;

        if ( ( t1 <= localTime ) && ( localTime < t2 ) )
        {
            EFT_F32_VEC3_COPY( &start, &animParam->keyValue[index1] );
            EFT_F32_VEC3_COPY( &end, &animParam->keyValue[index2] );
            //start.Set( animParam->keyValue[index1].x, animParam->keyValue[index1].y, animParam->keyValue[index1].z );
            //end.Set  ( animParam->keyValue[index2].x, animParam->keyValue[index2].y, animParam->keyValue[index2].z );
            dis   = t2 - t1;
            nw::math::VEC3 ret = ( start + ( end - start ) / dis * ( localTime - t1 ) );

            EFT_F32_VEC3_COPY( animV, &ret );
            //animV->x = ret.x;
            //animV->y = ret.y;
            //animV->z = ret.z;
            return;
        }
    }

    // ここには来ないはず
    return;
}


#if 0
//---------------------------------------------------------------------------
//  エミッタ時間 キーフレームアニメを計算します。
//---------------------------------------------------------------------------
void CalcEmitterKeyFrameAnim( nw::math::VEC3* retValue, const ResKeyFrameAnim* keyFrameAnim, f32 time )
{
    // keyFrameAnim の直後にアニメ配列( ResAnimKey )がある
    const ResAnimKey* animKeyArray = reinterpret_cast<ResAnimKey*>(
                        (u32)keyFrameAnim + sizeof(ResKeyFrameAnim) );

    // キーが１つの場合
    if( keyFrameAnim->keyNum == 1 )
    {
        retValue->x = animKeyArray->value.x;
        retValue->y = animKeyArray->value.y;
        retValue->z = animKeyArray->value.z;
        return;
    }

    const ResAnimKey* startKey = &animKeyArray[0];
    const ResAnimKey* endKey   = &animKeyArray[keyFrameAnim->keyNum - 1];

    if( keyFrameAnim->isLoop )
    {
        // ループの場合は正規化する
        time = nw::math::FMod( time, endKey->time );

        // 左端
        if( time <= startKey->time )
        {
            retValue->x = startKey->value.x;
            retValue->y = startKey->value.y;
            retValue->z = startKey->value.z;
            return;
        }
    }
    else
    {
        // 左端
        if( time <= startKey->time )
        {
            retValue->x = startKey->value.x;
            retValue->y = startKey->value.y;
            retValue->z = startKey->value.z;
            return;
        }

        // 右端
        if( endKey->time <= time )
        {
            retValue->x = endKey->value.x;
            retValue->y = endKey->value.y;
            retValue->z = endKey->value.z;
            return;
        }
    }

    s32 index = -1;

    // 現在のキーを検索する
    for( u32 i = 0; i < keyFrameAnim->keyNum; ++i )
    {
        if( time < animKeyArray[i].time )
        {
            break;
        }
        index++;
    }

    // 区間の左と右のキーを設定
    const ResAnimKey* k0 = &animKeyArray[index];
    const ResAnimKey* k1 = &animKeyArray[index];

    if( static_cast<u32>(index + 1) != keyFrameAnim->keyNum )
    {
        k1 = &animKeyArray[index + 1];
    }

    f32 w = k1->time - k0->time;

    // 比率
    f32 t;
    if( w == 0.0f)
    {
        // 区間の長さが０
        retValue->x = k0->value.x;
        retValue->y = k0->value.y;
        retValue->z = k0->value.z;
        return;
    }
    else
    {
        t = ( time - k0->time ) / w;
    }

    // スムース補間
    if( keyFrameAnim->interpolation == EFT_KEY_FRAME_ANIM_SMOOTH )
    {
        t = t * t * ( 3.0f - 2.0f * t );
    }

    retValue->x = k0->value.x * (1.0f - t) + k1->value.x * t;
    retValue->y = k0->value.y * (1.0f - t) + k1->value.y * t;
    retValue->z = k0->value.z * (1.0f - t) + k1->value.z * t;
    return;
}
#endif



} // namespace eft2
} // namespace nw

