﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


//------------------------------------------------------------------------------
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_System.h>
#include <nw/eft/eft2_KeyFrameAnim.h>
#include <nw/eft/eft2_Emitter.h>
#include <nw/eft/eft2_AreaLoop.h>
#include <nw/eft/eft2_Stripe.h>
#include <nw/eft/eft2_StripeConnection.h>
#include <nw/eft/eft2_SuperStripe.h>

namespace nw   {
namespace eft2 {

//---------------------------------------------------------------------------
//  コンストラクタ。
//---------------------------------------------------------------------------
EmitterCalc::EmitterCalc( System* system )
{
    m_System = system;

    m_PositionBuffer.Initialize();
    const f32 width = 0.5f;
    nw::math::VEC4* pos = static_cast<nw::math::VEC4*>( AllocFromStaticHeap( sizeof(nw::math::VEC4) * EFT_PARTICLE_PRIM_VERTICES_NUM ) );
    {
        // Triangle Strip と Quads で頂点定義順が異なるので、モードによって定義順を切り替え。
#if EFT_PARTICLE_PRIM_TRIANGLES_MODE
        // Triangle Strip
        pos[0].Set( -width,  width, 0.0f, 0.0f );
        pos[1].Set( -width, -width, 0.0f, 1.0f );
        pos[2].Set(  width,  width, 0.0f, 2.0f );
        pos[3].Set(  width, -width, 0.0f, 3.0f );
#else
        // Quads
        pos[0].Set( -width,  width, 0.0f, 0.0f );
        pos[1].Set( -width, -width, 0.0f, 1.0f );
        pos[2].Set(  width, -width, 0.0f, 2.0f );
        pos[3].Set(  width,  width, 0.0f, 3.0f );
#endif
    }
    m_PositionBuffer.SetVertexBuffer( pos, sizeof(nw::math::VEC4) * EFT_PARTICLE_PRIM_VERTICES_NUM, EFT_PARTICLE_PRIM_VERTICES_NUM );
    m_PositionBuffer.Validate();

    m_IndexBuffer.Initialize();
    u32* index = static_cast<u32*>( AllocFromStaticHeap( sizeof(u32) * EFT_PARTICLE_PRIM_INDICES_NUM ) );
    {
        index[0] = 0;
        index[1] = 1;
        index[2] = 2;
        index[3] = 3;
    }
    m_IndexBuffer.SetVertexBuffer( index, sizeof(u32) * EFT_PARTICLE_PRIM_INDICES_NUM, EFT_PARTICLE_PRIM_PRIMITIVE_NUM );
    m_IndexBuffer.Validate();
#if EFT_IS_CAFE
    DCFlushRange( index, sizeof(u32) * EFT_PARTICLE_PRIM_INDICES_NUM );
#endif

    m_ResTextureSamplerForCurlNoise.filter      = EFT_TEXTURE_FILTER_TYPE_LINEAR;
    m_ResTextureSamplerForCurlNoise.wrapU       = EFT_TEXTURE_WRAP_TYPE_REPEAT;
    m_ResTextureSamplerForCurlNoise.wrapV       = EFT_TEXTURE_WRAP_TYPE_REPEAT;
    m_ResTextureSamplerForCurlNoise.isSphereMap = false;
    m_ResTextureSamplerForCurlNoise.mipLevel    = 15.99f;
    m_ResTextureSamplerForCurlNoise.mipMapBias  = 0.0f;
    m_TextureSamplerForCurlNoise.Invalidate();
    m_TextureSamplerForCurlNoise.Initialize( &m_ResTextureSamplerForCurlNoise );

    m_ResTextureSamplerForColorAndDepth.filter = EFT_TEXTURE_FILTER_TYPE_LINEAR;
    m_ResTextureSamplerForColorAndDepth.wrapU = EFT_TEXTURE_WRAP_TYPE_CLAMP;
    m_ResTextureSamplerForColorAndDepth.wrapV = EFT_TEXTURE_WRAP_TYPE_CLAMP;
    m_ResTextureSamplerForColorAndDepth.isSphereMap = false;
    m_ResTextureSamplerForColorAndDepth.mipLevel = 15.99f;
    m_ResTextureSamplerForColorAndDepth.mipMapBias = 0.0f;
    m_TextureSamplerForColorAndDepth.Invalidate();
    m_TextureSamplerForColorAndDepth.Initialize( &m_ResTextureSamplerForColorAndDepth );
}

//---------------------------------------------------------------------------
//! @brief        デストラクタ。
//---------------------------------------------------------------------------
EmitterCalc::~EmitterCalc()
{
    FreeFromStaticHeap( m_PositionBuffer.GetVertexBuffer() );
    FreeFromStaticHeap( m_IndexBuffer.GetVertexBuffer() );
}

//---------------------------------------------------------------------------
//  エミッタ計算処理をします。
//---------------------------------------------------------------------------
bool EmitterCalc::Calc( Emitter* emitter, f32 frameRate, bool swapBuffer, bool isFade, bool isEmission )
{
    ResEmitter* res                 = emitter->emitterData;
    EmitterSet* emitterSet          = emitter->emitterSet;
    bool        isEmit              = true;
    f32         parentLife          =  0.0f;

    // 削除予約されている場合は、何も処理せず削除フローに移る
    if ( emitter->m_IsKillReservation )
    {
        return false;
    }

    isEmit = isEmission;

    // バッファをスワップ
    if ( swapBuffer )
    {
        emitter->Swap( EFT_BUFFER_MODE_DOUBLE );
    }

    // エミッタ計算処理前コールバック
#ifndef EFT_DEGRADATION_SPEC
    {
        EmitterPreCalcArg arg;
        arg.emitter  = emitter;
        InvokeEmitterPreCalcCallback( arg );
    }
#endif

    // 放出開始フレーム
    f32 emitStart = static_cast<f32>( res->emission.start );

    // エミッタ時間
    f32 emitterTime = emitter->time;

    // 放出終了フレーム
    f32 emitEnd = static_cast<f32>( res->emission.start + res->emission.emitDuration );

    // フレームレート
    emitter->frameRate = frameRate;

    // チャイルドエミッタ時の補足処理
    if ( emitter->isChild )
    {
        EFT_NULL_ASSERT( emitter->parentEmitter->emitterCalc );
        EFT_ASSERT( emitter->parentEmitter->emitterCreateID == emitter->parentEmitterCreateID );

        // 放出開始フレーム調整
        parentLife  = emitter->parentParticleLife;
        emitStart   = static_cast<f32>( parentLife * ( res->emission.emitTiming / 100.f ) );

        // 放出終了フレームは、ワンタイムの場合即時もしくは親パーティクルの死亡時間
        emitEnd = ( res->emission.isOneTime ) ? ( emitStart + res->emission.emitDuration ) : static_cast< float >( emitter->parentParticleLife );
    }

    //-------------------------
    // エミッタ処理
    //-------------------------
    // エミッタ時間アニメの値を反映
    emitter->calcedEmitterAnimCount = 0;
    if ( emitter->emitterRes->isUseEmitterAnim )
    {
        for ( u32 i = 0; i < EFT_EMITTER_ANIM_MAX; i++ )
        {
            if ( emitter->emitterRes->emitterAnimArray[i] &&
                emitter->emitterRes->emitterAnimArray[i]->enable &&
                ( !emitter->m_EmitterAnimEnd[i] || emitter->emitterRes->emitterAnimArray[i]->loop )  )
            {
                // 現時間のエミッタ時間アニメの値を計算。
                CalcEmitterKeyFrameAnim( &emitter->m_EmitterAnimV[i],
                    &emitter->m_EmitterAnimEnd[i],
                    emitter->emitterRes->emitterAnimArray[i],
                    emitter->time );
                emitter->calcedEmitterAnimCount++;
            }
        }

        // 更に乗算が必要なエミッタアニメについて個別処理（※エミッタアニメの更新が入った時のみ）
        {
            u32 i = 0;
            i = EFT_EMITTER_ANIM_ALL_DIR_VEL;
            if( emitter->emitterRes->emitterAnimArray[i] &&
                emitter->emitterRes->emitterAnimArray[i]->enable &&
                ( !emitter->m_EmitterAnimEnd[i] || emitter->emitterRes->emitterAnimArray[i]->loop )  )
            {
                // 全方向初速のスケールを乗算
                emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_ALL_DIR_VEL].x *= emitterSet->GetAllDirectionalVel();
            }
        }
        {
            u32 i = 0;
            i = EFT_EMITTER_ANIM_VOLUME_SCALE;
            if( emitter->emitterRes->emitterAnimArray[i] &&
                emitter->emitterRes->emitterAnimArray[i]->enable &&
                ( !emitter->m_EmitterAnimEnd[i] || emitter->emitterRes->emitterAnimArray[i]->loop )  )
            {
                // エミッタ形状のスケールを乗算
                const nw::math::VEC3& runtimeVolumeScale = emitterSet->GetEmitterVolumeScale();
                emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_VOLUME_SCALE].x *= runtimeVolumeScale.x;
                emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_VOLUME_SCALE].y *= runtimeVolumeScale.y;
                emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_VOLUME_SCALE].z *= runtimeVolumeScale.z;
            }
        }
    }
    else
    {
        // エミッタ時間アニメなし

        // 全方向初速のスケールを乗算
        emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_ALL_DIR_VEL].x = emitter->emitterRes->emitterData->ptclVel.allDirection *
                                                                  emitterSet->GetAllDirectionalVel();

        // エミッタ形状のスケールを乗算
        nw::math::VEC3*       resVolumeScale     = &emitter->emitterRes->emitterData->volume.volumeFormScale;
        const nw::math::VEC3& runtimeVolumeScale = emitterSet->GetEmitterVolumeScale();
        emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_VOLUME_SCALE].x = resVolumeScale->x * runtimeVolumeScale.x;
        emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_VOLUME_SCALE].y = resVolumeScale->y * runtimeVolumeScale.y;
        emitter->m_EmitterAnimV[EFT_EMITTER_ANIM_VOLUME_SCALE].z = resVolumeScale->z * runtimeVolumeScale.z;
    }

    // エミッタアニメの値を補正
    if ( emitter->calcedEmitterAnimCount > 0 )
    {
        // 放出個数が変化する系は、データに埋め込まれた値以上に増えないように補正
        if ( emitter->m_EmitterAnim.vparticleLife.x > static_cast<f32>( emitter->emitterRes->emitterData->ptcl.life ) )
        {
            emitter->m_EmitterAnim.vparticleLife.x = static_cast<f32>( emitter->emitterRes->emitterData->ptcl.life );
        }
    }

    if ( !emitter->isChild )
    {
        // エミッタマトリクス更新
        if ( emitter->emitterRes->isEmitterSRTAnim )
        {
            // エミッタ時間アニメがある場合は、アニメ * res * set の順で演算
            nw::math::MTX34 animMatSRT;
            nw::math::MTX34 animMatRT;
            nw::math::MTX34Identity(&animMatSRT);
            nw::math::MTX34Identity(&animMatRT);

            nw::math::VEC3 animScale( 1, 1, 1 );
            nw::math::VEC3 animRotate( 0, 0, 0 );
            nw::math::VEC3 animTrans( 0, 0, 0 );

            //if ( emitter->emitterRes->emitterAnimArray[ EFT_EMITTER_ANIM_SCALE ] )
            {
                EFT_F32_VEC3_COPY( &animScale, &emitter->m_EmitterAnimV[ EFT_EMITTER_ANIM_SCALE ] );
            }
            if ( emitter->emitterRes->emitterAnimArray[ EFT_EMITTER_ANIM_ROTATE ] )
            {
                //Deg -> Rad 変換
                animRotate.x = nw::math::DegToRad( emitter->m_EmitterAnimV[ EFT_EMITTER_ANIM_ROTATE ].x );
                animRotate.y = nw::math::DegToRad( emitter->m_EmitterAnimV[ EFT_EMITTER_ANIM_ROTATE ].y );
                animRotate.z = nw::math::DegToRad( emitter->m_EmitterAnimV[ EFT_EMITTER_ANIM_ROTATE ].z );
            }
            else
            {
                // 変換無しで受け取る
                EFT_F32_VEC3_COPY( &animRotate, &emitter->m_EmitterAnimV[ EFT_EMITTER_ANIM_ROTATE ] );
            }
            //if ( emitter->emitterRes->emitterAnimArray[ EFT_EMITTER_ANIM_TRANS ] )
            {
                EFT_F32_VEC3_COPY( &animTrans, &emitter->m_EmitterAnimV[ EFT_EMITTER_ANIM_TRANS ] );
            }

            nw::math::VEC3 tempScale( 1, 1, 1 );
            nw::math::MTX34MakeSRT( &animMatRT, &tempScale, &animRotate, &animTrans );
            nw::math::MTX34MakeSRT( &animMatSRT, &animScale, &animRotate, &animTrans );

            const nw::math::Matrix34& esetMatrixSRT = emitterSet->GetSRTMatrix();
            const nw::math::Matrix34& esetMatrixRT  = emitterSet->GetRTMatrix();

            emitter->matrixSRT.SetMult( esetMatrixSRT, animMatSRT );
            emitter->matrixRT.SetMult( esetMatrixRT, animMatRT );
        }
        else
        {
            // エミッタアニメが一切なく、プログラム側でSRTが修正された場合、SRT行列を更新
            if( emitterSet->m_IsEmitterSRTDirty )
            {
                // アニメ無しの場合、リソースの値を直接参照
                const nw::math::Matrix34& esetMatrixSRT = emitterSet->GetSRTMatrix();
                const nw::math::Matrix34& esetMatrixRT  = emitterSet->GetRTMatrix();
                emitter->matrixSRT.SetMult( esetMatrixSRT, emitter->resMatrixSRT );
                emitter->matrixRT.SetMult( esetMatrixRT, emitter->resMatrixRT );
            }
        }
    }
    else
    {
         // 親パーティクルの位置＋エミッタリソースのマトリクス
         MTX34Copy( &emitter->matrixSRT, &emitter->resMatrixSRT );
         MTX34Copy( &emitter->matrixRT,  &emitter->resMatrixRT );
         emitter->matrixSRT._03 += emitter->parentParticleWorldPos.x;
         emitter->matrixSRT._13 += emitter->parentParticleWorldPos.y;
         emitter->matrixSRT._23 += emitter->parentParticleWorldPos.z;
         emitter->matrixRT._03  += emitter->parentParticleWorldPos.x;
         emitter->matrixRT._13  += emitter->parentParticleWorldPos.y;
         emitter->matrixRT._23  += emitter->parentParticleWorldPos.z;
    }

    // エミッタマトリクスセットコールバック
#ifndef EFT_DEGRADATION_SPEC
    InvokeEmitterMatrixSetCallback( emitter );
#endif

    // エミッタ移動距離差分を計算
    if ( emitter->time != 0.0f )
    {
        nw::math::VEC3 emitterPos;
        nw::math::VEC3 emitterLocalPos;
        nw::math::VEC3 emitterPrevLocalPos;
        emitterPos.x = emitter->matrixSRT._03;
        emitterPos.y = emitter->matrixSRT._13;
        emitterPos.z = emitter->matrixSRT._23;

        nw::math::MTX34 invMat;
        MTX34Inverse( &invMat, &emitter->matrixSRT );
        VEC3Transform( &emitterLocalPos, &invMat, &emitterPos );
        VEC3Transform( &emitterPrevLocalPos, &invMat, &emitter->emitterPrevPos );
        emitter->emitterLocalVec.x = emitterLocalPos.x - emitterPrevLocalPos.x;
        emitter->emitterLocalVec.y = emitterLocalPos.y - emitterPrevLocalPos.y;
        emitter->emitterLocalVec.z = emitterLocalPos.z - emitterPrevLocalPos.z;
    }

    // フェードイン処理
    // MEMO: 「フェードイン中」という状態は持たない
    {
        // αフェードインもしくはスケールフェードインの指定がある場合のみ処理
        if( res->emitter.isAlphaFadeIn || res->emitter.isScaleFadeIn )
        {
            if( emitter->fadeInAlpha < 1.0f )
            {
                const int fadeTime = res->emitter.fadeInTime;
                if( fadeTime > 0 )
                {
                    // 初期値 0.0 なので、徐々に 1.0 まで加算していく
                    emitter->fadeInAlpha += frameRate / fadeTime;
                    if( emitter->fadeInAlpha > 1.0f )
                    {
                        emitter->fadeInAlpha = 1.0f;
                    }
                }
                else
                {
                    // フェード時間 0（データとしてはあまり好ましくない）
                    emitter->fadeInAlpha = 1.0f;
                }
            }
        }
    }

    // フェード指定
    if( isFade || emitter->m_IsSoloFade )
    {
        // 「放出停止」の指定がある場合放出を止める
        if( res->emitter.isFadeEmit )
        {
            isEmit = false;
        }

        // フェードアウト用のアルファ値を計算する
        if ( res->emitter.isFadeAlphaFade || res->emitter.isScaleFade )
        {
            u32 fadeTime = res->emitter.alphaFadeTime;
            if ( fadeTime <= 0.0f )
            {
                emitter->fadeOutAlpha = 0;
                return false;
            }

            emitter->fadeOutAlpha -= frameRate / fadeTime;
            if ( emitter->fadeOutAlpha < 0.0f )
            {
                emitter->fadeOutAlpha = 0;
                return false;
            }
        }
    }

    // 手動放出分
    if ( emitter->manualEmissionNum > 0 )
    {
        emitter->emitLastFrame = emitter->time;

        if ( emitter->manualEmitPoints )
        {
            Emit( emitter, emitter->manualEmissionNum );
        }
    }
    emitter->manualEmitPoints  = NULL;
    emitter->manualEmissionNum = 0;

    // 放出処理
    if( isEmit && emitter->isEnableEmit && emitStart <= emitterTime )
    {
        // ワンタイム or チャイルドの場合は放出区間がある
        if ( ( res->emission.isOneTime || emitter->isChild ) && emitEnd <= emitterTime && emitter->isEmit )
        {
            goto exit_emit;
        }

        //-------------------------
        // 時間放出
        //-------------------------
        if( !emitter->emitterData->emission.isEmitDistEnabled )
        {
            // 初回時は、emitter->interval=0.0 で必ず放出される。
            if ( emitter->emitCnt >= emitter->emitInterval )
            {
                const f32 emitCntSub = emitter->emitCnt - emitter->emitInterval;

                //放出レートランダム
                const f32 emitRateRandom = emitter->emitterData->emission.rateRandom / 100.0f * res->emission.rate;
                const f32 emitReduction  = emitRateRandom * emitter->random.GetF32();

                f32 emitRatio = res->emission.rate;
                // 球等分割は放出レートが固定値なので、アニメの加味を行わない。
                if( emitter->emitterData->volume.volumeType != EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE &&
                    emitter->emitterData->volume.volumeType != EFT_EMITTER_VOLUME_TYPE_SPHERE_SAME_DIVIDE64 )
                {
                    emitRatio = emitter->m_EmitterAnim.emissionRate;
                }

                emitter->emitSaving += ( emitRatio - emitReduction ) * emitter->emitRatio * emitterSet->GetEmissionRatioScale();
                if(emitter->emitSaving < 0)
                {
                    emitter->emitSaving = 0;
                }

                s32 num = static_cast<s32>( emitter->emitSaving );
                if ( num != 0 )
                {
                    Emit( emitter, num );

                    // 放出ごとに行う処理
                    emitter->UpdateByEmit();
                    {
                        // MEMO: 理想的には UpdateByEmit() 内で処理する方がキレイ
                        //       ここで常に Dirty フラグを立てておかないと、Calcの最後で機械的にフラグが降ろされて更新されないので暫定。
                        emitterSet->m_IsEmitterSRTDirty = 1;
                    }

                    emitter->emitSaving     -= num;
                    emitter->emitCnt        = frameRate + emitCntSub;   //放出間隔を通り過ぎた分（emitCntSub）も進める。
                    emitter->emitLastFrame  = emitter->time;
                }
                else
                {
                    emitter->emitCnt        = 0;
                }
            }
            else
            {
                emitter->emitCnt += frameRate;
            }
        }
        else
        {
        //-------------------------
        // 距離放出
        //-------------------------

            // 今回の移動距離
            f32 length = emitter->emitterLocalVec.Length();

            // 貯金
            f32 vessel = emitter->emitSaving;

            // 今の区間の距離スケーリング
            //f32 scale = 1.0f;

            // 仮想的な長さ
            f32 virtualLength = length;

            // 移動距離切り捨ての閾値
            if( virtualLength < emitter->emitterData->emission.emitDistMargin )
            {
                virtualLength = 0.0f;
            }
            if( virtualLength == 0.0f )
            {
                virtualLength = emitter->emitterData->emission.emitDistMin;
            }

            else if( virtualLength < emitter->emitterData->emission.emitDistMin )
            {
                // 最小距離にかかる場合
                virtualLength = length * emitter->emitterData->emission.emitDistMin / length;
            }
            else if( emitter->emitterData->emission.emitDistMax < virtualLength )
            {
                // 最大距離にかかる場合
                virtualLength = length * emitter->emitterData->emission.emitDistMax / length;
            }

            // 今回の区間の加算
            vessel += virtualLength;

            // 今回の放出個数を決定
            s32 count = 0;
            if( emitter->emitterData->emission.emitDistUnit != 0.0 )
            {
                count = static_cast<s32>( vessel / emitter->emitterData->emission.emitDistUnit );
            }

            // 放出処理
            if( count > 0 )
            {
                const nw::math::VEC3 prevPos( emitter->emitterPrevPos.x, emitter->emitterPrevPos.y, emitter->emitterPrevPos.z );
                const nw::math::VEC3 currPos( emitter->matrixRT._03, emitter->matrixRT._13, emitter->matrixRT._23 );
                //const nw::math::VEC3 move( prevPos - currPos );

                for( s32 i=0; i<count; ++i )
                {
                    vessel -= emitter->emitterData->emission.emitDistUnit;

                    // 位置の比率
                    f32 ratio = 0.0f;
                    if( virtualLength != 0.0f ) { ratio = vessel / virtualLength; }
                    // 位置
                    const nw::math::VEC3 pos = currPos * ( 1.0f - ratio ) + prevPos * ratio;

                    emitter->matrixRT._03  = pos.x;
                    emitter->matrixRT._13  = pos.y;
                    emitter->matrixRT._23  = pos.z;
                    emitter->matrixSRT._03 = pos.x;
                    emitter->matrixSRT._13 = pos.y;
                    emitter->matrixSRT._23 = pos.z;

                    Emit( emitter, 1 );

                    emitter->matrixRT._03  = currPos.x;
                    emitter->matrixRT._13  = currPos.y;
                    emitter->matrixRT._23  = currPos.z;
                    emitter->matrixSRT._03 = currPos.x;
                    emitter->matrixSRT._13 = currPos.y;
                    emitter->matrixSRT._23 = currPos.z;
                }

                emitter->emitLastFrame  = emitter->time;
            }

            emitter->emitSaving = vessel;
        }
    }

exit_emit:

    // CPUパーティクルパーティクル挙動計算を行う。
    if ( res->emitter.calcType == EFT_EMITTER_CALC_TYPE_CPU )
    {
        emitter->emitterCalc->CalcParticle( emitter );
    }
    else
    {
        emitter->ptclProcessingNum = emitter->ptclNum;
    }

    // エミッタ計算処理後コールバック
#ifndef EFT_DEGRADATION_SPEC
    InvokeEmitterPostCalcCallback( emitter );
#endif

    // emitter->particleAttrFill を調整
    if ( emitter->emitterData->ptcl.lifeRandom != 0 && emitter->ptclAttrFillIndex > 0 )
    {
        for ( u32 j = emitter->ptclAttrFillIndex-1; j > 0; j-- )
        {
            ParticlePosAttribute* posAttr = &emitter->particlePosAttr[j];
            f32 time = posAttr->GetTime( emitter->time );
            f32 life = posAttr->GetLife();

            // 死亡チェック
            if( time <= life )
            {
                break;
            }

            emitter->ptclAttrFillIndex = j;
            emitter->ptclNum = emitter->ptclAttrFillIndex;
        }
    }

    // エミッタ動的ユニフォームブロック生成
    if ( emitter->ptclNum > 0 || emitter->emitterRes->emitterPluginData )
    {
        MakeDynamicUniformBlock( emitter, emitter->frameRate );
    }

    // 今フレームのエミッタ位置を保存
    emitter->emitterPrevPos.Set( emitter->matrixSRT._03, emitter->matrixSRT._13, emitter->matrixSRT._23 );

    // タイムを進める
    emitter->time += frameRate;

    // エミッタ死亡判定
    // TODO : パーティクル放出時に最大エミッタ寿命を計算しておく。
    if ( res->emission.isOneTime && emitterSet->m_IsManualEmission == false )
    {
        // 寿命が有限
        if( res->ptcl.isLifeInfinity != 1 )
        {
            f32 endTime = emitEnd + res->ptcl.life;

            if( res->emitter.isFadeAlphaFade || res->emitter.isScaleFade )
            {
                endTime += res->emitter.alphaFadeTime;
            }

            // エミッタプラグインを持つ場合、ワンタイム時に追加で待つ時間があるかどうか確認する
            if( emitter->emitterRes->emitterPluginIndex > 0 )
            {
                // MEMO: 最終的にはこの辺はインターフェースでまとめられると良い
                switch( emitter->emitterRes->emitterPluginIndex )
                {
                    // MEMO: 連結式ストライプ・範囲内ループは何もしない
                case ConnectionStripeSystem::PluginId:
                case AreaLoopSystem::PluginId:
                    break;
                case StripeSystem::PluginId:
                    endTime += StripeSystem::GetExtendedEndTimeForOneTimeEmitter( emitter );
                    break;
                case SuperStripeSystem::PluginId:
                    endTime += SuperStripeSystem::GetExtendedEndTimeForOneTimeEmitter( emitter );
                    break;
                default:
                    break;
                }
            }

            if( emitterTime > endTime )
            {
                return false;
            }
        }
    }
    else
    {
        if( isFade || emitter->m_IsSoloFade || emitter->isChild )
        {
            if ( emitterTime > emitStart + emitter->emitLastFrame + res->ptcl.life )
            {
                return false;
            }
        }

        if( emitterSet->m_IsManualEmission )
        {
            // 非常駐型のマニュアル放出エミッタで、指定時間を経過して粒の数が0なら消える
            if( emitter->IsManualEmitterReadyToExit() )
            {
                return false;
            }
        }
    }

    return true;
}



//---------------------------------------------------------------------------
//  パーティクル計算処理をします。
//---------------------------------------------------------------------------
void EmitterCalc::CalcParticle( Emitter* emitter )
{
    // CPUパーティクル指定の場合は、POS/VEC 挙動計算を行う。
    ResEmitter* res = emitter->emitterData;
    if ( res->emitter.calcType != EFT_EMITTER_CALC_TYPE_CPU ) return;

    // パーティクルコールバック
    ParticleRemoveCallback ptclRmvCbEP = NULL;
    ParticleRemoveCallback ptclRmvCbCA = NULL;
    ParticleRemoveCallback ptclRmvCbCS = NULL;
    ParticleCalcCallback   ptclClcCbEP = NULL;
    ParticleCalcCallback   ptclClcCbCA = NULL;
    ParticleCalcCallback   ptclClcCbCS = NULL;
    bool                   callParticleRemoveCallback = false;
    bool                   callParticleCalcCallback   = false;

    if ( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP] )
    {
        ptclRmvCbEP = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->particleRemove;
        ptclClcCbEP = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_EP]->particleCalc;
    }
    if ( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA] )
    {
        ptclRmvCbCA = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->particleRemove;
        ptclClcCbCA = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CA]->particleCalc;
    }
    if ( emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS] )
    {
        ptclRmvCbCS = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->particleRemove;
        ptclClcCbCS = emitter->callbackSet[EFT_CALLBACK_SET_TYPE_CS]->particleCalc;
    }

    if( ptclRmvCbEP || ptclRmvCbCA || ptclRmvCbCS ) { callParticleRemoveCallback = true; }
    if( ptclClcCbEP || ptclClcCbCA || ptclClcCbCS ) { callParticleCalcCallback   = true; }

    u32 processNum = 0;
    u32 processEnd = 0;

    for ( u32 i = 0; i < emitter->ptclNum; i++ )
    {
        if ( emitter->particleData[i].createID == 0 ) { continue; }

        ParticlePosAttribute* posAttr  = &emitter->particlePosAttr[i];
        ParticleAttribute*    ptclAttr = &emitter->particleAttr[i];
        ParticleData*         ptclData = &emitter->particleData[i];

        // ワンエミッタ向け対応。
        f32 time = emitter->particleData[i].GetTime( emitter->time );
        f32 life = emitter->particleData[i].GetLife();

        // フロントバッファに生成時刻・寿命を書き移す
        posAttr->localPos.w = ptclData->life;
        posAttr->localVec.w = ptclData->createTime;

        // パーティクル削除チェック
        if( time >= life )
        {
            // パーティクル削除コールバック
#ifndef EFT_DEGRADATION_SPEC
            if ( callParticleRemoveCallback )
            {
                //TODO: この時点ではPosAttrのフロントバッファは2フレーム前の値を指している。
                //      削除コールバックの性質上、Get/Setは実質無意味なので一旦そのままにしておくが、
                //      何らかの理由で使う必要が生じたら、フロントバッファに値をコピーする処理が必須。
                if ( emitter->childEmitterResSet[0] )
                {

                    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[i].localPos, &emitter->particlePosAttrBack[i].localPos );
                    EFT_F32_VEC3_COPY( &emitter->particlePosAttr[i].localVec, &emitter->particlePosAttrBack[i].localVec );
                }
                InvokeParticleRemoveCallback( emitter, i, time, life, ptclRmvCbEP, ptclRmvCbCA, ptclRmvCbCS );
            }
#endif
            // 削除をマーキング
            emitter->particleData[i].createID = 0;

            // キャッシュを落とす
            MemUtil::FlushCache( &emitter->particlePosAttr[i].localPos, sizeof( nw::math::VEC4 ) );
            MemUtil::FlushCache( &emitter->particlePosAttr[i].localVec, sizeof( nw::math::VEC4 ) );
        }
        else
        {
            nw::math::VEC3 oldPos;
            if( time > 0 )
            {
                if( emitter->IsParticleAttributeDoubleBuffer() && ptclData->isBothInitialized == false )
                {
                    // 「2回目」の更新では、フロントバッファの値がまだ入っていないので、バックから正しい値をコピーする
                    MemUtil::Copy( &emitter->particleAttr[i], &emitter->particleAttrBack[i], sizeof( ParticleAttribute ) );
                    MemUtil::FlushCache( &emitter->particleAttr[i], sizeof( ParticleAttribute ) );
                    ptclData->isBothInitialized = true;
                }
                EFT_F32_VEC3_COPY( &oldPos, &emitter->particlePosAttrBack[i].localPos );
            }
            else
            {
                // 初回フレームは更新前の自分自身と比較
                EFT_F32_VEC3_COPY( &oldPos, &emitter->particlePosAttr[i].localPos );

                // LocalDiffにLocalVecを入れておく
                EFT_F32_VEC3_COPY( &posAttr->localDiff, &posAttr->localVec );
            }

            // 通常パーティクル挙動
            // この中でフロントバッファの更新を行う
            _CalcParticleBehavior_Integrating( emitter, i, &posAttr->localPos, &posAttr->localVec, &emitter->particlePosAttrBack[i].localPos, &emitter->particlePosAttrBack[i].localVec, time );

            // パーティクル計算コールバック
#ifndef EFT_DEGRADATION_SPEC
            if( callParticleCalcCallback )
            {
                if( time == 0 && emitter->IsParticleAttributeDoubleBuffer() )
                {
                    // ダブルバッファ時の初回フレームは、まだBackに値が入っていない。
                    // 従ってInit時と同じく、一時的にポインタを移しておく
                    ParticleAttribute* particleAttrBackOrg = emitter->particleAttrBack;
                    emitter->particleAttrBack = emitter->particleAttr;
                    InvokeParticleCalcCallback( emitter, i, time, life, ptclClcCbEP, ptclClcCbCA, ptclClcCbCS );
                    emitter->particleAttrBack = particleAttrBackOrg;
                }
                else
                {
                    // この時点では更新後の値がフロントバッファに入っているので、そのままフロントバッファから値を参照させる。
                    InvokeParticleCalcCallback( emitter, i, time, life, ptclClcCbEP, ptclClcCbCA, ptclClcCbCS );
                }

                // lifeの更新（ Killされた場合 ）
                posAttr->localPos.w = ptclData->life;
            }
#endif
            // パーティクル移動差分を計算
            const float POSDIFF_MIN_VEL = 0.01f;
            nw::math::VEC3 vel;
            vel.x = ( posAttr->localPos.x - oldPos.x );
            vel.y = ( posAttr->localPos.y - oldPos.y );
            vel.z = ( posAttr->localPos.z - oldPos.z );

            if( fabsf(vel.x) > POSDIFF_MIN_VEL || fabsf(vel.y) > POSDIFF_MIN_VEL || fabsf(vel.z) > POSDIFF_MIN_VEL )
            {
                EFT_F32_VEC3_COPY( &posAttr->localDiff, &vel );
            }
            else
            {
                if( time > 0.0f )
                {
                    // 余りにも微小な移動量の場合、前のフレームの値を採用
                    // （初回フレームは既にvecをコピー済みなので何もしない）
                    EFT_F32_VEC3_COPY( &posAttr->localDiff, &emitter->particlePosAttrBack[i].localDiff );
                }
            }

            processEnd = i;
            processNum++;
        }

        // 子エミッタに対して 位置/速度 を伝える
        // この処理はもうPosAttrのフロントバッファが更新済みなので前から取る。
        if ( emitter->childEmitterResSet[0] )
        {
            for ( u32 j = 0; j < emitter->emitterRes->childEmitterResCount; j++ )
            {
                Emitter* childEmitter = emitter->parentParticleData[i].childEmitter[j];
                if ( !childEmitter ) { continue; }

                // 子エミッタが生存しているか
                if ( childEmitter->IsAlive() && childEmitter->parentEmitterCreateID == emitter->emitterCreateID )
                {
                    // パーティクルの時間を子エミッタに伝達
                    childEmitter->parentParticleTime = posAttr->GetTime( emitter->time );
                    childEmitter->parentParticleLife = posAttr->GetLife();

                    // パーティクルの位置を子エミッタに伝達
                    if ( emitter->GetFollowType() == EFT_EMITTER_FOLLOW_TYPE_ALL )
                    {
                        childEmitter->parentParticleWorldPos.x = emitter->matrixSRT.f._00 * posAttr->localPos.x + emitter->matrixSRT.f._01 * posAttr->localPos.y + emitter->matrixSRT.f._02 * posAttr->localPos.z + emitter->matrixSRT.f._03;
                        childEmitter->parentParticleWorldPos.y = emitter->matrixSRT.f._10 * posAttr->localPos.x + emitter->matrixSRT.f._11 * posAttr->localPos.y + emitter->matrixSRT.f._12 * posAttr->localPos.z + emitter->matrixSRT.f._13;
                        childEmitter->parentParticleWorldPos.z = emitter->matrixSRT.f._20 * posAttr->localPos.x + emitter->matrixSRT.f._21 * posAttr->localPos.y + emitter->matrixSRT.f._22 * posAttr->localPos.z + emitter->matrixSRT.f._23;
                    }
                    else
                    {
                        childEmitter->parentParticleWorldPos.x = ptclAttr->emitterMat0.x * posAttr->localPos.x + ptclAttr->emitterMat0.y * posAttr->localPos.y + ptclAttr->emitterMat0.z * posAttr->localPos.z + ptclAttr->emitterMat0.w;
                        childEmitter->parentParticleWorldPos.y = ptclAttr->emitterMat1.x * posAttr->localPos.x + ptclAttr->emitterMat1.y * posAttr->localPos.y + ptclAttr->emitterMat1.z * posAttr->localPos.z + ptclAttr->emitterMat1.w;
                        childEmitter->parentParticleWorldPos.z = ptclAttr->emitterMat2.x * posAttr->localPos.x + ptclAttr->emitterMat2.y * posAttr->localPos.y + ptclAttr->emitterMat2.z * posAttr->localPos.z + ptclAttr->emitterMat2.w;
                    }

                    // パーティクルの速度を子エミッタに伝達
                    {
                        // Local系 -> World系に変換して渡す
                        childEmitter->parentParticleWorldVec.x = posAttr->localVec.x * emitter->matrixRT.m[0][0] + posAttr->localVec.y * emitter->matrixRT.m[0][1] + posAttr->localVec.z * emitter->matrixRT.m[0][2];
                        childEmitter->parentParticleWorldVec.y = posAttr->localVec.x * emitter->matrixRT.m[1][0] + posAttr->localVec.y * emitter->matrixRT.m[1][1] + posAttr->localVec.z * emitter->matrixRT.m[1][2];
                        childEmitter->parentParticleWorldVec.z = posAttr->localVec.x * emitter->matrixRT.m[2][0] + posAttr->localVec.y * emitter->matrixRT.m[2][1] + posAttr->localVec.z * emitter->matrixRT.m[2][2];
                    }
                }
            }
        }
    }

    // 詰まったアトリビュートのキャッシュをフラッシュする。
    if ( processNum != 0 )
    {
        MemUtil::FlushCache( emitter->particlePosAttr, emitter->ptclNum * sizeof( ParticlePosAttribute ) );
    }
    else
    {
        emitter->ptclNum = 0;
        emitter->ptclAttrFillIndex = 0;
    }

    emitter->ptclProcessingNum = processNum;
    emitter->ptclProcessingEnd = processEnd;
}


//---------------------------------------------------------------------------
//  エミッタの動的ユニフォームブロックを生成します。
//---------------------------------------------------------------------------
void EmitterCalc::MakeDynamicUniformBlock( Emitter* emitter, f32 frameRate )
{
    emitter->dynamicUBO[emitter->bufferID].InValidate();

    EmitterDynamicUniformBlock* emDynamicUbo =
        reinterpret_cast<EmitterDynamicUniformBlock*>( emitter->dynamicUBO[emitter->bufferID].GetBuffer() );
    EmitterSet* emitterSet = emitter->emitterSet;

    const nw::math::VEC4& emitterSetColor = emitterSet->GetColor();

    emDynamicUbo->emitterColor0.x = emitter->color0.x * emitter->m_EmitterAnim.vcolor0.x * emitterSetColor.x;
    emDynamicUbo->emitterColor0.y = emitter->color0.y * emitter->m_EmitterAnim.vcolor0.y * emitterSetColor.y;
    emDynamicUbo->emitterColor0.z = emitter->color0.z * emitter->m_EmitterAnim.vcolor0.z * emitterSetColor.z;
    emDynamicUbo->emitterColor0.w = emitter->color0.w * emitter->m_EmitterAnim.valpha0.x;

    emDynamicUbo->emitterColor1.x = emitter->color1.x * emitter->m_EmitterAnim.vcolor1.x * emitterSetColor.x;
    emDynamicUbo->emitterColor1.y = emitter->color1.y * emitter->m_EmitterAnim.vcolor1.y * emitterSetColor.y;
    emDynamicUbo->emitterColor1.z = emitter->color1.z * emitter->m_EmitterAnim.vcolor1.z * emitterSetColor.z;
    emDynamicUbo->emitterColor1.w = emitter->color1.w * emitter->m_EmitterAnim.valpha1.x;

    // 子エミッタで親の値を毎フレーム継承する場合、ここで親エミッタのα値を乗算して継承。
    if( emitter->isChild )
    {
        const Emitter* pParent = emitter->parentEmitter;
        if( emitter->emitterRes->emitterData->inherit.alpha0 && emitter->emitterRes->emitterData->inherit.alpha0EachFrame )
        {
            emDynamicUbo->emitterColor0.w *= pParent->GetEmitterAnimValue().valpha0.x;
        }
        if( emitter->emitterRes->emitterData->inherit.alpha1 && emitter->emitterRes->emitterData->inherit.alpha1EachFrame )
        {
            emDynamicUbo->emitterColor1.w *= pParent->GetEmitterAnimValue().valpha1.x;
        }
    }

    // αフェード
    f32 alpha = emitterSetColor.w;
    alpha *= emitter->CalculateCurrentAlphaFadeValue();

    // 子エミッタで親の値を毎フレーム継承する場合、ここで親エミッタのフェード（In/Out）値を乗算して継承。
    if( ( emitter->emitterRes->emitterData->inherit.alpha0 && emitter->emitterRes->emitterData->inherit.alpha0EachFrame ) ||
        ( emitter->emitterRes->emitterData->inherit.alpha1 && emitter->emitterRes->emitterData->inherit.alpha1EachFrame ) )
    {
        const Emitter* pParent = emitter->parentEmitter;
        alpha *= pParent->CalculateCurrentAlphaFadeValue();
    }

    // スケールフェード
    const f32 fScale = emitter->CalculateCurrentScaleFadeValue();

    emDynamicUbo->emitterParam0.x = emitter->time;
    emDynamicUbo->emitterParam0.y = 1.0f;
    emDynamicUbo->emitterParam0.z = 1.0f;
    emDynamicUbo->emitterParam0.w = frameRate;
    emDynamicUbo->emitterParam1.x = alpha;
    emDynamicUbo->emitterParam1.y = emitterSet->m_ParticleScaleForCalc.x * fScale;
    emDynamicUbo->emitterParam1.z = emitterSet->m_ParticleScaleForCalc.y * fScale;
    emDynamicUbo->emitterParam1.w = emitterSet->m_ParticleScaleForCalc.z * fScale;

    EFT_F32_VEC4_COPY( &emDynamicUbo->emitterMatrix.v[0], &emitter->matrixSRT.v[0] );
    EFT_F32_VEC4_COPY( &emDynamicUbo->emitterMatrix.v[1], &emitter->matrixSRT.v[1] );
    EFT_F32_VEC4_COPY( &emDynamicUbo->emitterMatrix.v[2], &emitter->matrixSRT.v[2] );
    emDynamicUbo->emitterMatrix.v[3].x = emDynamicUbo->emitterMatrix.v[3].y = emDynamicUbo->emitterMatrix.v[3].z = 0.0f;
    emDynamicUbo->emitterMatrix.v[3].w = 1.0f;
    EFT_F32_VEC4_COPY( &emDynamicUbo->emitterMatrixRT.v[0], &emitter->matrixRT.v[0] );
    EFT_F32_VEC4_COPY( &emDynamicUbo->emitterMatrixRT.v[1], &emitter->matrixRT.v[1] );
    EFT_F32_VEC4_COPY( &emDynamicUbo->emitterMatrixRT.v[2], &emitter->matrixRT.v[2] );
    emDynamicUbo->emitterMatrixRT.v[3].x = emDynamicUbo->emitterMatrixRT.v[3].y = emDynamicUbo->emitterMatrixRT.v[3].z = 0.0f;
    emDynamicUbo->emitterMatrixRT.v[3].w = 1.0f;

    emitter->dynamicUBO[emitter->bufferID].Validate();
}


} // namespace eft2
} // namespace nw
