﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/



#include <nw/eft/eft2_AreaLoop.h>
#include <nw/eft/eft2_EmitterCalc.h>
#include <nw/eft/eft2_System.h>

#ifndef EFT_DEGRADATION_SPEC

namespace nw   {
namespace eft2 {

bool AreaLoopSystem::Initialize( System* system )
{
    EFT_UNUSED_VARIABLE(system);

    CallbackSet set;
    set.emitterPreCalc      = NULL;
    set.emitterInitialize   = NULL;
    set.particleEmit        = NULL;
    set.particleRemove      = NULL;
    set.particleCalc        = NULL;
    set.emitterPostCalc     = NULL;
    set.emitterDraw         = AreaLoopSystem::_EmitterDrawCallback;
    set.emitterFinalize     = NULL;
    set.renderStateSet      = AreaLoopSystem::_RenderStateSetCallback;
    system->SetEmitterPluginCallbackSet( EFT_EMITTER_PLUGIN_CALLBACK_ID_4, set );

    return true;
}

bool AreaLoopSystem::_RenderStateSetCallback( RenderStateSetArg& arg )
{
    EFT_UNUSED_VARIABLE( arg );
    // 何もしない！（自動UBOセットをキャンセル）
    return true;
}

bool AreaLoopSystem::_EmitterDrawCallback( EmitterDrawArg& arg )
{
    AreaLoopSystem::Draw( arg.emitter, arg.shaderType, arg.userParam );
    return true;
}

bool AreaLoopSystem::Draw( Emitter* emitter, ShaderType shaderType, void* userParam )
{
    //-------------------------------------------------------------------------
    // 範囲内ループの UBO 構造体
    //-------------------------------------------------------------------------
    struct AreaLoopUBO
    {
        nw::math::VEC4 param0;      //!< リピート毎のオフセット(xyz) / カメラ前ループフラグ
        nw::math::VEC4 param1;      //!< 端をαで薄める割合(xyz) / empty
        nw::math::Matrix44 mat;     //!< 座標系変換行列
        nw::math::Matrix44 invMat;  //!< 座標系変換逆行列
        nw::math::VEC4 param2;      //!< クリッピングタイプ / クリッピングY座標 / empty / empty
        nw::math::VEC4 param3;      //!< 仮想箱の大きさ(xyz) / empty
    };

    System* system = emitter->emitterSet->GetSystem();
    ResEPAreaLoop* areaLoopData = reinterpret_cast< ResEPAreaLoop* >( emitter->emitterRes->emitterPluginData );

    //-------------------------------------------------------------------------
    // シェーダ切り替え
    //-------------------------------------------------------------------------
    Shader* shader = emitter->shader[ shaderType ];
    if ( !shader )
    {
        Warning( emitter, EFT_WARNING_SHADER_IS_NOT_EXIST );
        return false;
    }
    shader->Bind();

    //-------------------------------------------------------------------------
    // UBO デフォルトパラメータを設定する
    //-------------------------------------------------------------------------
    AreaLoopUBO defalutUbo;

    EFT_F32_VEC3_COPY( &defalutUbo.param1, &areaLoopData->alphaRatio );
    defalutUbo.param1.w = 0;
    defalutUbo.param2.x = static_cast< f32 >( areaLoopData->clippingType );
    defalutUbo.param2.y = areaLoopData->clippingWorldHeight;
    defalutUbo.param2.z = 0;
    defalutUbo.param2.w = 0;
    EFT_F32_VEC3_COPY( &defalutUbo.param3, &areaLoopData->areaSize );
    defalutUbo.param3.w = 0;

    //-------------------------------------------------------------------------
    // 変換行列の事前計算
    //-------------------------------------------------------------------------
    {
        defalutUbo.mat = nw::math::Matrix44::Identity();

        // 共通化できる部分もあるが、可読性を重視して行列を作る部分は完全に分岐させる
        if( areaLoopData->isCameraLoop == 0 )
        {
            // 標準タイプ
            nw::math::Matrix44 boxMatrix = nw::math::Matrix44::Identity();

            // 仮想箱系へのRT行列（箱の大きさは割合位置を求めるのにvsh側で使用する）
            // Rotate
            nw::math::Matrix44 boxRotMat;
            MakeRotationMatrixXYZ( &boxRotMat, nw::math::VEC3( areaLoopData->areaRotate.x, areaLoopData->areaRotate.y, areaLoopData->areaRotate.z ) );
            boxMatrix.SetMult( boxRotMat, boxMatrix );

            // Translate
            // 標準の場合、エミッタセットSRTを考慮
            EFT_F32_VEC3_COPY( &boxMatrix.v[3], &areaLoopData->areaPos );

            // 標準の場合、EmitterSetSRTを適用する
            defalutUbo.mat.SetMult( boxMatrix, nw::math::Matrix44( emitter->matrixSRT ).Transpose() );
        }
        else
        {
            // カメラ前ループの場合
            nw::math::Matrix44 boxMatrix = nw::math::Matrix44::Identity();
            ViewParam* vp = system->GetViewParam();

            // カメラ位置への変換行列を作成。
            nw::math::Matrix44 cameraMatrix = nw::math::Matrix44::Identity();
            EFT_F32_VEC3_COPY( &cameraMatrix.v[3], &vp->eyePos );

            // 仮想箱系へのRT行列（箱の大きさは割合位置を求めるのにvsh側で使用する）
            // Rotate:
            nw::math::Matrix44 boxRotMat;
            MakeRotationMatrixXYZ( &boxRotMat, nw::math::VEC3( areaLoopData->areaRotate.x, areaLoopData->areaRotate.y, areaLoopData->areaRotate.z ) );
            boxMatrix.SetMult( boxRotMat, boxMatrix );

            // Translate:
            // ビルボード行列を使ってカメラ系での移動量に回転させておく
            nw::math::Matrix44 bldMatrix = vp->bldMat;
            nw::math::VEC3 pos = areaLoopData->areaPos;
            boxMatrix.m[3][0] = pos.x * bldMatrix.m[0][0] + pos.y * bldMatrix.m[0][1] + pos.z * bldMatrix.m[0][2];
            boxMatrix.m[3][1] = pos.x * bldMatrix.m[1][0] + pos.y * bldMatrix.m[1][1] + pos.z * bldMatrix.m[1][2];
            boxMatrix.m[3][2] = pos.x * bldMatrix.m[2][0] + pos.y * bldMatrix.m[2][1] + pos.z * bldMatrix.m[2][2];

            // EmitterSetSRTは考慮しない（それほど意味が無い上に混乱の元になるので）
            // カメラ系行列 → 仮想箱系の順番
            defalutUbo.mat.SetMult( boxMatrix, cameraMatrix );
        }

        //-------------------------------------------------------------------------
        // 逆行列をあらかじめ計算して保存しておく
        //-------------------------------------------------------------------------
        defalutUbo.invMat = nw::math::Matrix44( defalutUbo.mat );
        defalutUbo.invMat.Inverse();
    }

    //-------------------------------------------------------------------------
    // 描画ループ
    //-------------------------------------------------------------------------
    nw::math::VEC3 repeatOffset = nw::math::VEC3( 0, 0, 0 );
    const u32 maxCount = static_cast< u32 >( areaLoopData->repeatNum + 1 );
    for( u32 i = 0; i < maxCount; ++i )
    {
        // メモリ確保
        TemporaryUniformBlockBuffer tempUbo;
        AreaLoopUBO* ubo = reinterpret_cast< AreaLoopUBO* >( tempUbo.Alloc( system, sizeof( AreaLoopUBO ) ) );
        if ( !ubo ){ return false; }

        // UBO のパラメータのセット
        EFT_F32_VEC3_COPY( &ubo->param0, &repeatOffset );
        ubo->param0.w = areaLoopData->isCameraLoop;
        EFT_F32_VEC4_COPY( &ubo->param1, &defalutUbo.param1 );
        EFT_F32_VEC4_COPY( &ubo->param2, &defalutUbo.param2 );
        EFT_F32_VEC4_COPY( &ubo->param3, &defalutUbo.param3 );

        // UBO の行列のセット
        ubo->mat = defalutUbo.mat;
        ubo->invMat = defalutUbo.invMat;

        // UBO を有効に。
        tempUbo.Validate();

        // UBO 設定
        shader->BindEmitterPluginUniformBlock( &tempUbo );

        // 描画処理に回す
        emitter->emitterCalc->DrawEmitterUsingBoundShader( emitter, shaderType, userParam );

        // リピート用オフセットを加算
        repeatOffset += areaLoopData->repeatOffsetPos;
    }

    return true;
}

} // namespace eft2
} // namespace nw

#endif      // EFT_DEGRADATION_SPEC

